/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
// Output structure of the tensorVoting2d function.
struct TensorVoting2dOutput
{
    /// The tensor output corresponding to the strengthened image.
    std::shared_ptr< iolink::ImageView > outputTensorImage;
    /// The orientation vector output image, defining the main orientation of the output tensor.
    std::shared_ptr< iolink::ImageView > outputVectorImage;
    /// The saliency output image representing the anisotropy of the structures.
    std::shared_ptr< iolink::ImageView > outputSaliencyImage;
    /// The ballness output image representing the isotropy of the structures.
    std::shared_ptr< iolink::ImageView > outputBallnessImage;
};

/// Strengthens local features in a two-dimensional image according to their consistency with a model of smooth curves.
class IMAGEDEV_CPP_API TensorVoting2d final : public GenericAlgorithm
{
public:
    /// The output images to be computed.
    enum OutputSelection
    {
    /// Enable the computation of the tensor output image.
        TENSOR = 1,
    /// Enable the computation of the orientation vector output image.
        ORIENTATION_VECTOR = 2,
    /// Enable the computation of the saliency output image.
        SALIENCY = 4,
    /// Enable the computation of the ballness output image.
        BALLNESS = 8
    };

    // Command constructor.
    TensorVoting2d();


    /// Gets the inputFeatureImage parameter.
    /// The feature input image. It can be produced by the EigenvaluesToStructureness2d algorithm.
    std::shared_ptr< iolink::ImageView > inputFeatureImage() const;
    /// Sets the inputFeatureImage parameter.
    /// The feature input image. It can be produced by the EigenvaluesToStructureness2d algorithm.
    void setInputFeatureImage( std::shared_ptr< iolink::ImageView > inputFeatureImage );

    /// Gets the inputVectorImage parameter.
    /// The vector input image indicating the orientation of the feature to strengthen. It can be produced by the EigenDecomposition2d algorithm. When used with an edge detector such as an Hessian matrix, the second eigenvector has to be selected since the first eigenvector is orthogonal to the structures.
    std::shared_ptr< iolink::ImageView > inputVectorImage() const;
    /// Sets the inputVectorImage parameter.
    /// The vector input image indicating the orientation of the feature to strengthen. It can be produced by the EigenDecomposition2d algorithm. When used with an edge detector such as an Hessian matrix, the second eigenvector has to be selected since the first eigenvector is orthogonal to the structures.
    void setInputVectorImage( std::shared_ptr< iolink::ImageView > inputVectorImage );

    /// Gets the scale parameter.
    /// The scale parameter acting on the gap length filled by the tensor voting algorithm. It is expressed in pixels.
    double scale() const;
    /// Sets the scale parameter.
    /// The scale parameter acting on the gap length filled by the tensor voting algorithm. It is expressed in pixels.
    void setScale( const double& scale );

    /// Gets the outputSelection parameter.
    /// The output images to be computed.
    int32_t outputSelection() const;
    /// Sets the outputSelection parameter.
    /// The output images to be computed.
    void setOutputSelection( const int32_t& outputSelection );

    /// Gets the outputTensorImage parameter.
    /// The tensor output corresponding to the strengthened image.
    std::shared_ptr< iolink::ImageView > outputTensorImage() const;
    /// Sets the outputTensorImage parameter.
    /// The tensor output corresponding to the strengthened image.
    void setOutputTensorImage( std::shared_ptr< iolink::ImageView > outputTensorImage );

    /// Gets the outputVectorImage parameter.
    /// The orientation vector output image, defining the main orientation of the output tensor.
    std::shared_ptr< iolink::ImageView > outputVectorImage() const;
    /// Sets the outputVectorImage parameter.
    /// The orientation vector output image, defining the main orientation of the output tensor.
    void setOutputVectorImage( std::shared_ptr< iolink::ImageView > outputVectorImage );

    /// Gets the outputSaliencyImage parameter.
    /// The saliency output image representing the anisotropy of the structures.
    std::shared_ptr< iolink::ImageView > outputSaliencyImage() const;
    /// Sets the outputSaliencyImage parameter.
    /// The saliency output image representing the anisotropy of the structures.
    void setOutputSaliencyImage( std::shared_ptr< iolink::ImageView > outputSaliencyImage );

    /// Gets the outputBallnessImage parameter.
    /// The ballness output image representing the isotropy of the structures.
    std::shared_ptr< iolink::ImageView > outputBallnessImage() const;
    /// Sets the outputBallnessImage parameter.
    /// The ballness output image representing the isotropy of the structures.
    void setOutputBallnessImage( std::shared_ptr< iolink::ImageView > outputBallnessImage );

    // Method to launch the command.
    void execute();

};

/// Strengthens local features in a two-dimensional image according to their consistency with a model of smooth curves.
/// @param inputFeatureImage The feature input image. It can be produced by the EigenvaluesToStructureness2d algorithm.
/// @param inputVectorImage The vector input image indicating the orientation of the feature to strengthen. It can be produced by the EigenDecomposition2d algorithm. When used with an edge detector such as an Hessian matrix, the second eigenvector has to be selected since the first eigenvector is orthogonal to the structures.
/// @param scale The scale parameter acting on the gap length filled by the tensor voting algorithm. It is expressed in pixels.
/// @param outputSelection The output images to be computed.
/// @param outputTensorImage The tensor output corresponding to the strengthened image.
/// @param outputVectorImage The orientation vector output image, defining the main orientation of the output tensor.
/// @param outputSaliencyImage The saliency output image representing the anisotropy of the structures.
/// @param outputBallnessImage The ballness output image representing the isotropy of the structures.
/// @return Returns a TensorVoting2dOutput structure containing the outputTensorImage, outputVectorImage, outputSaliencyImage and outputBallnessImage output parameters.
IMAGEDEV_CPP_API 
TensorVoting2dOutput
tensorVoting2d( std::shared_ptr< iolink::ImageView > inputFeatureImage,
                std::shared_ptr< iolink::ImageView > inputVectorImage,
                double scale,
                int32_t outputSelection,
                std::shared_ptr< iolink::ImageView > outputTensorImage = nullptr,
                std::shared_ptr< iolink::ImageView > outputVectorImage = nullptr,
                std::shared_ptr< iolink::ImageView > outputSaliencyImage = nullptr,
                std::shared_ptr< iolink::ImageView > outputBallnessImage = nullptr );
} // namespace imagedev
