// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <iolink/DataType.h>
#include <iolink/MemoryLayout.h>
#include <iolink/VectorX.h>

namespace iolink
{

/**
 * This class is used to index a classic array into a multi dimensional array.
 *
 * Its main use is to convert a multi-dimensional index into the corresponding
 * offset in the flat array used for storage. An indexer contains all the needed
 * tools and describes how the data is really stored. The three main elements are:
 * the shape, the stride, and the layout.
 *
 * The shape is simply the size of the indexed array in each dimension, for example
 * an array with two columns and two rows will have a shape of {2, 2}.
 *
 * The stride contains the offset between each iteration of a dimension, in elements.
 * For example, our {2, 2} array will usually have a stride of {1, 2}, as each column
 * element is adjacent (hence the stride of one), and when iterating element on a line,
 * you must skip one element on the underlying array (hence the stride of 2).
 * The stride dimension are in the same order as the shape, but they do not have to
 * be in ascendent order. Actually, reverse order arrays (C convention) will have
 * inversed strides, for example, {2, 1} in our {2, 2} array case.
 *
 * The layout describes in wich order each dimension is ordered internally,
 * and can be used to enhance performance. The usual IOLink layout is
 * {0, 1, ..., n - 1, n}. As we use the Fortran convention in IOLink, indexing a C array
 * would use an inverted layout {n, n-1, ..., 1, 0}. This field enables user to easily know
 * how data is organised without analysing the strides directly.
 */
class IOLINK_API Indexer final
{
public:
  Indexer();

  /**
   * Create an indexer by giving all its informations.
   *
   * Here all the arguments are given, and are not checked, so
   * if they are not consistant, they will give an undefined behavior.
   *
   * @param shape The shape of the array to index.
   * @param stride The stride of each dimension, the offset in elements
   *               between each increment in this dimension.
   * @param layout The internal order of dimensions, {0, 1} is row-major and
   *               {1, 0} is col-major for example.
   *
   * @deprecated Deprecated in 1.8, will be dropped in 2.0, you should use a
   *             constructor that specify the data type instead.
   */
  Indexer(const VectorXu64& shape, const VectorXu64& stride, const VectorXu64& layout);

  /**
   * Create an indexer by giving all its informations.
   *
   * Here all the arguments are given, and are not checked, so
   * if they are not consistant, they will give an undefined behavior.
   *
   * @param shape The shape of the array to index.
   * @param stride The stride of each dimension, the offset in elements
   *               between each increment in this dimension.
   * @param standardLayout A standard layout for the array.
   *
   * @throw Error When the layout is unsupported
   *
   * @deprecated Deprecated in 1.8, will be dropped in 2.0, you should use a
   *             constructor that specify the data type instead.
   */
  Indexer(const VectorXu64& shape, const VectorXu64& stride, MemoryLayout standardLayout);

  /**
   * Initialize the indexer using a custom memory layout.
   *
   * @param shape The shape of the indexed array
   * @param layout The internal order of dimensions, {0, 1} is row-major and
   *               {1, 0} is col-major for example.
   *
   * @deprecated Deprecated in 1.8, will be dropped in 2.0, you should use a
   *             constructor that specify the data type instead.
   */
  Indexer(const VectorXu64& shape, const VectorXu64& layout);

  /**
   * Auto compute the Indexer's strides from shape and standard layout.
   *
   * @param shape The narray sizes in each dimension
   * @param standardLayout A standard layout for the array.
   *
   * @throw Error When the layout is unsupported
   *
   * @deprecated Deprecated in 1.8, will be dropped in 2.0, you should use a
   *             constructor that specify the data type instead.
   */
  Indexer(const VectorXu64& shape, MemoryLayout standardLayout);

  /**
   * Simpliest way to initialize an indexer from a shape.
   *
   * The layout here is Fortran one, and the strides
   * are computed according to it.
   *
   * @param shape The desired shape of the multi dimensional array.
   *
   * @deprecated Deprecated in 1.8, will be dropped in 2.0, you should use a
   *             constructor that specify the data type instead.
   */
  explicit Indexer(const VectorXu64& shape);

  /**
   * @brief Constructor with shape and datatype, using default layout and stride.
   *
   * The default byte stride being computed from the shape and data type.
   * The default layout being the Fortran array layout.
   *
   * @param shape The shape of the indexed array.
   * @param dtype The data type of the indexed array.
   */
  Indexer(const VectorXu64& shape, DataType dtype);

  /**
   * @brief Constructor with shape, datatype and standard layout, using default stride.
   *
   * The default byte stride being computed from the shape, datatype, and standard layout.
   *
   * @param shape The shape of the indexed array.
   * @param dtype The data type of the indexed array.
   * @param standardLayout A standard layout for the array.
   */
  Indexer(const VectorXu64& shape, DataType dtype, MemoryLayout standardLayout);

  /**
   * @brief Constructor with shape, datatype and byte stride, using default layout.
   *
   * The default layout being the Fortran array layout.
   *
   * @param shape The shape of the indexed array.
   * @param dtype The data type of the indexed array.
   * @param byteStride The stride in bytes of the indexed array.
   */
  Indexer(const VectorXu64& shape, DataType dtype, const VectorXu64& byteStride);

  /**
   * @brief Constructor with shape, datatype, byte stride and layout.
   *
   * @param shape The shape of the indexed array.
   * @param dtype The data type of the indexed array.
   * @param byteStride The stride in bytes of the indexed array.
   * @param layout The layout of the indexed array.
   */
  Indexer(const VectorXu64& shape, DataType dtype, const VectorXu64& byteStride, const VectorXu64& layout);

  Indexer(const Indexer& other);
  Indexer& operator=(const Indexer& other);

  Indexer(Indexer&& other) noexcept;
  Indexer& operator=(Indexer&& other) noexcept;

  ~Indexer();

  bool operator==(const Indexer& other) const;
  bool operator!=(const Indexer& other) const;

  /**
   * The number of dimensions managed by this indexer.
   */
  size_t dimensionCount() const;

  /**
   * Return the shape
   */
  const VectorXu64& shape() const;

  /**
   * @return the data type of this Indexer.
   */
  DataType dtype() const;

  /**
   * Return the stride
   *
   * @deprecated Deprecated in 1.8, will be dropped in 2.0, you should use the
   *             byteStride method instead.
   */
  const VectorXu64& stride() const;

  /**
   * @return the stride in bytes of this Indexer.
   */
  const VectorXu64& byteStride() const;

  /**
   * Return the layout dimension order.
   */
  const VectorXu64& layout() const;

  /**
   * Return the current size for given dimension.
   *
   * @param index Index of the dimension to consider
   */
  size_t shape(size_t index) const;

  /**
   * Return the current stride for given dimension.
   *
   * @param index Index of the dimension to consider
   *
   * @deprecated Deprecated in 1.8, will be dropped in 2.0, you should use the
   *             byteStride method instead.
   */
  size_t stride(size_t index) const;

  /**
   * @return the stride in bytes for the given dimension.
   */
  size_t byteStride(size_t index) const;

  /**
   * Return the dimension index of the nth one in order.
   *
   * @param index Index of the order dimension index.
   */
  size_t layout(size_t index) const;

  /**
   * Returns the number of elements managed by this Indexer.
   */
  size_t elementCount() const;

  /**
   * @brief Get the total number of bytes needed to adress this indexer, taking
   *        the padding into account.
   *
   * @return the number of bytes managed by this Indexer.
   */
  size_t byteCount() const;

  /**
   * Return the computed 1D offset corresponding to indexed sample in multi dimensional array.
   *
   * @param index The element's position for which offset must be computed.
   *
   * @deprecated Deprecated in 1.8, will be dropped in 2.0, you should use the
   *             byteOffset method instead.
   */
  size_t offset(const VectorXu64& index) const;

  /**
   * @brief Computes the offset in bytes to the given index.
   *
   * @param index The index of the element for which the offset must be computed.
   *
   * @return the offset in bytes to the given index.
   *
   * @throw InvalidArgument If the index is invalid.
   */
  size_t byteOffset(const VectorXu64& index) const;

  /**
   * @brief Checks if the indexed elements are contiguous.
   *
   * Elements are considered contiguous if they follow each other without any padding.
   *
   * @return true if the indexed elements are continuous, false otherwise.
   */
  bool isContinuous() const;

private:
  class Private;
  Private* m_private;
};

} // end namespace iolink
