// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <array>
#include <cstdint>
#include <initializer_list>
#include <ostream>
#include <string>

#include <iolink/IOLinkAPI.h>
#include <iolink/Vector.h>

namespace iolink
{

/**
 * An arithmetic square matrix.
 *
 * Do not use this class directly, only uses its aliases.
 */
template <typename ValueType, size_t N>
class Matrix
{
public:
  /**
   * Return an identity matrix.
   */
  static Matrix identity();

  /**
   * Initialize an uniform matrix with an unique value.
   *
   * @param value The value to assign to each element of the matrix
   */
  static Matrix uniform(ValueType value);

  /**
   * Default constructor.
   * Object will have indeterminate values.
   */
  Matrix();

  /**
   * Constructor with C-array parameter.
   *
   * With the following matrix:
   *
   * | a11 | a12 | a13 | a14 |@n
   * | a21 | a22 | a23 | a24 |@n
   * | a31 | a32 | a33 | a34 |@n
   * | a41 | a42 | a43 | a44 |@n@n
   *
   * Values should be given to this function in ROW-major order:
   *    a11 a12 a13 a14 a21 a22 a23 a24 ...
   *
   * @param values Pointer on first element to use to initialize the NxN internal values
   */
  explicit Matrix(const ValueType* values);

  /**
   * Constructor with C-array parameter.
   *
   * With the following matrix:
   *
   * | a11 | a12 | a13 | a14 |@n
   * | a21 | a22 | a23 | a24 |@n
   * | a31 | a32 | a33 | a34 |@n
   * | a41 | a42 | a43 | a44 |@n@n
   *
   * Values should be given to this function in:
   * - ROW-major order (a11 a12 a13 a14 a21...)
   * - COL-major order (a11, a21, a31, a41, a12...)
   *
   * @param values Pointer on first element to use to initialize the NxN internal values
   * @param isRowMajor Boolean which indicates how given data must be stored
   */
  Matrix(const ValueType* values, bool isRowMajor);

  /**
   * Create a matrix giving initialization values.
   *
   * With the following matrix:
   *
   * | a11 | a12 | a13 | a14 |@n
   * | a21 | a22 | a23 | a24 |@n
   * | a31 | a32 | a33 | a34 |@n
   * | a41 | a42 | a43 | a44 |@n@n
   *
   * This fonction should be used this way:
   *
   * Matrix4f m{a11, a12, a13, a14,
   *            a21, a22, a23, a24,
   *            a31, a32, a33, a34,
   *            a41, a42, a43, a44};
   */
  explicit Matrix(std::initializer_list<ValueType> init);

  bool operator==(const Matrix& other) const;
  bool operator!=(const Matrix& other) const;

  /**
   * Access an element of the matrix.
   *
   * @param column The column of the element
   * @param row The row of the element
   */
  ValueType at(size_t row, size_t column) const;

  /**
   * Access an element of the matrix.
   *
   * @param column The column of the element
   * @param row The row of the element
   */
  ValueType operator()(size_t row, size_t column) const;

  /**
   * Set an element of the matrix.
   *
   * @param column The column of the element
   * @param row The row of the element
   * @param value The value to assign to the element
   */
  void setAt(size_t row, size_t column, ValueType value);

  /**
   * Transpose the matrix in place.
   */
  void transposeInPlace();

  /**
   * Return the transposed matrix.
   */
  Matrix transpose() const;

  /**
   * @brief Return the inverse matrix.
   *
   * @warning Calling this method on a non-invertable matrix will cause
   *          undefined behaviour.
   */
  Matrix inverse() const;

  /**
   * Exposes the internal data of the matrix, use it only if you know what you do.
   *
   * As the internal layout of our matrices is column-major, the data will be
   * in a different order than the initialisation list.
   *
   * Example for the following matrix:
   *
   * | a11 | a12 | a13 | a14 |@n
   * | a21 | a22 | a23 | a24 |@n
   * | a31 | a32 | a33 | a34 |@n
   * | a41 | a42 | a43 | a44 |@n@n
   *
   * Data will return a buffer with the following data:
   *
   * {a11, a21, a31, a41, a12, a22, a32 a42, a13, a23, a33, a43, a14, a24, a34, a44}
   *
   */
  ValueType* data();

  /**
   * @copydoc data()
   */
  const ValueType* data() const;

  /**
   * Return a string representation of the matrix.
   */
  std::string toString() const;

  // ==================== //
  // Arithmetic operators //
  // ==================== //

  Matrix operator-() const;

  Matrix& operator+=(const Matrix& m);
  Matrix& operator-=(const Matrix& m);

  inline Matrix operator+(const Matrix& m) const { return Matrix(*this) += m; }
  inline Matrix operator-(const Matrix& m) const { return Matrix(*this) -= m; }

  Matrix& operator*=(ValueType value);
  Matrix& operator/=(ValueType value);

  inline Matrix operator*(ValueType v) const { return Matrix(*this) *= v; }
  inline Matrix operator/(ValueType v) const { return Matrix(*this) /= v; }

  Vector<ValueType, N> operator*(const Vector<ValueType, N>& v) const;
  Matrix operator*(const Matrix& m) const;

private:
  std::array<ValueType, N * N> m_data;
};

/**
 * Stream operator to be able to print matrices in standard streams.
 */
template <typename T, size_t N>
inline std::ostream&
operator<<(std::ostream& os, const Matrix<T, N>& m)
{
  os << m.toString();
  return os;
}

//=======================//
// Template declarations //
//=======================//

extern template class IOLINK_API_IMPORT Matrix<float, 3>;
extern template class IOLINK_API_IMPORT Matrix<double, 3>;

extern template class IOLINK_API_IMPORT Matrix<float, 4>;
extern template class IOLINK_API_IMPORT Matrix<double, 4>;

//======================//
// Aliases declarations //
//======================//

using Matrix3f = Matrix<float, 3>;  ///< Square matrix of size 3, storing float data
using Matrix3d = Matrix<double, 3>; ///< Square matrix of size 3, storing double data

using Matrix4f = Matrix<float, 4>;  ///< Square matrix of size 4, storing float data
using Matrix4d = Matrix<double, 4>; ///< Square matrix of size 4, storing double data

} // end namespace iolink
