// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <string>

#include <iolink/IOLinkAPI.h>
#include <iolink/Matrix.h>
#include <iolink/Vector.h>

namespace iolink
{

/**
 * @brief Create matrices for 2D transforms.
 */
class IOLINK_API TransformMatrix3Factory
{
public:
  /**
   * @brief Create a translation matrix.
   *
   * @param v The vector used for the translation.
   * @return The translation matrix.
   */
  static Matrix3d translation(const Vector2d& v);

  /**
   * @brief Create a rotation matrix.
   *
   * The rotation is done counter-clockwise, with the Y axis being up.
   *
   * @param angle The angle of the rotation.
   * @return The rotation matrix.
   */
  static Matrix3d rotation(double angle);

  /**
   * @brief Create a Scaling matrix.
   *
   * @param v The scaling factors.
   * @return The scaling matrix.
   */
  static Matrix3d scaling(const Vector2d& v);
};

/**
 * @brief Create matrices for 3D transforms.
 */
class IOLINK_API TransformMatrix4Factory
{
public:
  /**
   * @brief Create a translation matrix.
   *
   * @param v The vector used for the translation.
   * @return The translation matrix.
   */
  static Matrix4d translation(const Vector3d& v);

  /**
   * @brief Create a rotation matrix for a rotation using Euler angles.
   *
   * The rotation is described by three Euler angles, chaining rotations, following
   * a given convention.
   *
   * The convention is decribed by a three-character-string, where each character
   * describes the axis of rotation for the corresponding angle.
   * A lowercase convention means that we are describing an extrinsic rotation,
   * while an uppercase convention means that we are describing an intrinsic
   * rotation. A mixed case convention is invalid.
   *
   * Extrinsinc and intrinsinc conventions are the inverse of each other, so
   * for example, intrinsinc convention "ZYX" is equivalent to the extrinsic
   * convention "xyz".
   *
   * Common conventions:
   * - "ZYX": Rotation around Z, then Y, then X.
   * - "ZXZ": Rotation around Z, then X, then Z.
   *
   * @param angles The Euler angles of the rotation.
   * @param convention The convention used for the rotation.
   * @return The rotation matrix.
   *
   * @throw InvalidArgument If the convention is invalid.
   */
  static Matrix4d rotationEuler(const Vector3d& angles, const std::string& convention);

  /**
   * @brief Create a rotation matrix for a rotation around an axis.
   *
   * Rotation is done counter-clockwise, around the given axis.
   *
   * @param angle The rotation angle.
   * @param axis The axis around which the rotation is done, must be normalized.
   * @return The rotation matrix.
   */
  static Matrix4d rotationOnAxis(double angle, const Vector3d& axis);

  /**
   * @brief Create a scaling matrix.
   *
   * @param v The scaling factors.
   * @return The scaling matrix.
   */
  static Matrix4d scaling(const Vector3d& v);
};

} // namespace iolink
