// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <iolink/IOLinkAPI.h>
#include <iolink/VectorX.h>

#include <array>
#include <cstdint>
#include <initializer_list>
#include <iostream>

namespace iolink
{

/**
 * An arithmetic vector.
 *
 * @tparam T The type wrote in the vector.
 * @tparam N The size, or number of dimension of the vector.
 *
 * Aliases can be used to made this class more practical to use. They use the
 * following convention:
 *    VectorNX
 * Where N is the size and X the type. The types currently available are:
 * - U8 and I8 respectively for unsigned and signed 8bits integers.
 * - U16 and I16 respectively for unsigned and signed 16bits integers.
 * - U32 and I32 respectively for unsigned and signed 32bits integers.
 * - U64 and I64 respectively for unsigned and signed 64bits integers.
 * - F for single precision floats.
 * - D for double precision floats.
 *
 * Examples:
 * - Vector2i32: 2 dimensional 32bits signed integer.
 * - Vector3f: 3 dimensionnal single precision float.
 */
template <typename T, size_t N>
class Vector final
{
public:
  using ValueType = T;

  using Iterator = typename std::array<T, N>::iterator;
  using ConstIterator = typename std::array<T, N>::const_iterator;

  /**
   * Create a vector with the given value for all components.
   *
   * @param value The value wich the vector's data will be filled.
   */
  static Vector createUniform(ValueType value);

  /**
   * Create a vector with unitialized values.
   */
  Vector();

  /**
   * Create a vector with the values contained in a bracket list.
   *
   * @throw InvalidArgument when initializing with a wrong number of elements
   */
  Vector(std::initializer_list<ValueType> init);

  /**
   * Build a Vector from a VectorX of same type.
   *
   * If the VectorX has a size different of N, behavior is undefined.
   */
  Vector(const VectorX<ValueType>& other);

  /**
   * Build a vector from a VectorX of same type.
   *
   * @param other VectorX to use to build the new vector
   * @param padding Value to put at the end of vector to fill missing values.
   *                Fox example, Vector4({4, 5}, 0) will create the Vector
   *                {4, 5, 0, 0}.
   */
  Vector(const VectorX<ValueType>& other, ValueType padding);

  /**
   * Build a vector from a vector storing another type.
   *
   * @tparam U The type of the source vector.
   * @param other The vector to convert.
   */
  template <typename U>
  explicit Vector(const Vector<U, N>& other)
  {
    for (size_t i = 0; i < N; ++i)
    {
      m_data[i] = static_cast<ValueType>(other[i]);
    }
  }

  Vector(const Vector& other) = default;
  Vector& operator=(const Vector& other) = default;

  Vector(Vector&& other) noexcept = default;
  Vector& operator=(Vector&& other) noexcept = default;

  /**
   * Returns the size of the vector, which also correspond to vector number of dimension.
   */
  size_t size() const { return N; }

  /**
   * Access vector's value at given index.
   */
  inline ValueType at(size_t index) const { return m_data[index]; }

  /**
   * Set value as given index.
   */
  inline void setAt(size_t index, ValueType value) { m_data[index] = value; }

  inline ValueType operator[](size_t index) const { return m_data[index]; }
  inline ValueType& operator[](size_t index) { return m_data[index]; }

  /**
   * Direct access to the vector's data.
   */
  inline ValueType* data() { return m_data.data(); }

  /**
   * Direct const access to the vector's data.
   */
  inline const ValueType* data() const { return m_data.data(); }

  /**
   * Returns the vector's norm, squared.
   *
   * Specifically, it is the L1 norm that's computed here.
   */
  double squaredLength() const;

  /**
   * Returns the vector's norm.
   *
   * Specifically, it is the L1 norm that's computed here.
   */
  double length() const;

  /**
   * Normalize the vector
   */
  void normalize();

  Iterator begin() noexcept { return m_data.begin(); }
  ConstIterator begin() const noexcept { return m_data.begin(); }
  Iterator end() noexcept { return m_data.end(); }
  ConstIterator end() const noexcept { return m_data.end(); }

  bool operator==(const Vector& other) const;
  bool operator!=(const Vector& other) const;

  std::string toString() const;

  // ==================== //
  // Arithmetic operators //
  // ==================== //

  Vector operator-() const;

  Vector& operator+=(const Vector& v);
  Vector& operator-=(const Vector& v);

  inline Vector operator+(const Vector& v) const { return Vector(*this) += v; }
  inline Vector operator-(const Vector& v) const { return Vector(*this) -= v; }

  Vector& operator*=(ValueType value);
  Vector& operator/=(ValueType value);

  inline Vector operator*(ValueType v) const { return Vector(*this) *= v; }
  inline Vector operator/(ValueType v) const { return Vector(*this) /= v; }

  friend inline Vector operator*(ValueType value, Vector v) { return v *= value; }

  /**
   * Dot product of two vectors.
   */
  ValueType dot(const Vector& v) const;

  /**
   * Cross product of two vectors.
   *
   * Work only for vectors of size 3.
   */
  Vector cross(const Vector& v) const;

  // ==================== //
  // GLSL style operators //
  // ==================== //

  Vector& operator*=(const Vector& v);
  Vector& operator/=(const Vector& v);

  inline Vector operator*(const Vector& v) const { return Vector(*this) *= v; }
  inline Vector operator/(const Vector& v) const { return Vector(*this) /= v; }

  bool operator<(const Vector& other) const;
  bool operator<=(const Vector& other) const;
  bool operator>(const Vector& other) const;
  bool operator>=(const Vector& other) const;

  operator VectorX<ValueType>() const;

private:
  std::array<ValueType, N> m_data;
};

template <typename T, size_t N>
inline std::ostream&
operator<<(std::ostream& os, const Vector<T, N>& v)
{
  os << v.toString();
  return os;
}

// ===================== //
// Template declarations //
// ===================== //

//------------
// 2D

extern template class IOLINK_API_IMPORT Vector<int8_t, 2>;
extern template class IOLINK_API_IMPORT Vector<int16_t, 2>;
extern template class IOLINK_API_IMPORT Vector<int32_t, 2>;
extern template class IOLINK_API_IMPORT Vector<int64_t, 2>;

extern template class IOLINK_API_IMPORT Vector<uint8_t, 2>;
extern template class IOLINK_API_IMPORT Vector<uint16_t, 2>;
extern template class IOLINK_API_IMPORT Vector<uint32_t, 2>;
extern template class IOLINK_API_IMPORT Vector<uint64_t, 2>;

extern template class IOLINK_API_IMPORT Vector<float, 2>;
extern template class IOLINK_API_IMPORT Vector<double, 2>;

//------------
// 3D

extern template class IOLINK_API_IMPORT Vector<int8_t, 3>;
extern template class IOLINK_API_IMPORT Vector<int16_t, 3>;
extern template class IOLINK_API_IMPORT Vector<int32_t, 3>;
extern template class IOLINK_API_IMPORT Vector<int64_t, 3>;

extern template class IOLINK_API_IMPORT Vector<uint8_t, 3>;
extern template class IOLINK_API_IMPORT Vector<uint16_t, 3>;
extern template class IOLINK_API_IMPORT Vector<uint32_t, 3>;
extern template class IOLINK_API_IMPORT Vector<uint64_t, 3>;

extern template class IOLINK_API_IMPORT Vector<float, 3>;
extern template class IOLINK_API_IMPORT Vector<double, 3>;

//------------
// 3D

extern template class IOLINK_API_IMPORT Vector<int8_t, 4>;
extern template class IOLINK_API_IMPORT Vector<int16_t, 4>;
extern template class IOLINK_API_IMPORT Vector<int32_t, 4>;
extern template class IOLINK_API_IMPORT Vector<int64_t, 4>;

extern template class IOLINK_API_IMPORT Vector<uint8_t, 4>;
extern template class IOLINK_API_IMPORT Vector<uint16_t, 4>;
extern template class IOLINK_API_IMPORT Vector<uint32_t, 4>;
extern template class IOLINK_API_IMPORT Vector<uint64_t, 4>;

extern template class IOLINK_API_IMPORT Vector<float, 4>;
extern template class IOLINK_API_IMPORT Vector<double, 4>;

//==================== //
// Aliases declaration //
//==================== //

//------------
// 2D

using Vector2u8 = Vector<uint8_t, 2>;   ///< Alias for uint8_t 2D vectors
using Vector2u16 = Vector<uint16_t, 2>; ///< Alias for uint16_t 2D vectors
using Vector2u32 = Vector<uint32_t, 2>; ///< Alias for uint32_t 2D vectors
using Vector2u64 = Vector<uint64_t, 2>; ///< Alias for uint64_t 2D vectors
using Vector2i8 = Vector<int8_t, 2>;    ///< Alias for int8_t 2D vectors
using Vector2i16 = Vector<int16_t, 2>;  ///< Alias for int16_t 2D vectors
using Vector2i32 = Vector<int32_t, 2>;  ///< Alias for int32_t 2D vectors
using Vector2i64 = Vector<int64_t, 2>;  ///< Alias for int64_t 2D vectors
using Vector2f = Vector<float, 2>;      ///< Alias for float 2D vectors
using Vector2d = Vector<double, 2>;     ///< Alias for double 2D vectors

//------------
// 3D

using Vector3u8 = Vector<uint8_t, 3>;   ///< Alias for uint8_t 3D vectors
using Vector3u16 = Vector<uint16_t, 3>; ///< Alias for uint16_t 3D vectors
using Vector3u32 = Vector<uint32_t, 3>; ///< Alias for uint32_t 3D vectors
using Vector3u64 = Vector<uint64_t, 3>; ///< Alias for uint64_t 3D vectors
using Vector3i8 = Vector<int8_t, 3>;    ///< Alias for int8_t 3D vectors
using Vector3i16 = Vector<int16_t, 3>;  ///< Alias for int16_t 3D vectors
using Vector3i32 = Vector<int32_t, 3>;  ///< Alias for int32_t 3D vectors
using Vector3i64 = Vector<int64_t, 3>;  ///< Alias for int64_t 3D vectors
using Vector3f = Vector<float, 3>;      ///< Alias for float 3D vectors
using Vector3d = Vector<double, 3>;     ///< Alias for double 3D vectors

//------------
// 4D

using Vector4u8 = Vector<uint8_t, 4>;   ///< Alias for uint8_t 4D vectors
using Vector4u16 = Vector<uint16_t, 4>; ///< Alias for uint16_t 4D vectors
using Vector4u32 = Vector<uint32_t, 4>; ///< Alias for uint32_t 4D vectors
using Vector4u64 = Vector<uint64_t, 4>; ///< Alias for uint64_t 4D vectors
using Vector4i8 = Vector<int8_t, 4>;    ///< Alias for int8_t 4D vectors
using Vector4i16 = Vector<int16_t, 4>;  ///< Alias for int16_t 4D vectors
using Vector4i32 = Vector<int32_t, 4>;  ///< Alias for int32_t 4D vectors
using Vector4i64 = Vector<int64_t, 4>;  ///< Alias for int64_t 4D vectors
using Vector4f = Vector<float, 4>;      ///< Alias for float 4D vectors
using Vector4d = Vector<double, 4>;     ///< Alias for double 4D vectors

} // end namespace iolink
