// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <memory>
#include <mutex>
#include <string>

#include <iolink/VariantDataValue.h>
#include <iolink/metadata/MetadataNodeIterator.h>

namespace iolink
{

/**
 *  Class which represents one node from a metadata tree. Each node contains
 *  a key and also an associated value (optional). It also contains a list of nodes which
 *  represents its children in the tree.
 *  Children can be parsed using '(Const)MetadataNodeIterator' objects provided by
 *  '(c)begin' and '(c)end' methods, or directly can be retrieved thanks to 'child' accessor method.
 */
class IOLINK_API MetadataNode
{
public:
  /**
   * Constructor which takes a mandatory key, and an optional value.
   * @param key string which identifies the node
   * @param value VariantDataValue object associated to the key (or null)
   */
  MetadataNode(const std::string& key, std::shared_ptr<VariantDataValue> value);

  MetadataNode(const MetadataNode& other);
  MetadataNode& operator=(const MetadataNode& other);

  MetadataNode(MetadataNode&& other) noexcept;
  MetadataNode& operator=(MetadataNode&& other) noexcept;

  ~MetadataNode();

  /**
   *  Method to clone the current node
   *  @return the new node
   */
  std::shared_ptr<MetadataNode> clone() const;

  /**
   * Accessor to the key of the current node
   * @return the current node key
   */
  const std::string& key() const;

  /**
   * Accessor to the value of the current node
   * @return the current node value
   */
  std::shared_ptr<VariantDataValue> value() const;

  /**
   * Return the children's count of this node.
   */
  size_t childCount() const;

  /**
   * Indicate if a child identified by its key is present or not.
   *
   * @param key the key corresponding to a child or not of current node
   * @return If the node has a child with given key
   */
  bool hasChild(const std::string& key) const;

  /**
   * Return the child node with the given key.
   *
   * @param key the key corresponding to one of the children of current node
   * @return the child corresponding to given key
   * @throw Error If the key is not found in the children of the current node
   */
  std::shared_ptr<const MetadataNode> child(const std::string& key) const;

  /**
   * Return the child node with the given key.
   *
   * @param key the key corresponding to one of the children of current node
   * @return the child corresponding to given key
   * @throw Error If the key is not found in the children of the current node
   */
  std::shared_ptr<MetadataNode> child(const std::string& key);

  /**
   * Add a node as child. If the given child key already exists in the current node,
   * existing child is replaced by new one
   * @param child node to add as child to the current node
   */
  void addChild(std::shared_ptr<MetadataNode> child);

  /**
   * Remove child identified by its key
   * @param key Key whose node must be removed
   */
  void removeChild(const std::string& key);

  /**
   * Return iterator on first child
   */
  MetadataNodeIterator begin();

  /**
   * Return iterator which indicates the end of children list
   */
  MetadataNodeIterator end();

  /**
   * Return iterator on first child
   */
  ConstMetadataNodeIterator begin() const;

  /**
   * Return iterator which indicates the end of children list
   */
  ConstMetadataNodeIterator end() const;

  /**
   * Return const iterator on first child
   */
  ConstMetadataNodeIterator cbegin() const;

  /**
   * Return const iterator which indicates the end of children list
   */
  ConstMetadataNodeIterator cend() const;

  /**
   * Return a string representation
   */
  std::string toString() const;

private:
  class Private;
  Private* m_private;
};

} // end namespace iolink
