// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <iolink/DataType.h>
#include <iolink/IOLinkAPI.h>
#include <iolink/ImageType.h>
#include <iolink/Vector.h>

#include <cstdint>

namespace iolink
{

/**
 * Interpretation of an Image
 */
enum class ImageInterpretation
{
  /** No interpretation at all, only raw data */
  UNKNOWN = 0x00,

  //==============//
  // Mono channel //
  //==============//

  /** A shade of grey, from black to white */
  GRAYSCALE = 0x01,

  /** Indices corresponding to a label */
  LABEL = 0x02,

  /** A boolean */
  BINARY = 0x03,

  //==================//
  // CIE color spaces //
  //==================//

  /** CIE XYZ color space (CIE 1931) */
  CIE_XYZ = 0x11,

  /** CIE XYZ color with chromaticity coordinates */
  CIE_XYZ_CHROMATICITY = 0x12,

  /** CIE U*V*W* color space (CIE 1964) */
  CIE_UVW = 0x14,

  /** CIE L*u*v* color space (CIE 1976) */
  CIE_LUV = 0x15,

  /** CIE L*a*b* color space (CIE 1976) */
  CIE_LAB = 0x16,

  //=====================//
  // RGB color encodings //
  //=====================//

  /** RGB color (CIE 1931) */
  RGB = 0x21,

  /** RGB color with chromaticity coordinates */
  RGB_CHROMATICITY = 0x22,

  /** Standard RGB color */
  SRGB = 0x23,

  //=============================//
  // Luma/Chroma color encodings //
  //=============================//

  /** YUV color */
  YUV = 0x31,

  /**
   * YCbCr color (PAL)
   *
   * Alias: YPbPr
   */
  YCC = 0x32,

  /** YIQ color (NTSC) */
  YIQ = 0x33,

  /** YDbDr color (SECAM & PAL-N) */
  YDD = 0x34,

  //===========================//
  // HSX color encoding family //
  //===========================//

  /**
   * HSL color (Hue, Saturation, Lightness)
   *
   * Aliases: HLS, HSI
   */
  HSL = 0x41,

  /** HSV color (Hue, Saturation, Value)
   *
   * Alias: HSB
   */
  HSV = 0x42,

  //============================//
  // CMYK color encoding family //
  //============================//

  /** CMYK color (Cyan, Magenta, Yellow, blacK) */
  CMYK = 0x51,

  //==========================//
  // Specific interpretations //
  //==========================//

  /** Multiple layers of spectral data */
  MULTISPECTRAL = 0xFF,
};

/**
 * This property describes the way the axes must be interpreted (IMAGE, VOLUME, SEQUENCE...) and also the
 * composition of the image by providing :
 *  - its interpretation (GRAYSCALE, RGB, ...)
 *  - the presence of a alpha channel
 *  - the bit depth where data is stored
 *  - the range of value of the data (It is actually the min and max values that are stored in the image)
 */
class IOLINK_API ImageInfoProperty final
{
public:
  /**
   * Method which gives you the number of channels for one pixel in a specific ImageInterpretation
   * @param interpretation ImageInterpretation for which we want to know the channel count
   * @return the channel count corresponding to the given ImageInterpretation
   */
  static size_t channelCount(ImageInterpretation interpretation);

  /**
   * Constructor
   * @param axesInterpretation Axes interpretation for the image
   * @param interpretation The image interpretation
   * @param hasAlpha Indicate if an Alpha channel is present
   * @param bitDepth Bit count effectively used to store each component of the pixel
   * @param valueRange Range of value of the component
   */
  ImageInfoProperty(ImageType axesInterpretation,
                    ImageInterpretation interpretation,
                    bool hasAlpha,
                    size_t bitDepth,
                    const Vector2d& valueRange);

  /**
   * Short constructor using the given data type standard properties to initialize with default values
   * @param axesInterpretation Axes interpretation for the image
   * @param interpretation The image interpretation
   * @param hasAlpha Indicate if an Alpha channel is present
   * @param dataType DataType to use as reference to initialize member data.
   */
  explicit ImageInfoProperty(ImageType axesInterpretation,
                             ImageInterpretation interpretation,
                             bool hasAlpha,
                             DataType dataType);

  /**
   * Shorter constructor using the given data type standard properties to initialize.
   * BitDepth and value range is initialized thanks to dataType.
   * hasAlpha property is set to False. Other properties are set to UNKNOWN.
   *
   * @param dataType DataType to use as reference to initialize member data.
   */
  explicit ImageInfoProperty(DataType dataType);

  /**
   * Shorter constructor using the given Image Type and Data type standard properties
   * hasAlpha property is set to False. Image interpretation is set to UNKNOWN.
   *
   * @param axesInterpretation Axes interpretation for the image
   * @param dataType DataType to use as reference to initialize member data.
   */
  ImageInfoProperty(ImageType axesInterpretation, DataType dataType);

  ImageInfoProperty(const ImageInfoProperty& other);
  ImageInfoProperty& operator=(const ImageInfoProperty& other);

  ImageInfoProperty(ImageInfoProperty&& other) noexcept;
  ImageInfoProperty& operator=(ImageInfoProperty&& other) noexcept;

  ~ImageInfoProperty();

  /**
   * Return an ImageType describing the interpretation of the axes of the image.
   *
   * For example a classical image ImageType will have the ImageDimension::COLUMN and
   * ImageDimension::ROW set. A sequence of 2D images will have ImageDimension::SEQUENCE
   * in addition.
   */
  ImageType axesInterpretation() const;

  /**
   *  Set the image's axes interpretation.
   *
   * @param type Type of Image to set
   */
  void setAxesInterpretation(ImageType type);

  /**
   *  Getter for Image interpretation
   */
  ImageInterpretation interpretation() const;

  /**
   * Return if an alpha channel is present
   */
  bool hasAlpha() const;

  /**
   * The bit depth of pixel components.
   *
   * For example, data can be stored on 16 bits, but only 12 of those
   * are used, so the image view will have a data type of UINT16, but
   * its real bit depth will be 12.
   */
  size_t bitDepth() const;

  /**
   * The value range of the pixel component
   */
  Vector2d valueRange() const;

  /**
   *  Setter for Image Interpretation
   */
  void setInterpretation(ImageInterpretation interpretation);

  /**
   *  Set if an alpha channel is present
   */
  void setAlpha(bool hasAlpha);

  /**
   * Set the bit depth of pixel components.
   */
  void setBitDepth(size_t bitdepth);

  /**
   * Set the value range of components
   */
  void setValueRange(const Vector2d& range);

  /**
   * Equality operator
   */
  bool operator==(const ImageInfoProperty& other) const;

  /**
   * Return string representation
   */
  std::string toString() const;

private:
  class Private;
  Private* m_private;
};

} // end namespace iolink
