// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <cstdint>
#include <memory>
#include <string>

#include <iolink/IOLinkAPI.h>
#include <iolink/Vector.h>
#include <iolink/property/SpatialDirections.h>

namespace iolink
{

/**
 * This Property is used to represent the image properties in world space.
 * - origin: the position of the image in space.
 * - spacing: the size of samples along each axis.
 * - orientation: the direction cosines among each axis.
 * - unit: the unit of the three axes defined by the precedent attributes.
 *
 * The units are described by a string following a basic convention:
 * Abridged units names, "cm" for centimeters or "in" for inches for example.
 * If there is no commonly used abridged unit, just use the unit name.
 * No punctuation at the end, unless it is part of the abridged notation.
 *
 * If the dimension has no unit, it is described by the empty string.
 *
 * Some examples:
 *  - "µm" for micrometers
 *  - "ft" for foot
 *  - "mi" for a mile
 *  - "Å" for angstrom
 *  - "min" for minute
 */
class IOLINK_API SpatialCalibrationProperty final
{
public:
  /**
   * Initialize the property with default values.
   *
   *  - origin: {0.0, 0.0, 0.0}
   *  - spacing: {1.0, 1.0, 1.0}
   *  - orientation: {1.0, 1.0, 1.0}
   *  - unit: ""
   */
  SpatialCalibrationProperty();

  /**
   * Initialize the property with only a custom origin.
   *
   * @param origin origin the origin of the calibration
   */
  explicit SpatialCalibrationProperty(const Vector3d& origin);

  /**
   * Initialize a calibration property with an origin and a spacing.
   *
   * @param origin The origin position of the ImageView.
   * @param spacing The size of each element of the raster grid.
   */
  SpatialCalibrationProperty(const Vector3d& origin, const Vector3d& spacing);

  /**
   * Initialize a calibration without a specific unit.
   *
   * @param origin The origin position of the ImageView.
   * @param spacing The size of each element of the raster grid.
   * @param direction The directions for each axes.
   */
  SpatialCalibrationProperty(const Vector3d& origin, const Vector3d& spacing, const SpatialDirections& direction);

  /**
   * Initialize a calibration property.
   *
   * @param origin The origin position of the ImageView.
   * @param spacing The size of each element of the raster grid.
   * @param direction The directions for each axes.
   * @param unit The unit of the axes.
   */
  SpatialCalibrationProperty(const Vector3d& origin,
                             const Vector3d& spacing,
                             const SpatialDirections& direction,
                             const std::string& unit);

  /**
   * Copy constructor.
   *
   * @param other The calibration to copy
   */
  SpatialCalibrationProperty(const SpatialCalibrationProperty& other);

  /**
   * Assignment operator.
   *
   * @param other The calibration to copy
   */
  SpatialCalibrationProperty& operator=(const SpatialCalibrationProperty& other);

  /**
   * Move constructor.
   *
   * @param other The calibration to copy
   */
  SpatialCalibrationProperty(SpatialCalibrationProperty&& other) noexcept;

  /**
   * Move assignment operator.
   *
   * @param other The calibration to copy
   */
  SpatialCalibrationProperty& operator=(SpatialCalibrationProperty&& other) noexcept;

  /**
   * Destructor.
   */
  ~SpatialCalibrationProperty();

  /**
   * Get the origin.
   */
  const Vector3d& origin() const;

  /**
   * Get the spacing.
   */
  const Vector3d& spacing() const;

  /**
   * Get the spatial axis system.
   */
  const SpatialDirections& directions() const;

  /**
   * Get the unit.
   */
  const std::string& unit() const;

  /**
   * Set the origin.
   */
  void setOrigin(const Vector3d& origin);

  /**
   * Set the spacing.
   */
  void setSpacing(const Vector3d& spacing);

  /**
   * Set the directions.
   */
  void setDirections(const SpatialDirections& directions);

  /**
   * Set the unit.
   */
  void setUnit(const std::string& unit);

  /**
   * Equality operator.
   *
   * @param other calibration property to compare to current one
   */
  bool operator==(const SpatialCalibrationProperty& other) const;

  /**
   * Not Equality operator.
   *
   * @param other calibration property to compare to current one
   */
  bool operator!=(const SpatialCalibrationProperty& other) const;

  /**
   * Return string representation
   */
  std::string toString() const;

private:
  class Private;
  Private* m_private;
};

} // end namespace iolink
