// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <iolink/IOLinkAPI.h>
#include <iolink/MemoryLayout.h>
#include <iolink/storage/RandomAccess.h>
#include <iolink/view/ImageView.h>
#include <iolink/view/MultiImageView.h>

#include <memory>

namespace iolink
{

/**
 * This factory is aimed at creating dataset views.
 *
 * By default, ImageViews returned by this factory are __not__ thread safe.
 * Only #makeThreadSafe method allows to apply threadsafety guarantee on an ImageView.
 *
 *
 * [READ]: \ref #ImageCapability.READ "READ"
 * [WRITE]: \ref #ImageCapability.WRITE "WRITE"
 * [BUFFER]: \ref #ImageCapability.BUFFER "BUFFER"
 * [RESHAPE]: \ref #ImageCapability.RESHAPE "RESHAPE"
 */
class IOLINK_API ImageViewFactory
{
public:
  ImageViewFactory() = delete;
  ImageViewFactory(const ImageViewFactory& other) = delete;            // copy constructor
  ImageViewFactory(ImageViewFactory&& other) = delete;                 // move constructor
  ImageViewFactory& operator=(const ImageViewFactory& other) = delete; // copy assignment
  ImageViewFactory& operator=(ImageViewFactory&& other) = delete;      // move assignment

  // ==================== //
  // Creating Image Views //
  // ==================== //

  /**
   * Creates a memory image with the given shape and type.
   *
   * Memory is allocated by this operation.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   * - WRITE
   * - BUFFER
   * - RESHAPE
   *
   * @param shape The shape of the image (its dimension and sizes in all dimensions)
   * @param type The type of data contained in that view.
   * @param properties Properties to set to the allocated ImageView. If null, default properties are generated.
   * @param metadata Metadata to set to the allocated ImageView
   *
   * @throw Error If given properties are not compatible with given shape and type
   *
   */
  static std::shared_ptr<ImageView> allocate(const VectorXu64& shape,
                                             DataType type,
                                             std::shared_ptr<ImageProperties> properties,
                                             std::shared_ptr<MetadataNode> metadata);

  /**
   * Creates a memory image with the given shape and type.
   *
   * Memory is allocated by this operation.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   * - WRITE
   * - BUFFER
   * - RESHAPE
   *
   * @param shape The shape of the image (its dimension and sizes in all dimensions)
   * @param type The type of data contained in that view.
   */
  static std::shared_ptr<ImageView> allocate(const VectorXu64& shape, DataType type);

  /**
   * Copy given image in memory.
   *
   * Memory is allocated by this operation. Data from original ImageView (if not in memory)
   * are duplicated.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   * - WRITE
   * - BUFFER
   * - RESHAPE
   *
   * If the input image already is in memory, return it (no copy are done).
   *
   * @param image Image to load
   *
   * @throw InvalidArgument If the source is null
   * @throw Error If the input ImageView does not support READ capability
   * @throw InvalidArgument If properties of input ImageView are not valid
   */
  static std::shared_ptr<ImageView> copyInMemory(std::shared_ptr<ImageView> image);

  /**
   * Create an image with the data stored into a temporary file (in given folder) on disk in raw.
   *
   * No memory is allocated by this operation, only disk space is used.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   * - WRITE
   * - RESHAPE
   *
   * The created file is automatically deleted when destroying the image.
   *
   * @param dirPath the path to the folder where the temporary file will be created
   * @param shape The shape of the image (its dimension and sizes in all dimensions)
   * @param type The type of data contained in that view.
   * @param properties Properties to set to the allocated ImageView
   * @param metadata Metadata to set to the allocated ImageView
   *
   * @throw Error if given properties are not compatible with given shape and type
   * @throw InvalidArgument if given directory path does not exist
   */
  static std::shared_ptr<ImageView> createOnDisk(const std::string& dirPath,
                                                 const VectorXu64& shape,
                                                 DataType type,
                                                 std::shared_ptr<ImageProperties> properties,
                                                 std::shared_ptr<MetadataNode> metadata);

  /**
   * Create an image with the data stored into a temporary file (in given folder) on disk in raw.
   *
   * No memory is allocated by this operation, only disk space is used.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   * - WRITE
   * - RESHAPE
   *
   * The created file is automatically deleted when destroying the image.
   *
   * @param dirPath the path to the folder where the temporary file will be created
   * @param shape The shape of the image (its dimension and sizes in all dimensions)
   * @param type The type of data contained in that view.
   *
   * @throw InvalidArgument if given directory path does not exist
   */
  static std::shared_ptr<ImageView> createOnDisk(const std::string& dirPath, const VectorXu64& shape, DataType type);

  /**
   * Create an image with the data stored into a temporary file on disk in raw.
   *
   * No memory is allocated by this operation, only disk space is used.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   * - WRITE
   * - RESHAPE
   *
   * The file is created in usual temporary directory and automatically deleted
   * when destroying the image.
   *
   * @param shape The shape of the image (its dimension and sizes in all dimensions)
   * @param type The type of data contained in that view.
   * @param properties Properties to set to the allocated ImageView
   * @param metadata Metadata to set to the allocated ImageView
   *
   * @throw Error if given properties are not compatible with given shape and type
   */
  static std::shared_ptr<ImageView> createOnDisk(const VectorXu64& shape,
                                                 DataType type,
                                                 std::shared_ptr<ImageProperties> properties,
                                                 std::shared_ptr<MetadataNode> metadata);

  /**
   * Create an image with the data stored into a temporary file on disk in raw.
   *
   * No memory is allocated by this operation, only disk space is used.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   * - WRITE
   * - RESHAPE
   *
   * The file is created in usual temporary directory and automatically deleted
   * when destroying the image.
   *
   * @param shape The shape of the image (its dimension and sizes in all dimensions)
   * @param type The type of data contained in that view.
   */
  static std::shared_ptr<ImageView> createOnDisk(const VectorXu64& shape, DataType type);

  /**
   * Load an image data into a temporary raw file on disk.
   *
   * The file is created in usual temporary directory and will be automatically deleted
   * from disk when the instance of the returned ImageView is deleted.
   *
   * Samples are copied chunk by chunk to optimize memory consumption.
   * Maximum chunk size is 1MB.
   *
   * No memory is allocated by this operation, only disk space is used.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   * - WRITE
   * - RESHAPE
   *
   * Original ImageView data are duplicated. Data from the returned ImageView are not linked to
   * original ImageView.
   *
   * If given image is already on disk, a copy will be created in spite of all.
   *
   * @param image ImageView to load on disk
   * @throw InvalidArgument If the image is null
   * @throw InvalidArgument If properties of input ImageView are not valid
   * @return ImageView on disk
   */
  static std::shared_ptr<ImageView> copyOnDisk(std::shared_ptr<ImageView> image);

  /**
   * Load an image data into a temporary raw file on disk.
   *
   * The file will be removed from disk when the instance of the
   * returned ImageView is deleted.
   *
   * Samples are copied chunk by chunk to optimize memory consumption.
   * Maximum chunk size is 1MB.
   *
   * No memory is allocated by this operation, only disk space is used.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   * - WRITE
   * - RESHAPE
   *
   * Original ImageView data are duplicated. Data from the returned ImageView are not linked to
   * original ImageView.
   *
   * If given image is already on disk, a copy will be created in spite of all.
   *
   * @param image ImageView to load on disk
   * @param dirPath path where temporary file is created (must exist)
   * @throw InvalidArgument If the image is null
   * @throw InvalidArgument if given directory path does not exist
   * @throw InvalidArgument If properties of input ImageView are not valid
   * @return ImageView on disk
   */
  static std::shared_ptr<ImageView> copyOnDisk(std::shared_ptr<ImageView> image, const std::string& dirPath);

  /**
   * Creates an image from existing buffer.
   *
   * No memory is allocated by this operation.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   * - WRITE
   * - BUFFER
   *
   * @param shape Shape of image
   * @param type Type of image
   * @param buffer User buffer. Should have at least a size of shape[0]*shape[1]*...*dataType.byteCount()
   * buffer is not handled by this view and the user should carefully manage buffer and view lifetime.
   * @param bufferSize Size of the buffer in bytes.
   * @param properties Properties of image. If null, image will build a default set of properties.
   * @param metadata Metadata of image.
   *
   * @throw InvalidArgument
   * @throw Error If given properties are not compatible with given shape and type
   */
  static std::shared_ptr<ImageView> fromBuffer(const VectorXu64& shape,
                                               DataType type,
                                               void* buffer,
                                               size_t bufferSize,
                                               std::shared_ptr<ImageProperties> properties,
                                               const std::shared_ptr<MetadataNode> metadata);

  /**
   * Creates an image from existing buffer.
   *
   * No memory is allocated by this operation.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   * - WRITE
   * - BUFFER
   *
   * @param shape Shape of image
   * @param type Type of image
   * @param buffer User buffer. Should have at least a size of shape[0]*shape[1]*...*dataType.byteCount()
   * buffer is not handled by this view and the user should carefully manage buffer and view lifetime.
   * @param bufferSize Size of the buffer in bytes.
   */
  static std::shared_ptr<ImageView> fromBuffer(const VectorXu64& shape, DataType type, void* buffer, size_t bufferSize);

  /**
   * Create an image from a RandomAccess.
   *
   * No memory is allocated by this operation.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   * - WRITE
   *
   * @param accessor The RandomAccess from which reading data
   * @param layout The memory layout to use
   * @param shape The shape of the image to create
   * @param type The data type of the image
   * @param properties The properties to attach to the image. If null, default properties are generated.
   * @param metadata A set of metadata to attach to the image
   * @throw InvalidArgument if accessor is null
   * @throw InvalidArgument if accessor size is not compatible with given shape and type
   * @throw Error If given properties are not compatible with given shape and type
   */
  static std::shared_ptr<ImageView> fromRandomAccess(std::shared_ptr<RandomAccess> accessor,
                                                     MemoryLayout layout,
                                                     const VectorXu64& shape,
                                                     const DataType type,
                                                     std::shared_ptr<ImageProperties> properties,
                                                     const std::shared_ptr<const MetadataNode> metadata);

  /**
   * Create an image from a RandomAccess.
   *
   * No memory is allocated by this operation.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   * - WRITE
   *
   * @param accessor The RandomAccess from which reading data
   * @param layout The memory layout to use
   * @param shape The shape of the image to create
   * @param type The data type of the image
   * @throw InvalidArgument if accessor is null
   */
  static std::shared_ptr<ImageView> fromRandomAccess(std::shared_ptr<RandomAccess> accessor,
                                                     MemoryLayout layout,
                                                     const VectorXu64& shape,
                                                     const DataType type);

  /**
   * Create an image from a given RandomAccess.
   *
   * No memory is allocated by this operation.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   * - WRITE
   *
   * Here the Memory layout is defaulted to Fortan.
   *
   * @param accessor The RandomAccess from which reading data
   * @param shape The shape of the image to create
   * @param type The data type of the image
   * @param properties The properties to attach to the image. If null, default properties are generated.
   * @param metadata A set of metadata to attach to the image
   *
   * @throw InvalidArgument if accessor is null
   * @throw Error if the storage cannot contain a raw image with given shape and datatype
   * @throw Error if given properties are not compatible with given shape and type
   */
  static std::shared_ptr<ImageView> fromRandomAccess(std::shared_ptr<RandomAccess> accessor,
                                                     const VectorXu64& shape,
                                                     const DataType type,
                                                     std::shared_ptr<ImageProperties> properties,
                                                     std::shared_ptr<const MetadataNode> metadata);

  /**
   * Create an image from a given RandomAccess.
   *
   * No memory is allocated by this operation.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   * - WRITE
   *
   * Here the Memory layout is defaulted to Fortan.
   *
   * @param accessor The RandomAccess from which reading data
   * @param shape The shape of the image to create
   * @param type The data type of the image
   *
   * @throw InvalidArgument if accessor is null
   * @throw Error if the storage cannot contain a raw image with given shape and datatype
   */
  static std::shared_ptr<ImageView> fromRandomAccess(std::shared_ptr<RandomAccess> accessor,
                                                     const VectorXu64& shape,
                                                     const DataType type);

  /**
   * Allows to apply specific properties and metadata to a ImageView without impact on originals.
   *
   * No memory is allocated by this operation.
   *
   * Returned ImageView has the same capabilities than original ImageView.
   *
   * @param view Input view
   * @param properties The new properties used by the returned image view.
   *                   If null, the returned image view uses a clone of the input view properties.
   * @param metadata The new metadata used by the returned image view (can be null).
   * @throw InvalidArgument If the view is null
   * @throw Error if given properties are not compatible with given shape and type
   */
  static std::shared_ptr<ImageView> fromImageView(std::shared_ptr<ImageView> view,
                                                  std::shared_ptr<const ImageProperties> properties,
                                                  std::shared_ptr<const MetadataNode> metadata);

  /**
   * Create an image with uniform value.
   *
   * No memory is allocated by this operation.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   *
   * This image doesn't consume memory, so you can create an image as big as you want.
   *
   * @param shape The shape of the image (its dimension and sizes in all dimensions)
   * @param type The type of data contained in that view.
   * @param value Buffer should correspond to given DataType. If null is given, image is filled with zeros.
   * @param properties Optional properties. If null, default properties are generated.
   * @param metadata Optional metadata. If null, no metadata are generated.
   *
   * @throw InvalidArgument If given properties are not compatible with given shape and type
   */
  static std::shared_ptr<ImageView> uniform(const VectorXu64& shape,
                                            DataType type,
                                            void* value,
                                            std::shared_ptr<ImageProperties> properties,
                                            std::shared_ptr<const MetadataNode> metadata);

  /**
   * Create an image with uniform value.
   *
   * No memory is allocated by this operation.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   *
   * This image doesn't consume memory, so you can create an image as big as you want.
   *
   * @param shape The shape of the image (its dimension and sizes in all dimensions)
   * @param type The type of data contained in that view.
   * @param value Buffer should correspond to given DataType. If null is given, image is filled with zeros.
   */
  static std::shared_ptr<ImageView> uniform(const VectorXu64& shape, DataType type, void* value);

  // ==================== //
  // Adapting Image Views //
  // ==================== //

  /**
   * Creates an ImageView from another view with given type by mapping dynamic range.
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will depend on original ImageView, and will be restricted to (see
   * #ImageCapability):
   * - READ
   * - WRITE
   *
   * This method will map [inputRange[0], inputRange[1]] -> [outputRange[0], outputRange[1]]
   *
   * One or many adapters will be added if necessary to convert from input view type to output View type.
   * Data will be mapped to full output type range. For example, if given view is 16 bits with valueRange
   * [1000, 2000] and type is uint8 with input range [20, 50], input range [1000, 2000] will be mapped to
   * output range [20, 50].
   *
   * If input image contains value outside of given inputRange, these values will be linearly mapped outside of
   * outputRange. For example, with [0, 1]->[0, 2], a value of -1 will be mapped to -2.
   *
   * @warning No clamp are done for values outside of type admissible values. Invalid ranges may lead to overflow
   * issues.
   *
   * @param view Input view
   * @param type Requested data type
   * @param inputRange Range of data in input view
   * @param outputRange Requested output range
   * @throw InvalidArgument If the view is null
   * @throw InvalidArgument If properties of input ImageView are not valid
   * @throw InvalidArgument If type cardinality is different of original type
   * @throw InvalidArgument If given type is not numeric
   * @throw Error if Input view does not support READ or WRITE capability
   */
  static std::shared_ptr<ImageView> adaptDynamicRange(std::shared_ptr<ImageView> view,
                                                      DataType type,
                                                      const Vector2d& inputRange,
                                                      const Vector2d& outputRange);

  /**
   * Creates an ImageView from another view with given type.
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will depend on original ImageView, and will be restricted to (see
   * #ImageCapability):
   * - READ
   * - WRITE
   *
   * One or many adapters will be added if necessary to convert from input view type to output View type.
   * Data will be mapped to full output type range. For example, if given view is 16 bits with valueRange
   * [1000, 2000] and type is uint8, input range [1000, 2000] will be mapped to output range [0, 255].
   *
   * Input data range are read from view properties. Output range is specified type standard range.
   * It can be retrieved with DataType::standardRange(type).
   *
   * @param view The source image
   * @param type The target datatype
   * @throw InvalidArgument If the view is null
   * @throw InvalidArgument If properties of input ImageView are not valid
   * @throw InvalidArgument If type cardinality is different of original type
   * @throw InvalidArgument If given type is not numeric
   * @throw Error if Input view does not support READ or WRITE capability
   */
  static std::shared_ptr<ImageView> adaptDataType(std::shared_ptr<ImageView> view, DataType type);

  /**
   * Returns an ImageView from another view containing only given channel Index (useful when channel is not clearly
   * identified).
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will depend on original ImageView, and will be restricted to (see
   * #ImageCapability):
   * - READ
   * - WRITE
   *
   * @param view Original view on which we want to isolate given channel
   * @param idxChannel Channel index between 0 and dataType dimension - 1
   *
   * @throw InvalidArgument If the view is null
   * @throw InvalidArgument If requested channel index does not exist
   * @throw InvalidArgument If properties of input ImageView are not valid
   * @throw Error If input image does not support READ or WRITE capability
   */
  static std::shared_ptr<ImageView> extractChannel(std::shared_ptr<ImageView> view, size_t idxChannel);

  /**
   * Creates an ImageView of given region from given view.
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will depend on original ImageView, and will be restricted to (see
   * #ImageCapability):
   * - READ
   * - WRITE
   *
   * @param view Original view on which we want to create a sub-region view
   * @param region Region of the original view that we want to isolate to create a new view
   *
   * @throw InvalidArgument If the view is null
   * @throw Error If input ImageView has nor READ or WRITE capability
   * @throw InvalidArgument If properties of input ImageView are not valid
   * @throw InvalidArgument if required region is empty
   */
  static std::shared_ptr<ImageView> extractRegion(std::shared_ptr<ImageView> view, const RegionXu64& region);

  /**
   * Creates an ImageView of given region from given view, packing flat dimensions.
   *
   * Returned ImageView can potentially have a dimension count lesser than original one. It will
   * depend on the required region. If one (or more) dimension of this region has its size equal to 1,
   * this dimension will be skipped in returned ImageView.
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will depend on original ImageView, and will be restricted to (see
   * #ImageCapability):
   * - READ
   * - WRITE
   *
   * @param view Original view on which we want to create a sub-region view
   * @param region Region of the original view that we want to isolate to create a new view
   *
   * @throw InvalidArgument If the view is null
   * @throw Error If input ImageView has nor READ or WRITE capability
   * @throw InvalidArgument If properties of input ImageView are not valid
   * @throw InvalidArgument If required region is empty
   */
  static std::shared_ptr<ImageView> extractAdjustedRegion(std::shared_ptr<ImageView> view, const RegionXu64& region);

  /**
   * Creates an ImageView from a multiImageView by agglomerating frames into a new given dimension.
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will depend on original ImageViews contained in MultiImageView,
   * and will be restricted to (see #ImageCapability):
   * - READ
   * - WRITE
   *
   * (i.e. a given multiImageView containing 2D frames (with Imagetype = IMAGE) and a given dimension
   * equal to SEQUENCE will give as result a IMAGE_SEQUENCE - video - containing all the images)
   * Concatenation of frames is possible only if their shapes, datatypes and ImageTypes are equal.
   *
   * A warning will be raised if given frames have an ImageType = UNKNOWN, because the given ImageDimension
   * is ignored in this case, but the stack is still applied, and output ImageType will be UNKNOWN
   *
   * If the given dimension is CHANNEL, and more than one frame is stacked, ImageInterpretation will become UNKNOWN.
   *
   * Underlying adapter use multithreading to accelerate the data access. To configure or deactivate it, use prototype
   * with `threadCount` as argument.
   *
   * @param view MultiImage view which contains frames of same format and shape
   * @param newDimension interpretation of the new dimension to create
   *
   * @throw InvalidArgument If the view is null
   * @throw InvalidArgument if all frames don't have same shape and datatype
   * @throw InvalidArgument if given new dimension already exist in given frames
   * @throw InvalidArgument if all frames don't have the same interpretation of axes
   */
  static std::shared_ptr<ImageView> stack(std::shared_ptr<MultiImageView> view, ImageDimension newDimension);

  /**
   * Creates an ImageView from a multiImageView by agglomerating frames into a new unidentified dimension.
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will depend on original ImageViews contained in MultiImageView,
   * and will be restricted to (see #ImageCapability):
   * - READ
   * - WRITE
   *
   * (i.e. a given multiImageView containing 2D ImageViews will give as result 3D ImageView - containing all the images)
   * Concatenation of frames is possible only if their shapes and datatypes are equal.
   * New dimension is always created in **last position**.
   * e.g.: 3 frames with shapes (5, 6) will be stacked into an 3D Image with shape (5, 6, 3)
   *
   * This method can be used with frames whose ImageType is not UNKNOWN, but as the new dimension is not given,
   * the ImageType will be lost (UNKNOWN) in output ImageView.
   *
   * Underlying adapter use multithreading to accelerate the data access. To configure or deactivate it, use prototype
   * with `threadCount` as argument.
   *
   * @param view MultiImage view which contains frames of same format and shape
   *
   * @throw InvalidArgument If the view is null
   * @throw InvalidArgument if all frames don't have same shape and datatype
   */
  static std::shared_ptr<ImageView> stack(std::shared_ptr<MultiImageView> view);

  /**
   * Creates an ImageView from a multiImageView by agglomerating frames into a new given dimension.
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will depend on original ImageViews contained in MultiImageView,
   * and will be restricted to (see #ImageCapability):
   * - READ
   * - WRITE
   *
   * (i.e. a given multiImageView containing 2D frames (with Imagetype = IMAGE) and a given dimension
   * equal to SEQUENCE will give as result a IMAGE_SEQUENCE - video - containing all the images)
   * Concatenation of frames is possible only if their shapes, datatypes and ImageTypes are equal.
   *
   * A warning will be raised if given frames have an ImageType = UNKNOWN, because the given ImageDimension
   * is ignored in this case, but the stack is still applied, and output ImageType will be UNKNOWN
   *
   * If the given dimension is CHANNEL, and more than one frame is stacked, ImageInterpretation will become UNKNOWN.
   *
   * Underlying adapter can use multithreading to accelerate the data access. By default (0), the thread count is
   * automatically determined, but you can also force the thread count to use (value greater than 1).
   * Setting a thread count to 1 will deactivate multithreading.
   *
   * @param view MultiImage view which contains frames of same format and shape
   * @param newDimension interpretation of the new dimension to create
   * @param threadCount Number of threads to use for DataAccess
   *
   * @throw InvalidArgument If the view is null
   * @throw InvalidArgument if all frames don't have same shape and datatype
   * @throw InvalidArgument if given new dimension already exist in given frames
   * @throw InvalidArgument if all frames don't have the same interpretation of axes
   */
  static std::shared_ptr<ImageView> stack(std::shared_ptr<MultiImageView> view,
                                          ImageDimension newDimension,
                                          size_t threadCount);

  /**
   * Adapt a N dimensions scalar ImageView into a N-1 dimensions vectorial one.
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will be restricted to (see #ImageCapability):
   * - READ
   *
   * It packs the values of CHANNEL dimension (or the last dimension if does not exist)
   * into one sample vector. The dimension of this vector will be the size of the last
   * dimension of the source ImageView.
   *
   * @param image The source ImageView
   *
   * @throw InvalidArgument If the image is null
   * @throw InvalidArgument If properties of input ImageView are not valid
   * @throw Error If original ImageView does not support READ capability
   */
  static std::shared_ptr<ImageView> assembleChannels(std::shared_ptr<ImageView> image);

  /**
   * Adapt a N dimensions vectorial ImageView into a N+1 dimensions scalar one.
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will be resctriced to (see #ImageCapability):
   * - READ
   *
   * It unpacks samples to generate a CHANNEL dimension for each component. This new dimension
   * will have the size of the original sample size.
   * (e.g. disassemble a RGBA 2D ImageView will result into a 3D ImageView with CHANNEL dimension size = 4)
   *
   * @param image The source ImageView
   *
   * @throw InvalidArgument If the image is null
   * @throw InvalidArgument If view already contains a CHANNEL dimension
   * @throw InvalidArgument If properties of input ImageView are not valid
   * @throw Error If original ImageView does not support READ capability
   */
  static std::shared_ptr<ImageView> disassembleChannels(std::shared_ptr<ImageView> image);

  /**
   * Interlace frames of a MultiImageView into an ImageView.
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will depend on capabilities of ImageViews contained in the multiImageView,
   * and be resctriced to (see #ImageCapability):
   * - READ
   * - WRITE
   *
   * Frames contained in MultiImageViews are assembled to create an ImageView whose samples are the concatenation
   * of samples from original ImageViews.
   * i.e. 3 grayscale ImageViews with UINT16 datatype contained in a multiImageView
   *   First ImageView contains ABCDEFG..... values
   *   Second ImageView contains abcdefg... values
   *   Third ImageView contains 123456.... values
   *  Interlace operation gives an ImageViews with VEC3UINT16 datatype whose content is:
   *   Aa1 Bb2 Cc3 Dd4 Ee5 Ff6 Gg7.....
   *
   * Final interlaced ImageView will have Write access only if all original ImageViews (in multiImageView) have Write
   * access
   *
   * @param multiView MultiImageView which contains ImageViews to interlace
   * @param newInterpretation interpretation to set for the interlaced ImageView (default is UNKNOWN)
   * @param hasAlpha indicates if the interlaced ImageView contains an Alpha channel (default is FALSE)
   *
   * @throw InvalidArgument If the image is null
   * @throw InvalidArgument If multiImageView contains incompatible ImageViews (different shape or datatypes)
   * @throw InvalidArgument If one frame from multiImageView contains a CHANNEL dimension
   * @throw InvalidArgument If type of at least one frame is not scalar
   *
   */
  static std::shared_ptr<ImageView> interlace(std::shared_ptr<MultiImageView> multiView,
                                              ImageInterpretation newInterpretation,
                                              bool hasAlpha);

  /**
   * Interlace frames of a MultiImageView into an ImageView.
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will depend on capabilities of ImageViews contained in the multiImageView,
   * and be resctriced to (see #ImageCapability):
   * - READ
   * - WRITE
   *
   * Frames contained in MultiImageViews are assembled to create an ImageView whose samples are the concatenation
   * of samples from original ImageViews.
   * i.e. 3 grayscale ImageViews with UINT16 datatype contained in a multiImageView
   *  First ImageView contains ABCDEFG..... values
   *  Second ImageView contains abcdefg... values
   *  Third ImageView contains 123456.... values
   *  Interlace operation gives an ImageViews with VEC3UINT16 datatype whose content is:
   *   Aa1 Bb2 Cc3 Dd4 Ee5 Ff6 Gg7.....
   *
   * Final interlaced ImageView will have Write access only if all original ImageViews (in multiImageView) have Write
   * access
   *
   * @param multiView MultiImageView which contains ImageViews to interlace
   *
   * @throw InvalidArgument If the image is null
   * @throw InvalidArgument If multiImageView contains incompatible ImageViews (different shape or datatypes)
   * @throw InvalidArgument If one frame from multiImageView contains a CHANNEL dimension
   * @throw InvalidArgument If type of at least one frame is not scalar
   *
   */
  static std::shared_ptr<ImageView> interlace(std::shared_ptr<MultiImageView> multiView);

  /**
   * Reinterpret an ImageView with a new ImageTypeId.
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will depend on capabilities of input ImageView and be resctriced to (see
   * #ImageCapability):
   * - READ
   * - WRITE
   *
   * It allows to modify the interpretation (VOLUME, IMAGE SEQUENCE, ...) of an any-dimension image.
   * It does not change samples data reading, only image properties.
   * When new interpretation is given, a quick check is done to verify compatibility with given ImageView. (number of
   * dimensions should match with new interpretation)
   *
   * Specific case: ImageType::UNKNOWN can be set for any given ImageView without error
   *
   * @throw InvalidArgument If the image is null
   * @throw Error If ImageType is incompatible with ImageView dimension
   * @param image ImageView whose interpretation must be changed
   * @param newImageType Image interpretation to apply to image
   * @return a ImageView with new properties to match new image interpretation
   * @throw InvalidArgument If the image is null
   * @throw InvalidArgument If properties of input ImageView are not valid
   * @throw Error If new imageType is not comparible with given ImageView
   */
  static std::shared_ptr<ImageView> reinterpret(std::shared_ptr<ImageView> image, ImageType newImageType);

  /**
   * Reinterpret an ImageView by identifying each dimension (axis).
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will depend on capabilities of input ImageView and be resctriced to (see
   * #ImageCapability):
   * - READ
   * - WRITE
   *
   * It allows to modify the interpretation (VOLUME, IMAGE SEQUENCE, ...) of an any-dimension image by directly
   * affecting an interpretation to each dimension.
   *
   * It can impact the way samples are read/written, because, according to given ImageDimension list, dimensions will be
   * reordered to match IOLink canonical order (see ImageTypeId)
   *
   * For example, an 3D ImageView with following list is given ([ROW](\ref #ImageDimension.ROW),
   * [SLICE](\ref #ImageDimension.SLICE), [COLUMN](\ref #ImageDimension.COLUMN)). Dimensions will be relocated to match
   * canonical order (COLUMN, ROW, SLICE).
   *
   * If canonical order is already respected in given list, this will only affect the properties of the ImageView.
   * A quick check is done to verify compatibility with given ImageView. (number of dimensions should match with the
   * given list size). An exception is raised in case of incompatibility.
   *
   * Warning: never insert a dimension more than once in given list, otherwise the behavior is unpredictable.
   *
   * This method can be useful for ImageViews based on user memory data, with different dimensions order, and without
   * original interpretation.
   *
   * @param image ImageView to reinterpret
   * @param dimensionsList ordered list of dimensions to affect to ImageView
   * @return an ImageView where dimensions are potentially reordered to match canonical order and with a new ImageTypeId
   *
   * @throw InvalidArgument If the image is null
   * @throw Error If dimensionsList size does not match with image dimension.
   * @throw InvalidArgument If properties of input ImageView are not valid
   */
  static std::shared_ptr<ImageView> reinterpretAxes(std::shared_ptr<ImageView> image,
                                                    std::initializer_list<ImageDimension> dimensionsList);

  /**
   * Reinterpret an ImageView by identifying each dimension (axis).
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will depend on capabilities of input ImageView and be resctriced to (see
   * #ImageCapability):
   * - READ
   * - WRITE
   *
   * @param image ImageView to reinterpret
   * @param dimensionsList Ordered list of dimensions to affect to ImageView
   * @param count Number of elements in the given list
   * @return a ImageView where dimensions are potentially reordered to match canonical order and with a new ImageTypeId
   *
   * @throw InvalidArgument If the image is null
   * @throw Error If dimensionsList size does not match with image dimension.
   * @throw InvalidArgument If properties of input ImageView are not valid
   */
  static std::shared_ptr<ImageView> reinterpretAxes(std::shared_ptr<ImageView> image,
                                                    const ImageDimension* dimensionsList,
                                                    size_t count);

  /**
   * Create a sub-sampled imageView from original.
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will be resctriced to (see #ImageCapability):
   * - READ
   *
   * @param image ImageView to sub-sample
   * @param step level of sub-sampling to apply
   *
   * New ImageView is created by picking one sample for every `step` samples of original ImageView.
   * A step of 1 will give the same Image as the original
   *
   * @return a Sub-Sampled ImageView from original ImageView
   *
   * @throw InvalidArgument If the image is null
   * @throw InvalidArgument If properties of input ImageView are not valid
   * @throw InvalidArgument If step is 0 or not compatible with ImageView shape
   * @throw Error If input image does not support READ capability
   */
  static std::shared_ptr<ImageView> subSample(std::shared_ptr<ImageView> image, size_t step);

  /**
   * Flip an ImageView following the given dimension.
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will be resctriced to (see #ImageCapability):
   * - READ
   *
   * @param image ImageView to flip
   * @param dimension ImageDimension used for flipping
   * @return a ImageView flipped following given dimension
   *
   * @throw InvalidArgument If the image is null
   * @throw Error If input image does not support READ capability
   * @throw InvalidArgument If image type is not known
   * @throw InvalidArgument If properties of input ImageView are not valid
   * @throw InvalidArgument If image does not contain given dimension
   */
  static std::shared_ptr<ImageView> flip(std::shared_ptr<ImageView> image, ImageDimension dimension);

  /**
   * Flip an ImageView following the given dimension.
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will be resctriced to (see #ImageCapability):
   * - READ
   *
   * @param image ImageView to flip
   * @param dimensionIndex Dimension index used for flipping
   * @return a ImageView flipped following given dimension
   * @throw InvalidArgument If the image is null
   * @throw Error If input image does not support READ capability
   * @throw InvalidArgument If properties of input ImageView are not valid
   * @throw InvalidArgument if dimension count of given image is lesser than given dimension index
   */

  static std::shared_ptr<ImageView> flip(std::shared_ptr<ImageView> image, size_t dimensionIndex);

  /**
   * Make an ImageView threadsafe.
   *
   * No memory is allocated by this operation.
   *
   * Capabilities of returned ImageView will be the same than original image's.
   *
   * @param image ImageView to make threadsafe
   * @return an ImageView protected for multi-thread accesses.
   * @throw InvalidArgument If the image is null
   */
  static std::shared_ptr<ImageView> makeThreadSafe(std::shared_ptr<ImageView> image);
};

} // end namespace iolink
