// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <cstdint>
#include <memory>

#include <iolink/FlagSet.h>
#include <iolink/IOLinkAPI.h>
#include <iolink/metadata/MetadataNode.h>
#include <iolink/view/ImageView.h>
#include <iolink/view/View.h>

namespace iolink
{

/**
 * Define capabilities of an MultiImageView.
 */
enum class MultiImageCapability
{
  // Flags
  READ = 0x1,  ///< frames access
  WRITE = 0x2, ///< frames addition and modification

  // Combinations
  READ_WRITE = 0x3, ///< Combines READ and WRITE
};

extern template class IOLINK_API_IMPORT FlagSet<MultiImageCapability>;

/**
 * FlagSet to handle MultiImageView capabilities
 */
using MultiImageCapabilitySet = FlagSet<MultiImageCapability>;

// Define bitwise operators for MultiImageCapability
IOLINK_DEFINE_ENUM_BITWISE_OPERATORS(MultiImageCapability)

/**
 * Interface representing a multi-image view, i.e. a stack of ImageView
 *
 * This interface capabilities are indicated by the MultiImageCapabilitySet
 * returned by the method MultiImageView::capabilities. Each capability enable
 * the use of a specific set of methods. When one of those methods is
 * called, if its corresponding capability is not supported, a NotImplemented
 * exception will be thrown.
 *
 * @see MultiImageCapability
 */
class IOLINK_API MultiImageView : public View
{
public:
  virtual ~MultiImageView() = default;

  /**
   * Return the capabilities of this multi image.
   */
  virtual MultiImageCapabilitySet capabilities() const = 0;

  /**
   * Checks if the MultiImageView supports the given capabilities.
   */
  inline bool support(MultiImageCapabilitySet flags) { return this->capabilities().has(flags); }

  /**
   * Return a string representation
   */
  std::string toString() const;

  // ==== //
  // READ //
  // ==== //

  /**
   * Returns the number of frames in the set
   *
   * @return the number of frames in the set
   *
   * @throw Error
   * @throw AccessError
   */
  virtual size_t frameCount() const;

  /**
   * Gets the ith frame of the set
   *
   * @param index the index of the frame in the set
   * @return the ith frame as ImageView
   */
  virtual std::shared_ptr<ImageView> frame(size_t index);

  /**
   * Returns the root node of the metadata attached to this multi-image.
   */
  virtual std::shared_ptr<const MetadataNode> metadata() const;

  // ===== //
  // WRITE //
  // ===== //

  /**
   * Add a new frame at the end of the set
   */
  virtual void addFrame(std::shared_ptr<ImageView> newFrame);

  /**
   * Set the frame at the given index (index must already exist, no allocation)
   */
  virtual void setFrame(size_t index, std::shared_ptr<ImageView> frame);

  /**
   * Remove the ith frame
   */
  virtual void removeFrame(size_t index);

  /**
   * Attach a new metadata node to the image.
   */
  virtual void setMetadata(std::shared_ptr<MetadataNode> metadata);
};

} // namespace iolink
