// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <memory>
#include <string>

#include <iolink/FlagSet.h>
#include <iolink/view/Extension.h>

namespace iolink
{

/**
 * Define multi-access capabilities of an ImageView.
 */
enum class ParallelAccessCapability
{
  /**
   * Image can be multi-accessed (READ and/or WRITE) without any race condition,
   * but with no control on thread execution order. Are concerned the following methods:
   * - `read`
   * - `readRegion`
   * - `write`
   * - `writeRegion`
   * - `readTile`
   * - `writeTile`
   */
  THREAD_SAFE = 0x1,

  /**
   * Hints that reading tiles in parallel will provide performance benefits.
   * But remember that parallel calls on the same tile will not provide benefits,
   * and can cause race conditions if there is no `THREAD_SAFE` capability.
   * Is concerned the following method:
   * - `readTile`
   */
  PARALLEL_TILE_READ = 0x2,

  /**
   * Hints that writing tiles in parallel will provide performance benefits,
   * But remember that parallel calls on the same tile will not provide benefits,
   * and can cause race conditions if there is no `THREAD_SAFE` capability. Is concerned
   * the following method:
   * - `writeTile`
   */
  PARALLEL_TILE_WRITE = 0x4,

  /**
   * Hints that regions of the image can be read in parallel with performance benefits.
   * But parallel access to overlapping regions in parallel will not provide the optimum
   * performance, and even race conditions if there is no `THREAD_SAFE` capability.
   * Is concerned the following method:
   * - `readRegion`
   */
  PARALLEL_READ = 0x8,

  /**
   * Hints that regions of the image can be written in parallel with performance benefits.
   * But parallel access to overlapping regions in parallel will not provide the optimum
   * performance, and even race conditions if there is no `THREAD_SAFE` capability.
   * Is concerned the following method:
   * - `writeRegion`
   */
  PARALLEL_WRITE = 0x10
};

extern template class IOLINK_API_IMPORT FlagSet<ParallelAccessCapability>;

/**
 * FlagSet to handle ImageView parallel Access capabilities
 */
using ParallelAccessCapabilitySet = FlagSet<ParallelAccessCapability>;

// Define bitwise operators for ParallelAccessCapability
IOLINK_DEFINE_ENUM_BITWISE_OPERATORS(ParallelAccessCapability)

/**
 * ImageView extension to give information about how to efficiently
 * access it in a multi-threaded environment.
 *
 * ImageView content can be accessed with following features:
 * - thread-safe: no race conditions when accessing with multiple threads
 * - parallel-access by tile: hints benefits when tiles are accessed in parallel
 * - parallel-access: hints benefits when regions are accessed in parallel
 *
 * In any case, parallel reading and writing into the same part of an image is not advised,
 * as there is no control over the ordering of operations.
 */
class IOLINK_API ParallelAccessExtension : public Extension
{
public:
  /**
   * Identifier used to load the extension using the View::extension method.
   */
  static constexpr size_t EXTENSION_ID = 0x2;

  /**
   * @brief Upcast a generic extension to this specific type.
   *
   * @param extension The extension to cast.
   * @return The upcasted extension. This will be null if no upcasting is possible.
   */
  inline static std::shared_ptr<ParallelAccessExtension> cast(const std::shared_ptr<Extension>& extension)
  {
    return std::dynamic_pointer_cast<ParallelAccessExtension>(extension);
  }

  /**
   * Return the parallel-access capabilities of this view.
   */
  virtual ParallelAccessCapabilitySet capabilities() const = 0;

  /**
   * Checks if the view supports the given capabilities.
   */
  inline bool support(ParallelAccessCapabilitySet flags) { return this->capabilities().has(flags); }

  /**
   * String representation of the extension.
   */
  std::string toString() const;
};

} // end namespace iolink
