#include <ImageDev/ImageDev.h>
#include <ioformat/IOFormat.h>
#include <string.h>

using namespace imagedev;

int
main( int argc, char* argv[] )
{
    int status = 0;

    try
    {
        // ImageDev library initialization
        if ( imagedev::isInitialized() == false )
            imagedev::init();

        // Open a tif file to denoise
        auto imageInput = ioformat::readImage( std::string( IMAGEDEVDATA_IMAGES_FOLDER ) + "mnu.tif" );

        // Apply an average filter
        std::cout << "Applying a box filter... " << std::endl;
        auto imageOutput = boxFilter2d( imageInput, 7, 7, BoxFilter2d::AutoScale::YES );
        ioformat::writeView( imageOutput, "T03_01_box.png" );

        // Apply a median filter processing
        std::cout << "Applying a median filter... " << std::endl;
        imageOutput =
            medianFilter2d( imageInput, 3, MedianFilter2d::KernelMode::SQUARE, MedianFilter2d::SearchMode::AUTOMATIC );
        ioformat::writeView( imageOutput, "T03_01_median.png" );

        // Apply a non-local means filter processing
        std::cout << "Applying a non-local means filter... " << std::endl;
        imageOutput = nonLocalMeansFilter2d( imageInput, 3, 3, 0.6, NonLocalMeansFilter2d::SQUARE );
        ioformat::writeView( imageOutput, "T03_01_nlm.png" );

        std::cout << "This example ran successfully." << std::endl;
    }
    catch ( const imagedev::Exception& error )
    {
        // Print potential exception in the standard output
        std::cerr << "ImageDev exception: " << error.what() << std::endl;
        status = -1;
    }

    // ImageDev library finalization
    imagedev::finish();

    // Check if we must ask for an enter key to close the program
    if ( !( ( argc == 2 ) && strcmp( argv[1], "--no-stop-at-end" ) == 0 ) )
        std::cout << "Press Enter key to close this window." << std::endl, getchar();

    return status;
}
