# This configuration file enables to find the ImageDevExamples package.
# If the detection of ImageDev failed, the following variables are defined:
#   - ImageDevExamples_FOUND                    -> False
#   - ImageDevExamples_INCLUDE_DIR              -> ImageDevExamples_INCLUDE_DIR-NOTFOUND
#   - ImageDevExamplesData_HOME                 -> ImageDevExamplesData_HOME-NOTFOUND
#   - ImageDevExamplesData_IMAGES_DIR           -> ImageDevExamplesData_IMAGES_DIR-NOTFOUND
#   - ImageDevExamplesData_OBJECTS_DIR          -> ImageDevExamplesData_OBJECTS_DIR-NOTFOUND
#   - ImageDevExamples_BINS_DIRECTORY_DEBUG     -> ImageDevExamples_BINS_DIRECTORY_DEBUG-NOTFOUND
#   - ImageDevExamples_BINS_DIRECTORY_RELEASE   -> ImageDevExamples_BINS_DIRECTORY_RELEASE-NOTFOUND
# Else, the following variables are defined:
#   - ImageDevExamples_FOUND                    -> True
#   - ImageDevExamples_INCLUDE_DIR              -> Path of the directory containing the IOFormat header files.
#   - ImageDevExamplesData_HOME                 -> Path of the directory containing the ImageDevExamples data.
#   - ImageDevExamplesData_IMAGES_DIR           -> Path of the directory containing the ImageDevExamples images.
#   - ImageDevExamplesData_OBJECTS_DIR          -> Path of the directory containing the ImageDevExamples objects.
#   - ImageDevExamples_BINS_DIRECTORY_DEBUG     -> Path of the directory containing the debug IOFormat libraries that need to be installed next to the application.
#   - ImageDevExamples_BINS_DIRECTORY_RELEASE   -> Path of the directory containing the release IOFormat libraries that need to be installed next to the application.
#
# To link your project with ImageDevExamples, the following line is sufficient:
#   target_link_libraries( ${PROJECT_NAME} PUBLIC ImageDevExamples )
# This adds
#   - ImageDevExamples_INCLUDE_DIR in the list of include directiories of your project.
#   - A link to the IOFormat library.
#   - The definition of the following preprocessor flags
#       + IMAGEDEVDATA_REPOSITORY="${ImageDevExamplesData_HOME}"
#       + IMAGEDEVDATA_IMAGES_FOLDER="${ImageDevExamplesData_IMAGES_DIR}"
#       + IMAGEDEVDATA_OBJECTS_FOLDER="${ImageDevExamplesData_OBJECTS_DIR}"
#
# To install all libraries used by ImageDevExamples, we must add the following lines:
#   install( DIRECTORY ${ImageDevExamples_BINS_DIRECTORY_DEBUG}   DESTINATION "your/output/debug/directory/path"   CONFIGURATIONS Debug )
#   install( DIRECTORY ${ImageDevExamples_BINS_DIRECTORY_RELEASE} DESTINATION "your/output/release/directory/path" CONFIGURATIONS Release )

# ---------------------------------------------------------------------------
# This macro exits of this file and writes the error message "error_message".
macro( exit_with_error error_message )
  if( ImageDevExamples_FIND_REQUIRED )
    message( FATAL_ERROR ${error_message} )
  elseif( NOT ImageDevExamples_FIND_QUIETLY )
    message( WARNING ${error_message} )
  endif()
  return()
endmacro()

# ---------------------------------------------------------------------------
# Initializes the variables of the ImageDev package
set( ImageDevExamples_FOUND False )
set( ImageDevExamples_INCLUDE_DIR "ImageDevExamples_INCLUDE_DIR-NOTFOUND" )
set( ImageDevExamplesData_HOME "ImageDevExamplesData_HOME-NOTFOUND" )
set( ImageDevExamplesData_IMAGES_DIR "ImageDevExamplesData_IMAGES_DIR-NOTFOUND" )
set( ImageDevExamplesData_OBJECTS_DIR "ImageDevExamplesData_OBJECTS_DIR-NOTFOUND" )
set( ImageDevExamples_BINS_DIRECTORY_DEBUG "ImageDevExamples_BINS_DIRECTORY_DEBUG-NOTFOUND" )
set( ImageDevExamples_BINS_DIRECTORY_RELEASE "ImageDevExamples_BINS_DIRECTORY_RELEASE-NOTFOUND" )

# ---------------------------------------------------------------------------
# Checks the version of CMake
if( CMAKE_VERSION VERSION_LESS 3.7 )
  exit_with_error( "${CMAKE_FIND_PACKAGE_NAME} requires at least CMake version 3.7" )
endif()
# Checks that the OS is not APPLE
if( APPLE )
  exit_with_error( "${CMAKE_FIND_PACKAGE_NAME} does not available for APPLE" )
endif()

# ---------------------------------------------------------------------------
# define ImageDevExamples_HOME for convenience
get_filename_component( ImageDevExamples_HOME "${CMAKE_CURRENT_LIST_DIR}/.." ABSOLUTE )

# ---------------------------------------------------------------------------
# This defines necessary variables in order to find
# ImageDevExamples package
if( CMAKE_COMPILER_IS_GNUCXX )
  string( SUBSTRING ${CMAKE_CXX_COMPILER_VERSION} 0 1 major_version )
  string( SUBSTRING ${CMAKE_CXX_COMPILER_VERSION} 2 1 release_version )
  if( CMAKE_CXX_COMPILER_ID STREQUAL "GNU" )
    set( ImageDevExamples_CXX_COMPILER_ID "gcc" )
    if( CMAKE_CXX_COMPILER_VERSION VERSION_GREATER 5.0 )
      set( release_version "" ) # From GCC 5, only major version is significant.
    endif()
  endif()
  set( ImageDevExamples_CXX_COMPILER_VERSION "${major_version}${release_version}" CACHE STRING "GCC version used for prepacked" )
  # List of the supported versions (by binary compatibility).
  # This list will be used if we can't find the libraries directories for the current version of compiler.
  set( ImageDevExamples_CXX_COMPILER_OTHER_SUPPORTED_VERSION 9 7 )
else()
  string( TOLOWER ${CMAKE_CXX_COMPILER_ID} ImageDevExamples_CXX_COMPILER_ID )

  # determine the MSVC version from the Visual Studio version
  # and the list of the supported versions (by binary compatibility).
  # This list will be used if we can't find the libraries directories for the current version of Visual Studio.
  if ( MSVC_VERSION GREATER_EQUAL 1930 )
    set( ImageDevExamples_CXX_COMPILER_VERSION "17" )
    set( ImageDevExamples_CXX_COMPILER_OTHER_SUPPORTED_VERSION 17 16 15 14 )
  elseif ( MSVC_VERSION GREATER_EQUAL 1920 )
    set( ImageDevExamples_CXX_COMPILER_VERSION "16" )
    set( ImageDevExamples_CXX_COMPILER_OTHER_SUPPORTED_VERSION 16 15 14 )
  elseif ( MSVC_VERSION GREATER_EQUAL 1910 )
    set( ImageDevExamples_CXX_COMPILER_VERSION "15" )
    set( ImageDevExamples_CXX_COMPILER_OTHER_SUPPORTED_VERSION 15 14 )
  elseif ( MSVC_VERSION GREATER_EQUAL 1900 )
    set( ImageDevExamples_CXX_COMPILER_VERSION "14" )
    set( ImageDevExamples_CXX_COMPILER_OTHER_SUPPORTED_VERSION 14 )
  endif()
endif()

if( WIN32 )
  set( ImageDevExamples_SYSTEM_PROCESSOR x86_64)
else()
  set( ImageDevExamples_SYSTEM_PROCESSOR ${CMAKE_SYSTEM_PROCESSOR} )
endif()

set( ImageDevExamples_CXX_PLATFORM_ID ${CMAKE_CXX_PLATFORM_ID} )

set( CMAKE_ARCHITECTURE "${ImageDevExamples_CXX_PLATFORM_ID}-${ImageDevExamples_SYSTEM_PROCESSOR}-${ImageDevExamples_CXX_COMPILER_ID}${ImageDevExamples_CXX_COMPILER_VERSION}" )

# these variables are used by find_module_config to avoid rebuilding them for every module
set( ImageDevExamples_ARCH_DEBUG arch-${CMAKE_ARCHITECTURE}-debug )
set( ImageDevExamples_ARCH_RELEASE arch-${CMAKE_ARCHITECTURE}-release )

# If the "arch" directory does not exist, we check if another "arch" compatible directory exists
if( ( WIN32 OR ( CMAKE_BUILD_TYPE MATCHES "[dD]ebug" ) ) AND ( NOT EXISTS "${ImageDevExamples_HOME}/${ImageDevExamples_ARCH_DEBUG}" ) )
  foreach( cur_supported_version IN LISTS ImageDevExamples_CXX_COMPILER_OTHER_SUPPORTED_VERSION )
    set( new_debug_dir_path "arch-${ImageDevExamples_CXX_PLATFORM_ID}-${ImageDevExamples_SYSTEM_PROCESSOR}-${ImageDevExamples_CXX_COMPILER_ID}${cur_supported_version}-debug" )
    if ( EXISTS "${ImageDevExamples_HOME}/${new_debug_dir_path}" )
      set( ImageDevExamples_ARCH_DEBUG "${new_debug_dir_path}" )
      break()
    endif()
  endforeach()
endif()
if( ( WIN32 OR ( CMAKE_BUILD_TYPE MATCHES "[rR]elease" ) ) AND ( NOT EXISTS "${ImageDevExamples_HOME}/${ImageDevExamples_ARCH_RELEASE}" ) )
  foreach( cur_supported_version IN LISTS ImageDevExamples_CXX_COMPILER_OTHER_SUPPORTED_VERSION )
    set( new_release_dir_path "arch-${ImageDevExamples_CXX_PLATFORM_ID}-${ImageDevExamples_SYSTEM_PROCESSOR}-${ImageDevExamples_CXX_COMPILER_ID}${cur_supported_version}-release" )
    if ( EXISTS "${ImageDevExamples_HOME}/${new_release_dir_path}" )
      set( ImageDevExamples_ARCH_RELEASE "${new_release_dir_path}" )
      break()
    endif()
  endforeach()
endif()

# set ImageDevExamples_ARCH for convenience as it may be used in some places
if( CMAKE_BUILD_TYPE MATCHES "[dD]ebug" )
  set( ImageDevExamples_ARCH ${ImageDevExamples_ARCH_DEBUG} )
elseif( CMAKE_BUILD_TYPE MATCHES "[rR]elease" )
  set( ImageDevExamples_ARCH ${ImageDevExamples_ARCH_RELEASE} )
else()
  set( ImageDevExamples_ARCH arch-${CMAKE_ARCHITECTURE}-$<CONFIG> )
endif()


# ---------------------------------------------------------------------------
# Include directory
set( ImageDevExamples_INCLUDE_DIR "${ImageDevExamples_HOME}/include" )
if( NOT EXISTS "${ImageDevExamples_INCLUDE_DIR}/ioformat/IOFormat.h" )
  set( ImageDevExamples_INCLUDE_DIR "ImageDevExamples_INCLUDE_DIR-NOTFOUND" )
  exit_with_error( "failed to find the include directory of IOFormat" )
endif()

# Libraries directories
set( ImageDevExamples_LIBS_DIRECTORY_DEBUG "${ImageDevExamples_HOME}/${ImageDevExamples_ARCH_DEBUG}/lib/" )
set( ImageDevExamples_LIBS_DIRECTORY_RELEASE "${ImageDevExamples_HOME}/${ImageDevExamples_ARCH_RELEASE}/lib/" )

# Libraries path
if ( WIN32 )
  set( IOFormat_LIBS_RELEASE "${ImageDevExamples_LIBS_DIRECTORY_RELEASE}ioformat.lib" )
  set( IOFormat_LIBS_DEBUG "${ImageDevExamples_LIBS_DIRECTORY_DEBUG}ioformatd.lib" )
else()
  set( IOFormat_LIBS_RELEASE "${ImageDevExamples_LIBS_DIRECTORY_RELEASE}libioformat.so" )
  set( IOFormat_LIBS_DEBUG "${ImageDevExamples_LIBS_DIRECTORY_DEBUG}libioformat.so" )
endif()

# Example data directory
set( ImageDevExamplesData_HOME "${ImageDevExamples_HOME}/examples/data/" )
if( NOT EXISTS "${ImageDevExamplesData_HOME}" )
  exit_with_error( "failed to find the example data directory of ImageDevExamples" )
endif()
set( ImageDevExamplesData_IMAGES_DIR "${ImageDevExamplesData_HOME}images/" )
if( NOT EXISTS "${ImageDevExamplesData_IMAGES_DIR}" )
  exit_with_error( "failed to find the images directory of ImageDevExamples" )
endif()
set( ImageDevExamplesData_OBJECTS_DIR "${ImageDevExamplesData_HOME}objects/" )
if( NOT EXISTS "${ImageDevExamplesData_OBJECTS_DIR}" )
  exit_with_error( "failed to find the objects directory of ImageDevExamples" )
endif()

# ---------------------------------------------------------------------------
# Binaries directories
if ( WIN32 )
  set( ImageDevExamples_BINS_DIRECTORY_DEBUG "${ImageDevExamples_HOME}/${ImageDevExamples_ARCH_DEBUG}/bin/" )
  set( ImageDevExamples_BINS_DIRECTORY_RELEASE "${ImageDevExamples_HOME}/${ImageDevExamples_ARCH_RELEASE}/bin/" )
else()
  set( ImageDevExamples_BINS_DIRECTORY_DEBUG "${ImageDevExamples_LIBS_DIRECTORY_DEBUG}" )
  set( ImageDevExamples_BINS_DIRECTORY_RELEASE "${ImageDevExamples_LIBS_DIRECTORY_RELEASE}" )
endif()
# Checks the binaries directories
if( NOT EXISTS "${ImageDevExamples_BINS_DIRECTORY_DEBUG}" )
  set( ImageDevExamples_BINS_DIRECTORY_DEBUG "ImageDevExamples_BINS_DIRECTORY_DEBUG-NOTFOUND" )
  exit_with_error( "failed to find the debug binaries directory of ImageDevExamples" )
endif()
if( NOT EXISTS "${ImageDevExamples_BINS_DIRECTORY_RELEASE}" )
  set( ImageDevExamples_BINS_DIRECTORY_RELEASE "ImageDevExamples_BINS_DIRECTORY_RELEASE-NOTFOUND" )
  exit_with_error( "failed to find the release binaries directory of ImageDevExamples" )
endif()

# Defines ImageDevExamples as an interface
set( ImageDevExamples_INTERFACE_NAME ImageDevExamples )
add_library( ${ImageDevExamples_INTERFACE_NAME} INTERFACE IMPORTED GLOBAL )

set_property( TARGET ${ImageDevExamples_INTERFACE_NAME} PROPERTY INTERFACE_LINK_LIBRARIES ""
                               $<$<CONFIG:Release>:${IOFormat_LIBS_RELEASE}>
                               $<$<CONFIG:RelWithDebInfo>:${IOFormat_LIBS_RELEASE}>
                               $<$<CONFIG:MinSizeRel>:${IOFormat_LIBS_RELEASE}>
                               $<$<CONFIG:Debug>:${IOFormat_LIBS_DEBUG}> )

set_property( TARGET ${ImageDevExamples_INTERFACE_NAME} PROPERTY INTERFACE_INCLUDE_DIRECTORIES "${ImageDevExamples_INCLUDE_DIR}" )

set_property( TARGET ${ImageDevExamples_INTERFACE_NAME}
              PROPERTY INTERFACE_COMPILE_DEFINITIONS "IMAGEDEVDATA_REPOSITORY=\"${ImageDevExamplesData_HOME}\""
                                                     "IMAGEDEVDATA_IMAGES_FOLDER=\"${ImageDevExamplesData_IMAGES_DIR}\""
                                                     "IMAGEDEVDATA_OBJECTS_FOLDER=\"${ImageDevExamplesData_OBJECTS_DIR}\"" )

# The ImageDevExamples package was found.
set( ImageDevExamples_FOUND True )
