#include <ImageDev/ImageDev.h>
#include <ioformat/IOFormat.h>
#include <string.h>

using namespace imagedev;
using namespace ioformat;
using namespace iolink;

int
main( int argc, char* argv[] )
{
    int status = 0;

    try
    {
        // ImageDev library initialization if not done
        if ( isInitialized() == false )
            imagedev::init();

        // Open a grayscale image from a tif file
        auto imageInput = readImage( std::string( IMAGEDEVDATA_IMAGES_FOLDER ) + "objects.tif" );

        // Threshold and label the binary input
        auto imageBin = thresholdingByCriterion( imageInput,
                                                 ThresholdingByCriterion::ComparisonCriterion::GREATER_THAN_OR_EQUAL_TO,
                                                 40 );
        auto imageLab = labeling2d( imageBin, Labeling2d::LABEL_8_BIT, Labeling2d::CONNECTIVITY_8 );

        // Create a group of measurements
        MeasurementGroup::Ptr group = std::make_shared< MeasurementGroup >();
        group->add( NativeMeasurements::inverseCircularity2d );
        group->add( NativeMeasurements::feretDiameter2d );

        // Save this group and reload it
        group->write( "T04_04_group.xml" );
        MeasurementGroup::Ptr saved_group = std::make_shared< MeasurementGroup >();
        saved_group->read( "T04_04_group.xml" );

        // Define the analysis features to be computed
        AnalysisMsr::Ptr analysis = std::make_shared< AnalysisMsr >();
        analysis->select( saved_group->getMeasurements() );

        // Launch the feature extraction on the segmented image
        labelAnalysis( imageLab, imageInput, analysis );

        // Export the analysis in a dataframe and save it in a csv file
        auto dataframe = analysis->toDataFrame();
        writeView( dataframe, "T04_04_analysis.csv" );

        std::cout << "This example ran successfully." << std::endl;
    }
    catch ( const imagedev::Exception& error )
    {
        // Print potential exception in the standard output
        std::cerr << "T04_04_MeasurementGroup exception: " << error.what() << std::endl;
        status = -1;
    }

    // ImageDev library finalization
    imagedev::finish();

    // Check if we must ask for an enter key to close the program
    if ( !( ( argc == 2 ) && strcmp( argv[1], "--no-stop-at-end" ) == 0 ) )
        std::cout << "Press Enter key to close this window." << std::endl, getchar();

    return status;
}
