﻿using System;
using ImageDev;
using IOLink;
using IOFormat;

namespace T04_01_GlobalAnalysis
{
    class Program
    {
        static void Main(string[] args)
        {
            int status = 0;

            try
            {
                // Initialize the ImageDev library if not done
                if (Initialization.IsInitialized() == false)
                    Initialization.Init();

                // Open a tif file to analyze
                ImageView imageInput = ViewIO.ReadImage("Data/images/micro_nucleus.tif") as ImageView;

                // Extract intensity statistics from an RGB image
                Console.WriteLine("Computing color intensity statistics... ");
                var statistics = Processing.IntensityStatistics(imageInput);
                Console.WriteLine("  - Average intensity [R, G, B] = [" + statistics.mean(0, 0).ToString("0.00") +
                                   ", " + statistics.mean(1, 0).ToString("0.00") + ", " +
                                   statistics.mean(2, 0).ToString("0.00") + "]");
                Console.WriteLine("  - Standard Deviation = [R, G, B] = [" +
                                   statistics.standardDeviation(0, 0).ToString("0.00") + ", " +
                                   statistics.standardDeviation(1, 0).ToString("0.00") + ", " +
                                   statistics.standardDeviation(2, 0).ToString("0.00") + "]");
                Console.WriteLine("  - Red intensity range = [" + statistics.minimum(0, 0) + ", " +
                                   statistics.maximum(0, 0) + "]");
                Console.WriteLine("  - Green intensity range = [" + statistics.minimum(1, 0) + ", " +
                                   statistics.maximum(1, 0) + "]");
                Console.WriteLine("  - Blue intensity range = [" + statistics.minimum(2, 0) + ", " +
                                   statistics.maximum(2, 0) + "]");

                // Threshold image pixels having an intensity lower than 190 in each RGB channel
                Console.WriteLine("Color thresholding and binary analysis... ");
                var imageOutput = Processing.ColorThresholding(
                    imageInput, new double[] { 0, 190 }, new double[] { 0, 190 }, new double[] { 0, 190 });

                // Measure the object area
                var area = Processing.Area2d(imageOutput);
                Console.WriteLine("  - Total area of detected objects = " + area.area(0));
                Console.WriteLine("  - Phase fraction of detected objects = " +
                                   (100.0 * area.areaFraction(0)).ToString("0.00") + " %");

                // Count the number of detected objects
                var count = Processing.ObjectCount(imageOutput);
                Console.WriteLine("  - Number of detected objects = " + count.outputMeasurement.count(0));
                // Rescale binary the image intensities between 0 and 255 for visualization and save it
                imageOutput =
                    Processing.RescaleIntensity(imageOutput, RescaleIntensity.OutputType.UNSIGNED_INTEGER_8_BIT);
                ViewIO.WriteView(imageOutput, "T04_01_output.png");

                // Notify the garbage collector that the created images can be freed
                imageInput.Dispose();
                imageOutput.Dispose();

            }
            catch (Exception error)
            {
                // Print potential exception in the standard output
                System.Console.WriteLine("HelloImageDev exception: " + error.ToString());
                status = -1;
            }

            // ImageDev library finalization
            Initialization.Finish();

            // Check if we must ask for an enter key to close the program
            if (!((args.Length >= 1) && (args[0] == "--no-stop-at-end")))
            {
                System.Console.WriteLine("Press Enter key to close this window.");
                System.Console.ReadKey();
            }

            System.Environment.Exit(status);
        }
    }
}
