﻿using System;
using ImageDev;
using IOLink;
using IOFormat;

namespace T04_05_CustomAnalysis
{
    class Program
    {
        static void Main( string[] args )
        {
            int status = 0;

            try
            {
                // Initialize the ImageDev library if not done
                if ( Initialization.IsInitialized() == false )
                    Initialization.Init();

                // Open a grayscale image from a tif file
                ImageView imageInput = ViewIO.ReadImage( "Data/images/objects.tif" );

                // Threshold and label the binary input
                var imageBin = Processing.ThresholdingByCriterion(
                    imageInput, ThresholdingByCriterion.ComparisonCriterion.GREATER_THAN_OR_EQUAL_TO, 40 );
                var imageLab = Processing.Labeling2d( imageBin );

                // Change the number of Feret diameter to 2 (0 and 90 degrees)
                MeasurementAttributes.feret2d().orientationCount = 2;
                MeasurementAttributes.feret2d().Resample();
                Console.WriteLine( "FeretDiameter[0] = " + MeasurementAttributes.feret2d().Orientation( 0 ) );
                Console.WriteLine( "FeretDiameter[1] = " + MeasurementAttributes.feret2d().Orientation( 1 ) );

                // Create a circularity factor between 0 and 1
                string myFormula = "4*Pi*" + NativeMeasurements.Area2d.name + "/" +
                                   NativeMeasurements.PolygonePerimeter2d.name + "**2";
                if ( !Processing.CheckMeasurementFormula( myFormula ) )
                    // Raise an exception if the formula is not valid
                    throw new Exception( "Invalid formula: " + myFormula );

                var myCircularity = AnalysisMsr.RegisterCustomMeasurement( "MyCircularity",
                                                                           myFormula,
                                                                           "A circularity factor between 0 and 1.",
                                                                           Data.PhysicalInformation.RATIO );

                // Define the analysis features to be computed
                AnalysisMsr analysis = new AnalysisMsr();
                analysis.Select( NativeMeasurements.FeretDiameter2d );
                analysis.Select( myCircularity );

                // Launch the feature extraction on the segmented image
                Processing.LabelAnalysis( imageLab, imageInput, analysis );

                // Export the analysis in a dataframe and save it in a csv file
                DataFrameView dataframe = analysis.ToDataFrame();
                ViewIO.WriteView( dataframe, "T04_05_analysis.csv" );

                // Notify the garbage collector that the created images can be freed
                imageInput.Dispose();
                imageBin.Dispose();
                imageLab.Dispose();

                Console.WriteLine( "This example ran successfully." );
            }
            catch ( Exception error )
            {
                // Print potential exception in the standard output
                System.Console.WriteLine( "T04_05_CustomAnalysis exception: " + error.ToString() );
                status = -1;
            }

            // ImageDev library finalization
            Initialization.Finish();

            // Check if we must ask for an enter key to close the program
            if ( !( ( args.Length >= 1 ) && ( args[0] == "--no-stop-at-end" ) ) )
            {
                System.Console.WriteLine( "Press Enter key to close this window." );
                System.Console.ReadKey();
            }

            System.Environment.Exit( status );
        }
    }
}
