using System;
using ImageDev;
using IOLink;
using IOFormat;

namespace T06_01_ParticleAnalysis
{
    class Program
    {
        static void Main(string[] args)
        {
            int status = 0;

            try
            {
                // ImageDev library initialization
                Initialization.Init();

                // Open a standard tif file and display the image properties
                ImageView imageInput = ViewIO.ReadImage("Data/images/polystyrene.tif");

                // Threshold the grayscale image
                ImageView imageThr = Processing.ThresholdingByCriterion(
                    imageInput, ThresholdingByCriterion.ComparisonCriterion.GREATER_THAN, 128);

                // Separate connected particles
                ImageView imageSep = Processing.SeparateObjects(imageThr, 2);
                imageThr.Dispose();

                // Remove particles touching image borders
                ImageView imageBdk = Processing.KillBorder2d(imageSep);
                imageSep.Dispose();

                // Connected component labeling of a binary image
                var imageLab = Processing.Labeling2d(imageBdk,
                                                      Labeling2d.LabelType.LABEL_8_BIT,
                                                      Labeling2d.Neighborhood.CONNECTIVITY_8);

                // The number of particles is the maximum label
                var extrema = Processing.IntensityExtrema(imageLab, 0) as IntensityExtremaMsr;
                int particleCount = (int)extrema.maximum(0, 0, 0);

                // Define the analysis features to be computed
                AnalysisMsr analysis = new AnalysisMsr();
                var centerX = analysis.Select(NativeMeasurements.BarycenterX);
                var centerY = analysis.Select(NativeMeasurements.BarycenterY);
                var diameter = analysis.Select(NativeMeasurements.EquivalentDiameter);
                var intensity = analysis.Select(NativeMeasurements.IntensityMean);

                // Launch the feature extraction on the segmented image
                Processing.LabelAnalysis(imageLab, imageInput, analysis);
                Console.WriteLine("Number of particles = " + particleCount);
                Console.WriteLine("Particle\t" + centerX.Name() + "\t" + centerY.Name() + "\t" + diameter.Name() + "\t" +
                                   intensity.Name());
                // Print the analysis results for 10% of the particles
                for (int i = 0; i < (int)(particleCount / 10); i++)
                {
                    Console.WriteLine((i + 1) + "\t\t" + centerX.Value(i).ToString("0.00") + "\t\t" +
                                       centerY.Value(i).ToString("0.00") + "\t\t" +
                                       diameter.Value(i).ToString("0.00") + "\t\t\t" +
                                       intensity.Value(i).ToString("0.00"));
                }

                // Save the created image with IOFormat
                ViewIO.WriteView(imageBdk, "T06_01_output.png");

            }
            catch (Exception error)
            {
                // Print potential exception in the standard output
                System.Console.WriteLine("HelloImageDev exception: " + error.ToString());
                status = -1;
            }

            // ImageDev library finalization
            Initialization.Finish();

            // Check if we must ask for an enter key to close the program
            if (!((args.Length >= 1) && (args[0] == "--no-stop-at-end")))
            {
                System.Console.WriteLine("Press Enter key to close this window.");
                System.Console.ReadKey();
            }

            System.Environment.Exit(status);
        }
    }
}
