using System;
using ImageDev;
using IOLink;
using IOFormat;

namespace T02_01_CreateImage2D
{
    class Program
    {
        static void Main(string[] args)
        {
            int status = 0;

            try
            {
                // Initialize the ImageDev library if not done
                if (Initialization.IsInitialized() == false)
                    Initialization.Init();

                // Initialize an unsigned 8-bit array storing data of a 2D image
                ulong rowCount = 768;
                ulong colCount = 1024;
                byte[] imageData = new byte[rowCount * colCount];

                // Define a synthetic square in this array
                ulong side = colCount / 2; // side in pixels of the square to draw

                // Loop on image rows
                for (ulong i = 0; i < rowCount; ++i)
                {
                    // Loop on image columns
                    for (ulong j = 0; j < colCount; ++j)
                    {
                        if ((i >= (rowCount - side) / 2) && (i <= (rowCount + side) / 2) &&
                             (j >= (colCount - side) / 2) && (j <= (colCount + side) / 2))
                            imageData[i * colCount + j] = 228; // Value inside the square
                        else
                            imageData[i * colCount + j] = 0; // Background value
                    }
                }

                // Create an image view of same dimensions
                VectorXu64 imageShape = new VectorXu64(colCount, rowCount);
                ImageView image = ImageViewFactory.Allocate(imageShape, DataTypeId.UINT8);
                image.AxesInterpretation = ImageTypeId.IMAGE;

                // Define the region where to write the data
                VectorXu64 imageOrig = new VectorXu64(0, 0);
                RegionXu64 imageRegion = new RegionXu64(imageOrig, imageShape);

                // Copy the data in the image view
                image.WriteRegion(imageRegion, imageData);

                // This image can now be processed by any ImageDev algorithm, for instance for building a color image
                ImageView imageVoid = Processing.ResetImage(image, 0.0f);
                ImageView imageRgb = Processing.GrayscaleToColor(image, image, imageVoid);

                // Save the created image with IOFormat
                ViewIO.WriteView(imageRgb, "T02_01_output.png");
            }
            catch (Exception error)
            {
                // Print potential exception in the standard output
                System.Console.WriteLine("HelloImageDev exception: " + error.ToString());
                status = -1;
            }

            // ImageDev library finalization
            Initialization.Finish();

            // Check if we must ask for an enter key to close the program
            if (!((args.Length >= 1) && (args[0] == "--no-stop-at-end")))
            {
                System.Console.WriteLine("Press Enter key to close this window.");
                System.Console.ReadKey();
            }

            System.Environment.Exit(status);
        }
    }
}
