using System;
using ImageDev;
using IOLink;
using IOFormat;

namespace T02_02_CreateImage3d
{
    class Program
    {
        static void Main(string[] args)
        {
            int status = 0;

            try
            {
                // Initialize the ImageDev library if not done
                if (Initialization.IsInitialized() == false)
                    Initialization.Init();

                // Initialize an unsigned 8-bit array storing data of a 3D image
                ulong rowCount = 220;
                ulong colCount = 192;
                ulong sliCount = 128;
                byte[] imageData = new byte[rowCount * colCount * sliCount];

                // Define a synthetic cube in this array
                ulong side = colCount / 2; // side in pixels of the cube to draw

                // Loop on image slices
                for (ulong k = 0; k < sliCount; ++k)
                {
                    // Loop on image rows
                    for (ulong i = 0; i < rowCount; ++i)
                    {
                        // Loop on image columns
                        for (ulong j = 0; j < colCount; ++j)
                        {
                            if ((i >= (rowCount - side) / 2) && (i <= (rowCount + side) / 2) &&
                                 (j >= (colCount - side) / 2) && (j <= (colCount + side) / 2) &&
                                 (k >= (sliCount - side) / 2) && (k <= (sliCount + side) / 2))
                                imageData[k * rowCount * colCount + i * colCount + j] = 228; // Value inside the cube
                            else
                                imageData[k * rowCount * colCount + i * colCount + j] = 0; // Background value
                        }
                    }
                }

                // Create an image view of same dimensions
                VectorXu64 imageShape = new VectorXu64(colCount, rowCount, sliCount);
                ImageView image = ImageViewFactory.Allocate(imageShape, DataTypeId.UINT8);
                image.AxesInterpretation = ImageTypeId.VOLUME;

                // Define the region where to write the data
                VectorXu64 imageOrig = new VectorXu64(0, 0, 0);
                RegionXu64 imageRegion = new RegionXu64(imageOrig, imageShape);

                // Copy the data in the image view
                image.WriteRegion(imageRegion, imageData);

                // This image can now be processed by any ImageDev algorithm, for instance to generate a blur effect on its
                // edges
                ImageView imageGauss = Processing.GaussianFilter3d(image,
                                                                    GaussianFilter3d.FilterMode.SEPARABLE,
                                                                    new double[] { 10.0, 10.0, 10.0 });

                // Save the created image with IOFormat
                ViewIO.WriteView(imageGauss, "T02_02_output.tif");

            }
            catch (Exception error)
            {
                // Print potential exception in the standard output
                System.Console.WriteLine("HelloImageDev exception: " + error.ToString());
                status = -1;
            }

            // ImageDev library finalization
            Initialization.Finish();

            // Check if we must ask for an enter key to close the program
            if (!((args.Length >= 1) && (args[0] == "--no-stop-at-end")))
            {
                System.Console.WriteLine("Press Enter key to close this window.");
                System.Console.ReadKey();
            }

            System.Environment.Exit(status);
        }
    }
}
