﻿using System;
using ImageDev;
using IOLink;
using IOFormat;

namespace T02_03_TransferData
{
    class Program
    {
        static void Main(string[] args)
        {
            int status = 0;

            try
            {
                // Initialize the ImageDev library if not done
                if (Initialization.IsInitialized() == false)
                    Initialization.Init();

                // Initialize an unsigned 8-bit array storing data of a 3D image
                int rowCount = 220;
                int colCount = 192;
                int sliCount = 128;
                byte[] imageData = new byte[rowCount * colCount * sliCount];

                // Define a synthetic sphere in this array
                int squareRadius = (sliCount / 2 - 10) * (sliCount / 2 - 10); // Radius of the sphere to draw
                int distToCenter;

                // Loop on image slices
                for (int k = 0; k < sliCount; ++k)
                {
                    // Loop on image rows
                    for (int i = 0; i < rowCount; ++i)
                    {
                        // Loop on image columns
                        for (int j = 0; j < colCount; ++j)
                        {
                            distToCenter = (i - rowCount / 2) * (i - rowCount / 2) +
                                           (j - colCount / 2) * (j - colCount / 2) +
                                           (k - sliCount / 2) * (k - sliCount / 2);
                            if (distToCenter <= squareRadius)
                                imageData[k * rowCount * colCount + i * colCount + j] = 200; // Value inside the sphere
                            else
                                imageData[k * rowCount * colCount + i * colCount + j] = 0; // Background value
                        }
                    }
                }

                // Create an image view of same dimensions
                VectorXu64 imageShape = new VectorXu64((ulong)colCount, (ulong)rowCount, (ulong)sliCount);
                ImageView image = ImageViewFactory.FromBuffer(
                    imageShape, DataTypeId.UINT8, imageData, (uint)(rowCount * colCount * sliCount * sizeof(byte)));
                image.AxesInterpretation = ImageTypeId.VOLUME;


                // This image can now be processed by any ImageDev algorithm, for instance to add a Gaussian noise inside
                ImageView imageNoise = Processing.RandomGaussianImage3d(
                    RandomGaussianImage3d.OutputType.SIGNED_INTEGER_8_BIT, colCount, rowCount, sliCount, 0.0f, 20.0f);
                ImageView imageOut = Processing.ArithmeticOperationWithImage(
                    image, imageNoise, ArithmeticOperationWithImage.ArithmeticOperator.ADD);

                // Save the created image with IOFormat
                ViewIO.WriteView(imageOut, "T02_03_output.tif");

            }
            catch (Exception error)
            {
                // Print potential exception in the standard output
                System.Console.WriteLine("HelloImageDev exception: " + error.ToString());
                status = -1;
            }

            // ImageDev library finalization
            Initialization.Finish();

            // Check if we must ask for an enter key to close the program
            if (!((args.Length >= 1) && (args[0] == "--no-stop-at-end")))
            {
                System.Console.WriteLine("Press Enter key to close this window.");
                System.Console.ReadKey();
            }

            System.Environment.Exit(status);
        }
    }
}
