﻿using System;
using ImageDev;
using IOLink;
using IOFormat;

namespace T03_02_Thresholding
{
    class Program
    {
        static void Main(string[] args)
        {
            int status = 0;

            try
            {
                // Initialize the ImageDev library if not done
                if (Initialization.IsInitialized() == false)
                    Initialization.Init();

                // Open a tif file to binarize
                ImageView imageInput = ViewIO.ReadImage("Data/images/objects.tif") as ImageView;

                // Threshold image pixels having an intensity between 60 and 255
                Console.WriteLine("Thresholding between a minimum and maximum value...");
                ImageView imageOutput = Processing.Thresholding(imageInput, new double[] { 60, 255 }) as ImageView;
                // Rescale the binary image intensities between 0 and 255 for visualization and save it
                imageOutput =
                    Processing.RescaleIntensity(imageOutput, RescaleIntensity.OutputType.UNSIGNED_INTEGER_8_BIT);
                ViewIO.WriteView(imageOutput, "T03_02_manual.png");

                // Threshold image pixels having an intensity greater than 40
                Console.WriteLine("Thresholding by comparison to a value...");
                imageOutput = Processing.ThresholdingByCriterion(
                    imageInput, ThresholdingByCriterion.ComparisonCriterion.GREATER_THAN_OR_EQUAL_TO, 40);
                // Rescale the binary image intensities between 0 and 255 for visualization and save it
                imageOutput =
                    Processing.RescaleIntensity(imageOutput, RescaleIntensity.OutputType.UNSIGNED_INTEGER_8_BIT);
                ViewIO.WriteView(imageOutput, "T03_02_criterion.png");

                // Threshold image pixels automatically with Otsu's method
                Console.WriteLine("Automatic thresholding with Otsu method...");
                var result = Processing.AutoThresholdingBright(imageInput,
                                                                AutoThresholdingBright.RangeMode.MIN_MAX,
                                                                null,
                                                                AutoThresholdingBright.ThresholdCriterion.FACTORISATION);
                Console.WriteLine("Otsu threshold value = " + result.outputMeasurement.threshold(0));
                // Rescale the binary image intensities between 0 and 255 for visualization and save it
                imageOutput = Processing.RescaleIntensity(result.outputBinaryImage,
                                                           RescaleIntensity.OutputType.UNSIGNED_INTEGER_8_BIT);
                ViewIO.WriteView(imageOutput, "T03_02_auto.png");

                // Notify the garbage collector that the created images can be freed
                imageInput.Dispose();
                imageOutput.Dispose();

            }
            catch (Exception error)
            {
                // Print potential exception in the standard output
                System.Console.WriteLine("HelloImageDev exception: " + error.ToString());
                status = -1;
            }

            // ImageDev library finalization
            Initialization.Finish();

            // Check if we must ask for an enter key to close the program
            if (!((args.Length >= 1) && (args[0] == "--no-stop-at-end")))
            {
                System.Console.WriteLine("Press Enter key to close this window.");
                System.Console.ReadKey();
            }

            System.Environment.Exit(status);
        }
    }
}
