﻿using System;
using ImageDev;
using IOLink;
using IOFormat;
using System.Linq;

namespace T04_03_MeasurementBrowsing
{
    class Program
    {
        static void Main( string[] args )
        {
            int status = 0;

            try
            {
                // Initialize the ImageDev library if not done
                if ( Initialization.IsInitialized() == false )
                    Initialization.Init();

                // Open a grayscale image from a tif file
                ImageView imageInput = ViewIO.ReadImage( "Data/images/objects.tif" );

                // Threshold and label the binary input
                var imageBin = Processing.ThresholdingByCriterion(
                    imageInput, ThresholdingByCriterion.ComparisonCriterion.GREATER_THAN_OR_EQUAL_TO, 40 );
                var imageLab = Processing.Labeling2d( imageBin );

                // Calibrate this image to match 1 pixel to 1.4 mm
                Vector3d spacing = new Vector3d( 1.4, 1.4, 1 );
                imageLab.SpatialSpacing = spacing;
                imageLab.SpatialUnit = "mm";

                // Define the analysis features to be computed
                AnalysisMsr analysis = new AnalysisMsr();
                analysis.Select( NativeMeasurements.Area2d );
                analysis.Select( NativeMeasurements.InverseCircularity2d );
                analysis.Select( NativeMeasurements.FeretDiameter2d );

                // Launch the feature extraction on the segmented image
                Processing.LabelAnalysis( imageLab, imageInput, analysis );

                // Print the analysis table header
                string lineToPrint = "Label\t";
                foreach ( var measure in analysis.Measurements() )
                {
                    // Build and print the table header
                    if ( measure.Shape().Length == 1 )
                        // The measurement is a scalar value
                        lineToPrint += measure.Name() + "(" + measure.information.physicalUnit + ")\t";
                    else if ( measure.Shape().Length == 2 )
                        // The measurement is an array, loop on it
                        for ( int j = 0; j < measure.Shape()[1]; ++j )
                            lineToPrint += measure.Name() + "[" + j + "](" + measure.information.physicalUnit + ")\t";
                }
                Console.WriteLine( lineToPrint );

                int[] index;
                // Print all measurement results for each label
                for ( int i = 0; i < analysis.LabelCount(); ++i )
                {
                    lineToPrint = ( i + 1 ) + "\t";
                    foreach ( var measure in analysis.Measurements() )
                    {
                        index = measure.Shape();
                        index[0] = i;
                        if ( measure.Shape().Length == 1 )
                            // The measurement is a scalar value
                            lineToPrint +=
                                measure.ToDouble( index.Select( item => ( long )item ).ToArray() ).ToString() + "\t";
                        else if ( measure.Shape().Length == 2 )
                            // The measurement is an array, loop on it
                            for ( int j = 0; j < measure.Shape()[1]; ++j )
                            {
                                index[1] = j;
                                lineToPrint +=
                                    measure.ToDouble( index.Select( item => ( long )item ).ToArray() ).ToString() +
                                    "\t";
                            }
                    }
                    Console.WriteLine( lineToPrint );
                }

                // Export the analysis in a dataframe and save it in a csv file
                DataFrameView dataframe = analysis.ToDataFrame();
                ViewIO.WriteView( dataframe, "T04_03_analysis.csv" );

                // Notify the garbage collector that the created images can be freed
                imageInput.Dispose();
                imageBin.Dispose();
                imageLab.Dispose();

                Console.WriteLine( "This example ran successfully." );
            }
            catch ( Exception error )
            {
                // Print potential exception in the standard output
                System.Console.WriteLine( "T04_03_MeasurementBrowsing exception: " + error.ToString() );
                status = -1;
            }

            // ImageDev library finalization
            Initialization.Finish();

            // Check if we must ask for an enter key to close the program
            if ( !( ( args.Length >= 1 ) && ( args[0] == "--no-stop-at-end" ) ) )
            {
                System.Console.WriteLine( "Press Enter key to close this window." );
                System.Console.ReadKey();
            }

            System.Environment.Exit( status );
        }
    }
}
