﻿using System;
using ImageDev;
using IOLink;
using IOFormat;

namespace T04_04_MeasurementGroup
{
    class Program
    {
        static void Main( string[] args )
        {
            int status = 0;

            try
            {
                // Initialize the ImageDev library if not done
                if ( Initialization.IsInitialized() == false )
                    Initialization.Init();

                // Open a grayscale image from a tif file
                ImageView imageInput = ViewIO.ReadImage( "Data/images/objects.tif" );

                // Threshold and label the binary input
                var imageBin = Processing.ThresholdingByCriterion(
                    imageInput, ThresholdingByCriterion.ComparisonCriterion.GREATER_THAN_OR_EQUAL_TO, 40 );
                var imageLab = Processing.Labeling2d( imageBin );

                // Create a group of measurements
                MeasurementGroup group = new MeasurementGroup();
                group.Add( NativeMeasurements.InverseCircularity2d );
                group.Add( NativeMeasurements.FeretDiameter2d );

                // Save this group and reload it
                group.Write( "T04_04_group.xml" );
                MeasurementGroup saved_group = new MeasurementGroup();
                saved_group.Read( "T04_04_group.xml" );

                // Define the analysis features to be computed
                AnalysisMsr analysis = new AnalysisMsr();
                analysis.Select( saved_group.GetMeasurements() );

                // Launch the feature extraction on the segmented image
                Processing.LabelAnalysis( imageLab, imageInput, analysis );

                // Export the analysis in a dataframe and save it in a csv file
                DataFrameView dataframe = analysis.ToDataFrame();
                ViewIO.WriteView( dataframe, "T04_04_analysis.csv" );

                // Notify the garbage collector that the created images can be freed
                imageInput.Dispose();
                imageBin.Dispose();
                imageLab.Dispose();

                Console.WriteLine( "This example ran successfully." );
            }
            catch ( Exception error )
            {
                // Print potential exception in the standard output
                System.Console.WriteLine( "T04_04_MeasurementGroup exception: " + error.ToString() );
                status = -1;
            }

            // ImageDev library finalization
            Initialization.Finish();

            // Check if we must ask for an enter key to close the program
            if ( !( ( args.Length >= 1 ) && ( args[0] == "--no-stop-at-end" ) ) )
            {
                System.Console.WriteLine( "Press Enter key to close this window." );
                System.Console.ReadKey();
            }

            System.Environment.Exit( status );
        }
    }
}
