import imagedev
import imagedev_data
import ioformat
import iolink

try:
    # Open a grayscale image from a tif file
    image_input = ioformat.read_image(imagedev_data.get_image_path('objects.tif'))

    # Threshold and label the binary input
    image_bin = imagedev.thresholding_by_criterion(image_input, comparison_value=40)
    image_lab = imagedev.labeling_2d(image_bin, imagedev.Labeling2d.LabelType.LABEL_8_BIT)

    # Calibrate this image to match 1 pixel to 1.4 mm
    image_lab.spatial_spacing = iolink.Vector3d(1.4, 1.4, 1)
    image_lab.spatial_unit = 'mm'

    # Define the analysis features to be computed
    analysis = imagedev.AnalysisMsr()
    analysis.select(imagedev.native_measurements.Area2d)
    analysis.select(imagedev.native_measurements.InverseCircularity2d)
    analysis.select(imagedev.native_measurements.FeretDiameter2d)

    # Launch the feature extraction on the segmented image
    imagedev.label_analysis(image_lab, image_input, analysis)

    # Print the analysis table header
    line_to_print = 'Label\t'
    for measure in analysis.measurements:
        # Build and print the table header
        if len(measure.shape) == 1:
            # The measurement is a scalar value
            line_to_print += measure.name + '(' + measure.information.physical_unit + ')\t'
        elif len(measure.shape) == 2:
            # The measurement is an array, loop on it
            for j in range(0, measure.shape[1]):
                line_to_print += measure.name + '[' + str(j) + '](' + measure.information.physical_unit + ')\t'
    print(line_to_print)

    # Print all measurement results for each label
    for i in range(0, analysis.label_count):
        line_to_print = str(i + 1) + '\t\t'
        for measure in analysis.measurements:
            if len(measure.shape) == 1:
                # The measurement is a scalar value
                line_to_print += '{:.2f}'.format(measure.value(i)) + '\t\t'
            elif len(measure.shape) == 2:
                # The measurement is an array, loop on it
                for j in range(0, measure.shape[1]):
                    line_to_print += '{:.2f}'.format(measure.value(i, j)) + '\t\t\t\t'
        print(line_to_print)

    # Export the analysis in a dataframe and save it in a csv file
    dataframe = analysis.to_data_frame()
    print(dataframe)
    ioformat.write_view(dataframe, 'T04_03_analysis.csv')

    print("This example ran successfully.")
except Exception as error:
    # Print potential exception in the standard output
    print("T04_03_MeasurementBrowsing exception: " + str(error))