import imagedev
import imagedev_data
import ioformat
import iolink

# Open and display a tif file
image_input = ioformat.read_image(imagedev_data.get_image_path("shale.am"))

# Preprocess the input with a bilateral filter for denoising
image_denoised = imagedev.bilateral_filter_3d(image_input)

# STEP 1: Pore segmentation with the watershed algorithm
# Threshold the low graylevels (pore markers)
image_low_bin = imagedev.thresholding_by_criterion(\
  image_denoised, imagedev.ThresholdingByCriterion.ComparisonCriterion.LESS_THAN, 40)

# Slightly erode these markers in order to avoid contact with the high level markers
image_low_ero = imagedev.erosion_3d(image_low_bin, 1)

# Initialize the marker image (label 1)
image_low_lab = imagedev.convert_image(image_low_ero, imagedev.ConvertImage.LABEL_16_BIT)

# Threshold the high graylevels (mineral markers)
image_high_bin = imagedev.thresholding_by_criterion(\
  image_denoised, imagedev.ThresholdingByCriterion.ComparisonCriterion.GREATER_THAN, 90)

# Merge high intensity markers with label 2
image_markers = imagedev.add_object_to_label(image_high_bin, image_low_lab, 2)

# Extract the gradient to build the relief image for watershed
image_grad = imagedev.gradient_magnitude_3d(image_denoised,
                                            imagedev.GradientMagnitude3d.GAUSSIAN, [0.9, 0.9, 0.9],
                                            imagedev.GradientMagnitude3d.RECURSIVE,
                                            imagedev.GradientMagnitude3d.SAME_AS_INPUT,
                                            imagedev.GradientMagnitude3d.AMPLITUDE_EUCLIDEAN)

# Perform the Watershed to detect pore boundaries
image_watershed = imagedev.marker_based_watershed_3d(image_grad, image_markers,\
                                                     output_type=imagedev.MarkerBasedWatershed3d.OutputType.LINES)

# Fill all pores in each slice of the volume
image_watershed.axes_interpretation = iolink.ImageTypeId.IMAGE_SEQUENCE
image_filled = imagedev.fill_holes_2d(image_watershed, imagedev.FillHoles2d.CONNECTIVITY_4)
image_filled.axes_interpretation = iolink.ImageTypeId.VOLUME

# Remove pores touching the image border
image_inside = imagedev.kill_border_3d(image_filled)

# Assign labels to the pores
image_pores = imagedev.labeling_3d(image_inside)

# STEP 2: Pore quantification
# Compute the pore volume percentage
pore_volume = imagedev.intensity_integral_3d(image_inside)
print("  - Pore volume fraction = " + "{:.2f}".format(pore_volume.volume_fraction()*100) + "%")

# Compute the pore number
_, pore_count = imagedev.object_count(image_pores)
print("  - Pore number = " + str(pore_count.count(0)))

# Define the analysis features to be computed
analysis = imagedev.AnalysisMsr()
diameter = analysis.select(imagedev.native_measurements.EquivalentDiameter)
volume = analysis.select(imagedev.native_measurements.Volume3d)
shape = analysis.select(imagedev.native_measurements.InverseSphericity3d)

# Launch the feature extraction on the segmented image
imagedev.label_analysis(image_pores, None, analysis)
print("Pore\t" + diameter.name + "\t" + volume.name + "\t" + shape.name)
# Print the analysis results for 5% of the pores
for i in range(int(analysis.label_count/20)):
  print(str(i+1) + '\t\t\t' + "{:.2f}".format(diameter.value(i)) +'\t\t\t\t' + "{:.2f}".format(volume.value(i)) +\
    '\t\t' + "{:.2f}".format(shape.value(i)))

# Save the segmented image with IOFormat
ioformat.write_view(image_pores, "T06_02_output.tif")