package inventor.advanced.multiInstancing.algebraicshape;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.io.File;
import java.nio.FloatBuffer;

import javax.swing.BoxLayout;
import javax.swing.JPanel;

import com.openinventor.inventor.SbDataType.DataTypes;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbVec3i32;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.devices.SoBufferObject.AccessModes;
import com.openinventor.inventor.devices.SoCpuBufferObject;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.nodes.SoAlgebraicShape.ASShaderSlots;
import com.openinventor.inventor.nodes.SoInstanceParameter.PredefinedParameters;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  private final static int NUM_INSTANCE_SPHERE_X = 50;
  private final static int NUM_INSTANCE_SPHERE_Y = NUM_INSTANCE_SPHERE_X;
  private final static int NUM_INSTANCE_SPHERE_Z = NUM_INSTANCE_SPHERE_X;

  private final static int NUM_INSTANCE_SPHERE =
      (NUM_INSTANCE_SPHERE_X * NUM_INSTANCE_SPHERE_Y * NUM_INSTANCE_SPHERE_Z);

  private IViewerExaminer myViewer;
  private SoAlgebraicSphere m_qShape = null;
  private SoShaderParameter1f m_cmMinParam = null;
  private SoShaderParameter1f m_cmMaxParam = null;

  @Override
  public void start()
  {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    // Add path to find our data
    SoInput.addDirectoryFirst(getDataPath() + "shaders");
    SoSeparator sceneGraph = createSceneGraph();

    myViewer.setSceneGraph(sceneGraph);
    myViewer.viewAll();

    // SWING part
    JPanel sliderPanel = createSliderPanel();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(sliderPanel, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  private SoSeparator createSceneGraph()
  {
    SoSeparator newScene = new SoSeparator();
    newScene.addChild(new SoGradientBackground());

    // add color map
    SoColorMap colorMap = new SoColorMap();
    colorMap.predefinedColorMap.setValue(SoColorMap.PredefinedColorMaps.STANDARD);
    newScene.addChild(colorMap);

    // create spheres
    {
      // Create SoMultipleInstance group node where to add our geometry
      SoMultipleInstance multiInstanceGroup = new SoMultipleInstance();
      multiInstanceGroup.numInstances.setValue(NUM_INSTANCE_SPHERE);
      newScene.addChild(multiInstanceGroup);

      // translation attribute
      {
        SoInstanceParameter vertexParameters = new SoInstanceParameter();
        vertexParameters.components.setValue(3);
        vertexParameters.type.setValue(DataTypes.FLOAT);
        {
          SoCpuBufferObject bufferValues = new SoCpuBufferObject();
          bufferValues.setSize(NUM_INSTANCE_SPHERE * Float.SIZE / 8 * 3);
          {
            FloatBuffer data = bufferValues.map(AccessModes.SET).asFloatBuffer();
            for ( int k = 0; k < NUM_INSTANCE_SPHERE_Z; k++ )
              for ( int j = 0; j < NUM_INSTANCE_SPHERE_Y; j++ )
                for ( int i = 0; i < NUM_INSTANCE_SPHERE_X; i++ )
                {
                  int index = (i + j * NUM_INSTANCE_SPHERE_X + k * NUM_INSTANCE_SPHERE_X * NUM_INSTANCE_SPHERE_Y) * 3;
                  data.put(index, i * 4.f);
                  data.put(index + 1, j * 4.f);
                  data.put(index + 2, k * 4.f);
                }
          }
          bufferValues.unmap();

          vertexParameters.value.setValue(bufferValues);
        }
        vertexParameters.name
            .setValue(SoInstanceParameter.getPredefinedParameterName(PredefinedParameters.TRANSLATION));
        multiInstanceGroup.parameters.set1Value(0, vertexParameters);
      }

      // scale attribute
      {
        SoInstanceParameter vertexParameters = new SoInstanceParameter();
        vertexParameters.components.setValue(3);
        vertexParameters.type.setValue(DataTypes.FLOAT);
        {
          SoCpuBufferObject bufferValues = new SoCpuBufferObject();
          bufferValues.setSize(NUM_INSTANCE_SPHERE * Float.SIZE / 8 * 3);
          {
            FloatBuffer data = bufferValues.map(AccessModes.SET).asFloatBuffer();
            for ( int k = 0; k < NUM_INSTANCE_SPHERE_Z; k++ )
            {
              for ( int j = 0; j < NUM_INSTANCE_SPHERE_Y; j++ )
              {
                for ( int i = 0; i < NUM_INSTANCE_SPHERE_X; i++ )
                {
                  int index = (i + j * NUM_INSTANCE_SPHERE_X + k * NUM_INSTANCE_SPHERE_X * NUM_INSTANCE_SPHERE_Y) * 3;
                  float s =
                      (float) (1.f + 0.5f * Math.cos(10.f * (float) Math.PI * (i / (float) (NUM_INSTANCE_SPHERE_X)))
                          * (float) Math.cos(10.f * (float) Math.PI * (j / (float) (NUM_INSTANCE_SPHERE_Y)))
                          * (float) Math.cos(10.f * (float) Math.PI * (k / (float) (NUM_INSTANCE_SPHERE_Z))));
                  data.put(index, s);
                  data.put(index + 1, s);
                  data.put(index + 2, s);
                }
              }
            }
            bufferValues.unmap();
          }
          vertexParameters.value.setValue(bufferValues);
        }
        vertexParameters.name.setValue(SoInstanceParameter.getPredefinedParameterName(PredefinedParameters.SCALE));
        multiInstanceGroup.parameters.set1Value(1, vertexParameters);
      }

      // rotation attribute
      {
        SoInstanceParameter vertexParameters = new SoInstanceParameter();
        vertexParameters.components.setValue(4);
        vertexParameters.type.setValue(DataTypes.FLOAT);
        {
          SoCpuBufferObject bufferValues = new SoCpuBufferObject();
          bufferValues.setSize(NUM_INSTANCE_SPHERE * Float.SIZE / 8 * 4);
          {
            FloatBuffer data = bufferValues.map(AccessModes.SET).asFloatBuffer();
            for ( int k = 0; k < NUM_INSTANCE_SPHERE_Z; k++ )
              for ( int j = 0; j < NUM_INSTANCE_SPHERE_Y; j++ )
                for ( int i = 0; i < NUM_INSTANCE_SPHERE_X; i++ )
                {
                  int index = (i + j * NUM_INSTANCE_SPHERE_X + k * NUM_INSTANCE_SPHERE_X * NUM_INSTANCE_SPHERE_Y) * 4;
                  data.put(index, 0);
                  data.put(index + 1, 0);
                  data.put(index + 2, 0);
                  data.put(index + 3, 1);
                }
          }
          vertexParameters.value.setValue(bufferValues);
        }
        vertexParameters.name.setValue(SoInstanceParameter.getPredefinedParameterName(PredefinedParameters.ROTATION));
        multiInstanceGroup.parameters.set1Value(2, vertexParameters);
      }

      // add the specific instanced shape
      {
        m_qShape = new SoAlgebraicSphere();

        SoVertexShader vs = new SoVertexShader();
        vs.sourceProgram.setValue("oivASSlotColorCustomColormap_vert.glsl");
        SoFragmentShader fs = new SoFragmentShader();
        fs.sourceProgram.setValue("oivASSlotColorCustomColormap_frag.glsl");

        m_qShape.shaderSlots.set1Value(ASShaderSlots.VERTEX_SHADER_ENTRY.getValue(), vs);
        m_qShape.shaderSlots.set1Value(ASShaderSlots.COMPUTE_COLOR.getValue(), fs);

        // shader parameters
        SoShaderParameter1i cmParam = new SoShaderParameter1i();
        cmParam.name.setValue("colorMap");
        cmParam.value.setValue(1);
        fs.parameter.addShaderParameter(cmParam);

        m_cmMinParam = new SoShaderParameter1f();
        m_cmMinParam.name.setValue("cmMin");
        m_cmMinParam.value.setValue(0);
        vs.parameter.addShaderParameter(m_cmMinParam);

        m_cmMaxParam = new SoShaderParameter1f();
        m_cmMaxParam.name.setValue("cmMax");
        m_cmMaxParam.value.setValue(1);
        vs.parameter.addShaderParameter(m_cmMaxParam);

        multiInstanceGroup.addChild(m_qShape);
      }
    }

    SoMaterial mat = new SoMaterial();
    mat.diffuseColor.setValue(0.5f, 0.5f, 0.5f);
    mat.specularColor.setValue(0.8f, 0.5f, 0.5f);
    mat.shininess.set1Value(0, 0.25f);
    newScene.addChild(mat);

    // create cylinders along X, Y and Z
    {
      SbVec3i32 numInstanceX = new SbVec3i32(NUM_INSTANCE_SPHERE_X - 1, NUM_INSTANCE_SPHERE_X, NUM_INSTANCE_SPHERE_X);
      SbVec3i32 numInstanceY = new SbVec3i32(NUM_INSTANCE_SPHERE_Y, NUM_INSTANCE_SPHERE_Y - 1, NUM_INSTANCE_SPHERE_Y);
      SbVec3i32 numInstanceZ = new SbVec3i32(NUM_INSTANCE_SPHERE_Z, NUM_INSTANCE_SPHERE_Z, NUM_INSTANCE_SPHERE_Z - 1);
      SbVec3i32 numInstance =
          new SbVec3i32(numInstanceX.array[0] * numInstanceY.array[0] * numInstanceZ.array[0], numInstanceX.array[1]
              * numInstanceY.array[1] * numInstanceZ.array[1], numInstanceX.array[2] * numInstanceY.array[2]
              * numInstanceZ.array[2]);

      // Create SoMultipleInstance group node where to add our geometry
      SoMultipleInstance multiInstanceGroup = new SoMultipleInstance();
      multiInstanceGroup.numInstances.setValue(numInstance.array[0] + numInstance.array[1] + numInstance.array[2]);
      newScene.addChild(multiInstanceGroup);

      // translation attribute
      {
        SoInstanceParameter vertexParameters = new SoInstanceParameter();
        vertexParameters.components.setValue(3);
        vertexParameters.type.setValue(DataTypes.FLOAT);
        {
          SoCpuBufferObject bufferValues = new SoCpuBufferObject();
          bufferValues.setSize((numInstance.array[0] + numInstance.array[1] + numInstance.array[2]) * Float.SIZE / 8
              * 3);
          {
            FloatBuffer data = bufferValues.map(AccessModes.SET).asFloatBuffer();
            int decal = 0;
            for ( int d = 0; d < 3; ++d )
            {
              for ( int k = 0; k < numInstanceZ.array[d]; k++ )
              {
                for ( int j = 0; j < numInstanceY.array[d]; j++ )
                {
                  for ( int i = 0; i < numInstanceX.array[d]; i++ )
                  {
                    SbVec3f tra = new SbVec3f(i * 4.f, j * 4.f, k * 4.f);
                    tra.array[d] += 2.f;
                    int index =
                        (decal + i + j * numInstanceX.array[d] + k * numInstanceX.array[d] * numInstanceY.array[d]) * 3;
                    data.put(index, tra.array[0]);
                    data.put(index + 1, tra.array[1]);
                    data.put(index + 2, tra.array[2]);
                  }
                }
              }
              decal += numInstance.array[d];
            }
          }
          bufferValues.unmap();
          vertexParameters.value.setValue(bufferValues);
        }
        vertexParameters.name
            .setValue(SoInstanceParameter.getPredefinedParameterName(PredefinedParameters.TRANSLATION));
        multiInstanceGroup.parameters.set1Value(0, vertexParameters);
      }

      // scale attribute
      {
        SoInstanceParameter vertexParameters = new SoInstanceParameter();
        vertexParameters.components.setValue(3);
        vertexParameters.type.setValue(DataTypes.FLOAT);
        {
          SoCpuBufferObject bufferValues = new SoCpuBufferObject();
          bufferValues.setSize(Float.SIZE / 8 * 3);
          FloatBuffer bufferData = bufferValues.map(AccessModes.SET).asFloatBuffer();
          bufferData.put(0, 0.25f);
          bufferData.put(1, 1.9f);
          bufferData.put(2, 0.25f);
          bufferValues.unmap();
          vertexParameters.value.setValue(bufferValues);
        }
        vertexParameters.name.setValue(SoInstanceParameter.getPredefinedParameterName(PredefinedParameters.SCALE));
        vertexParameters.divisor.setValue(numInstance.array[0] + numInstance.array[1] + numInstance.array[2]);
        multiInstanceGroup.parameters.set1Value(1, vertexParameters);
      }

      // rotation attribute
      {
        SoInstanceParameter vertexParameters = new SoInstanceParameter();
        vertexParameters.components.setValue(4);
        vertexParameters.type.setValue(DataTypes.FLOAT);
        {
          SoCpuBufferObject bufferValues = new SoCpuBufferObject();
          bufferValues.setSize(3 * Float.SIZE / 8 * 4);
          FloatBuffer bufferData = bufferValues.map(AccessModes.SET).asFloatBuffer();
          float hsqrt2 = 0.5f * (float) Math.sqrt(2);
          bufferData.put(0, 0.f);
          bufferData.put(1, 0.f);
          bufferData.put(2, hsqrt2);
          bufferData.put(3, hsqrt2);
          bufferData.put(4, 0.f);
          bufferData.put(5, 0.f);
          bufferData.put(6, 0.f);
          bufferData.put(7, 1.f);
          bufferData.put(8, hsqrt2);
          bufferData.put(9, 0.f);
          bufferData.put(10, 0.f);
          bufferData.put(11, hsqrt2);
          bufferValues.unmap();
          vertexParameters.value.setValue(bufferValues);
        }
        vertexParameters.name.setValue(SoInstanceParameter.getPredefinedParameterName(PredefinedParameters.ROTATION));
        vertexParameters.divisor.setValue(numInstance.array[0]);
        multiInstanceGroup.parameters.set1Value(2, vertexParameters);
      }

      // add the specific instanced shape
      {
        SoAlgebraicCylinder rcs = new SoAlgebraicCylinder();
        multiInstanceGroup.addChild(rcs);
      }
    }

    // replace the original scene graph by the global collected shape
    return newScene;
  }

  private String getDataPath()
  {
    String pkgName = this.getClass().getPackage().getName();
    pkgName = pkgName.replace('.', File.separatorChar);
    return SoPreferences.getValue("OIVJHOME") + File.separator + "examples" + File.separator + pkgName + File.separator;
  }

  private JPanel createSliderPanel()
  {
    SliderPanel radiusSphereSlider = new SliderPanel(0.1f, 2.0f, 1.f, 3);
    radiusSphereSlider.addInfoText("Radius : ");
    radiusSphereSlider.setSliderSize(new Dimension(200, 20));
    radiusSphereSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_qShape.radius.setValue(value);
      }
    });

    SliderPanel cmMinSlider = new SliderPanel(0.f, 1.f, 0.f, 3);
    cmMinSlider.addInfoText("Min colormap : ");
    cmMinSlider.setSliderSize(new Dimension(200, 20));
    cmMinSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_cmMinParam.value.setValue(value);
      }
    });

    SliderPanel cmMaxSlider = new SliderPanel(0.f, 1.f, 1.f, 3);
    cmMaxSlider.addInfoText("Max colormap : ");
    cmMaxSlider.setSliderSize(new Dimension(200, 20));
    cmMaxSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_cmMaxParam.value.setValue(value);
      }
    });

    JPanel sliderPanel = new JPanel();
    sliderPanel.setLayout(new BoxLayout(sliderPanel, BoxLayout.Y_AXIS));
    sliderPanel.add(radiusSphereSlider);
    sliderPanel.add(cmMinSlider);
    sliderPanel.add(cmMaxSlider);

    return sliderPanel;
  }

  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("Instanced Algebraic Shapes");
  }
}
