package inventor.advanced.overlayPanels;

import java.util.ArrayList;
import java.util.Random;

import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoFaceSet;
import com.openinventor.inventor.nodes.SoLineSet;
import com.openinventor.inventor.nodes.SoShapeHints;
import com.openinventor.inventor.nodes.SoTransform;
import com.openinventor.inventor.nodes.SoOverlayGroup;
import com.openinventor.inventor.nodes.SoSelection;
import com.openinventor.inventor.nodes.SoDrawStyle;
import com.openinventor.inventor.nodes.SoText3;
import com.openinventor.inventor.nodes.SoVertexProperty;
import com.openinventor.inventor.nodes.SoFont;

/*
The class is building a scene graph composed of panels on which are aggrageted different co-planar geometries with of without transparency.
Each panel requires its specific SoOverlayGroup in which the geometries must added from back to front. 
*/

public class SceneOverlayPanels {
    private int numPtsProfile = 4;
    private int numPanels = numPtsProfile - 1;
    private int numPolylines = 4;
    ArrayList<SbVec3f> profile = new ArrayList<SbVec3f>();
    Random rand = new Random(2);

    private SoSelection scene = null;

    public SceneOverlayPanels() {
    }

    public SoSelection getSceneGraph() {
      if (scene == null)
        scene = getMultiPanelScene();
      return scene;
    }

    void generatedProfile() {
      float x0 = -numPtsProfile / 2;
      for (int i = 0; i < numPtsProfile; i++)
      {
        float x = x0 + i;
        float z = -.5f + (float)(rand.nextDouble());
        profile.add(new SbVec3f(x, 0, z));
      }
    }

    SoMaterial getRandomMat() {
      float R = (float)(rand.nextDouble()) * .6f + .4f;
      float G = (float)(rand.nextDouble()) * .6f + .4f;
      float B = (float)(rand.nextDouble()) * .6f + .4f;
      var rMat = new SoMaterial();
      rMat.diffuseColor.setValue( new SbVec3f(R, G, B));
      rMat.transparency.setValue(0.0f);
      return rMat;
    }

    SoFaceSet getFence(int overlayIndex, int panelIndex, float bottom, float range)
    {

      var strip = new SoFaceSet();
      if (panelIndex < numPanels)
      {

        var vertices = new SoVertexProperty();
        strip.vertexProperty.setValue(vertices);

        SbVec3f p1 = profile.get(panelIndex);
        SbVec3f p2 = profile.get(panelIndex + 1);
        SbVec3f normal = new SbVec3f(  p2.minus(p1));
        vertices.vertex.set1Value(0, p1.getX(), bottom, p1.getZ());
        vertices.vertex.set1Value(1, p2.getX(), bottom, p2.getZ());
        vertices.vertex.set1Value(2, p2.getX(), bottom + range, p2.getZ());
        vertices.vertex.set1Value(3, p1.getX(), bottom + range, p1.getZ());
        vertices.normal.set1Value(0, -normal.getZ(), 0, normal.getX());
        strip.numVertices.setValue(4);

        vertices.normalBinding.setValue(SoVertexProperty.Bindings.PER_FACE);

        var name = "Panel_" + overlayIndex + "_" + panelIndex;
        strip.set(name);
      }
      return strip;
    }

    SoLineSet getFenceLine(int overlayIndex, int panelIndex, float bottom, float range)
    {

      var polyLine = new SoLineSet();
      if (panelIndex < numPanels)
      {
        int nptsPerFace = 10;
        SoVertexProperty vertices = new SoVertexProperty();

        int pindex = 0;
        SbVec3f p1 = profile.get(panelIndex);
        SbVec3f p2 = profile.get(panelIndex + 1);

        vertices.vertex.set1Value(pindex++, p1.getX(), bottom + range / 2.0f, p1.getZ());

        for (int i = 0; i < nptsPerFace - 2; i++)
        {
          float y = bottom + range * (float)(rand.nextDouble());
          float step = 1.0f / (float)(nptsPerFace - 1) * (i + 1);
          SbVec3f p = p1.plus(p2.minus(p1).times(step));
          vertices.vertex.set1Value(pindex++, p.getX(), y, p.getZ());
        }
        vertices.vertex.set1Value(pindex++, p2.getX(), bottom + range / 2, p2.getZ());

        polyLine.vertexProperty.setValue(vertices);

        var name = "Panel_" + overlayIndex + "_" + panelIndex;
        polyLine.set(name);
      }
      return polyLine;
    }


    SoTransform getPanelNamePosition(int panelIndex, float bottom, float range)
    {

      var namePosition = new SoTransform();
      if (panelIndex < numPanels)
      {
        SbVec3f p1 = profile.get(panelIndex);
        SbVec3f p2 = profile.get(panelIndex + 1);

        SbVec3f p = p1.plus(p2.minus(p1).times(0.1f));
        namePosition.translation.setValue(new SbVec3f(p.getX(), bottom + range * 0.9f, p.getZ()));

        namePosition.rotation.setValue(new SbRotation(new SbVec3f(1, 0, 0), p2.minus(p1)));
      }
      return namePosition;
    }


    SoSelection getMultiPanelScene()
    {
      generatedProfile();

      var scene = new SoSelection();

      var sh = new SoShapeHints();
      scene.addChild(sh);
      sh.vertexOrdering.setValue(SoShapeHints.VertexOrderings.COUNTERCLOCKWISE);

      var font = new SoFont();
      font.size.setValue(0.1f);
      scene.addChild(font);

      var overlaySep = new SoSeparator();
      scene.addChild(overlaySep);

      for (int i = 0; i < numPanels; i++)
      {
        var overlay = new SoOverlayGroup();
        overlaySep.addChild(overlay);
      }

      var mat = new SoMaterial();
      mat.diffuseColor.setValue( new SbVec3f(0.8f, 0.7f, 0.1f));

      int overlayIndex = 0;
      for (int i = 0; i < numPanels; i++)
      {
        SoOverlayGroup overlay = (SoOverlayGroup)(overlaySep.getChild(i));

        overlay.addChild(mat);
        overlay.addChild(getFence(overlayIndex, i, -1.0f, 2.0f));
      }
      overlayIndex++;

      mat = new SoMaterial();
      mat.diffuseColor.setValue(new SbVec3f(0, 0.3f, 0.9f));
      var ds = new SoDrawStyle();
      ds.lineWidth.setValue(3.0f);
      for (int i = 0; i < numPanels; i++)
      {
        SoOverlayGroup overlay = (SoOverlayGroup)(overlaySep.getChild(i));
        float bottom = -0.9f;
        float range = 1.8f;
        overlay.addChild(ds);
        overlay.addChild(mat);
        overlay.addChild(getFenceLine(overlayIndex, i, bottom, range));
      }
      overlayIndex++;

      mat = new SoMaterial();
      mat.diffuseColor.setValue(new SbVec3f(0.9f, 0, 0.2f));
      mat.transparency.setValue(0.0f);
      for (int i = 0; i < numPanels; i++)
      {
        SoOverlayGroup overlay = (SoOverlayGroup)(overlaySep.getChild(i));

        overlay.addChild(mat);
        overlay.addChild(getFence(overlayIndex, i, -0.1f, .7f));
      }
      overlayIndex++;

      for (int nl = 0; nl < numPolylines / 2; nl++)
      {
        ds = new SoDrawStyle();
        ds.lineWidth.setValue( nl % 3 > 0 ? 3.0f : 2.0f);
        ds.linePattern.setValue( nl % 2 > 0 ? (short)0xf00f : (short)0xffff);

        mat = getRandomMat();
        for (int i = 0; i < numPanels; i++)
        {
          SoOverlayGroup overlay = (SoOverlayGroup)(overlaySep.getChild(i));
          float bottom = -0.9f + 0.1f * nl;
          float range = 1.2f;
          overlay.addChild(mat);
          overlay.addChild(ds);
          overlay.addChild(getFenceLine(overlayIndex, i, bottom, range));
        }
        overlayIndex++;
      }


      mat = new SoMaterial();
      mat.diffuseColor.setValue( new SbVec3f(0, 0.7f, 0.2f));
      mat.transparency.setValue(0.5f);
      for (int i = 0; i < numPanels; i++)
      {
        SoOverlayGroup overlay = (SoOverlayGroup)(overlaySep.getChild(i));

        overlay.addChild(mat);
        overlay.addChild(getFence(overlayIndex, i, -0.6f, 0.7f));
      }
      overlayIndex++;


      for (int nl = 0; nl < numPolylines / 2; nl++)
      {
        ds = new SoDrawStyle();
        ds.lineWidth.setValue( nl % 3 > 0 ? 3.0f : 2.0f);
        ds.linePattern.setValue( nl % 2 > 0 ? (short)0xf00f : (short)0xffff);

        mat = getRandomMat();
        for (int i = 0; i < numPanels; i++)
        {
          SoOverlayGroup overlay = (SoOverlayGroup)(overlaySep.getChild(i));
          float bottom = -0.6f + 0.1f * nl;
          float range = 1.2f;
          overlay.addChild(mat);
          overlay.addChild(ds);
          overlay.addChild(getFenceLine(overlayIndex, i, bottom, range));
        }
        overlayIndex++;
      }

      mat = new SoMaterial();
      mat.emissiveColor.setValue(new SbVec3f(1, 1, 1));
      for (int i = 0; i < numPanels; i++)
      {
        SoOverlayGroup overlay = (SoOverlayGroup)(overlaySep.getChild(i));
        float bottom = -1.0f;
        float range = 2.0f;

        var textSep = new SoSeparator();
        overlay.addChild(textSep);

        textSep.addChild(mat);
        textSep.addChild(getPanelNamePosition(i, bottom, range));

        var panelName = new SoText3();
        textSep.addChild(panelName);
        var pname = "Panel " + i;
        panelName.string.setValue(pname);
      }
      return scene;
    }
  }
