///////////////////////////////////////////////////////////////////////
//
// This example shows how to generate geometry using tessellation shaders.
//
// The basic shape is an icosahedron and sliders allow to modify tessellation parameters.
// When inner parameter change the number of generated point inside each triangles while
// outer parameter change the number of generated point on each edges.
//
// This example subdivides the icosahedron in such a way that it approaches a perfect sphere.
//
///////////////////////////////////////////////////////////////////////

package inventor.advanced.shaders.tessellationShader;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.io.File;
import java.nio.ByteBuffer;

import javax.swing.BoxLayout;
import javax.swing.JPanel;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbDataType;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.devices.SoBufferObject.AccessModes;
import com.openinventor.inventor.devices.SoCpuBufferObject;
import com.openinventor.inventor.nodes.SoBBox;
import com.openinventor.inventor.nodes.SoBufferedShape;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoShaderParameter1f;
import com.openinventor.inventor.nodes.SoShaderProgram;
import com.openinventor.inventor.nodes.SoTessellationControlShader;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{

//@formatter:off
  // Icosahedron faces
  private static final int[] FACES = {
    2, 1, 0,
    3, 2, 0,
    4, 3, 0,
    5, 4, 0,
    1, 5, 0,
    11, 6,  7,
    11, 7,  8,
    11, 8,  9,
    11, 9,  10,
    11, 10, 6,
    1, 2, 6,
    2, 3, 7,
    3, 4, 8,
    4, 5, 9,
    5, 1, 10,
    2,  7, 6,
    3,  8, 7,
    4,  9, 8,
    5, 10, 9,
    1, 6, 10,
  };

  // Icosahedron vertices
  private static final float[] VERTICES =
  {
    0.000f,  0.000f,  1.000f,
    0.894f,  0.000f,  0.447f,
    0.276f,  0.851f,  0.447f,
    -0.724f,  0.526f,  0.447f,
    -0.724f, -0.526f,  0.447f,
    0.276f, -0.851f,  0.447f,
    0.724f,  0.526f, -0.447f,
    -0.276f,  0.851f, -0.447f,
    -0.894f,  0.000f, -0.447f,
    -0.276f, -0.851f, -0.447f,
    0.724f, -0.526f, -0.447f,
    0.000f,  0.000f, -1.000f
  };
//@formatter:on

  // Tessellation shaders parameters
  private SoShaderParameter1f m_tessLevelInner = null;
  private SoShaderParameter1f m_tessLevelOuter = null;

  private IViewerExaminer myViewer;

  @Override
  public void start()
  {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    SoSeparator sceneGraph = createSceneGraph();

    myViewer.setSceneGraph(sceneGraph);
    myViewer.viewAll();

    // SWING part
    JPanel sliderPanel = createSliderPanel();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(sliderPanel, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  private SoSeparator createSceneGraph()
  {
    SbColor bkgColor = new SbColor(0.5f, 0.5f, 0.5f);
    SoGradientBackground background = new SoGradientBackground();
    background.color0.setValue(bkgColor);
    background.color1.setValue(bkgColor);

    SoBBox bbox = new SoBBox();
    bbox.mode.setValue(SoBBox.Modes.USER_DEFINED);
    bbox.boundingBox.setValue(-1.f, -1.f, -1.f, 1.f, 1.f, 1.f);

    SoSeparator scene = new SoSeparator();
    {
      scene.addChild(background);
      scene.addChild(bbox);

      // add shader program
      scene.addChild(configureShaders());
      // add shape
      scene.addChild(makeIcosahedronFaceSet());
    }

    // Assemble the scene graph
    SoSeparator root = new SoSeparator();
    {
      root.addChild(bbox);
      root.addChild(scene);
    }

    return root;
  }

  private SoBufferedShape makeIcosahedronFaceSet()
  {
    SoBufferedShape pShape = new SoBufferedShape();

    SoCpuBufferObject vertBuffer = new SoCpuBufferObject();
    vertBuffer.setSize(VERTICES.length * Float.SIZE / 8);
    ByteBuffer vertData = vertBuffer.map(AccessModes.SET);

    vertData.rewind();
    for ( float val : VERTICES )
      vertData.putFloat(val);
    vertBuffer.unmap();

    SoCpuBufferObject indBuffer = new SoCpuBufferObject();
    indBuffer.setSize(FACES.length * Integer.SIZE / 8);
    ByteBuffer indData = indBuffer.map(AccessModes.SET);

    indData.rewind();
    for ( int val : FACES )
      indData.putInt(val);
    indBuffer.unmap();

    pShape.shapeType.setValue(SoBufferedShape.Types.TRIANGLES);
    pShape.indexType.setValue(SbDataType.DataTypes.UNSIGNED_INT32);
    pShape.numVertices.set1Value(0, FACES.length);
    pShape.vertexBuffer.setValue(vertBuffer);
    pShape.indexBuffer.setValue(indBuffer);

    return pShape;
  }

  private SoShaderProgram configureShaders()
  {
    String pkgName = this.getClass().getPackage().getName();
    pkgName = pkgName.replace('.', File.separatorChar);
    String filePrefix = SoPreferences.getValue("OIVJHOME") + File.separator + "examples" + File.separator + pkgName
        + File.separator;

    // Initialize and set the shader program
    SoShaderProgram shaderProgram = new SoShaderProgram();
    shaderProgram.setVertexShader(0, filePrefix + "TessShaderVtx.glsl");

    SoTessellationControlShader tessellationControlShader = shaderProgram.setTessellationControlShader(1, filePrefix
        + "TessShaderTessControl.glsl");
    shaderProgram.setTessellationEvaluationShader(2, filePrefix + "TessShaderTessEvaluation.glsl");
    shaderProgram.setGeometryShader(3, filePrefix + "TessShaderGeom.glsl");
    shaderProgram.setFragmentShader(4, filePrefix + "TessShaderFrag.glsl");

    // Tessellation parameters
    // Inner tessellation level controls the number of nested primitives, and
    // the outer tessellation level controls the number of times to subdivide
    // each edge.
    m_tessLevelInner = new SoShaderParameter1f();
    m_tessLevelInner.name.setValue("TessLevelInner");
    m_tessLevelInner.value.setValue(1.f);
    tessellationControlShader.parameter.addShaderParameter(m_tessLevelInner);

    m_tessLevelOuter = new SoShaderParameter1f();
    m_tessLevelOuter.name.setValue("TessLevelOuter");
    m_tessLevelOuter.value.setValue(1.f);
    tessellationControlShader.parameter.addShaderParameter(m_tessLevelOuter);

    // Setup the patch length for tessellation.
    shaderProgram.patchLength.setValue(3);

    return shaderProgram;
  }

  private JPanel createSliderPanel()
  {
    SliderPanel levelInnerSlider = new SliderPanel(1.f, 64.f, 1.f, 3);
    levelInnerSlider.addInfoText("Tessellation level inner : ");
    levelInnerSlider.setSliderSize(new Dimension(200, 20));
    levelInnerSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_tessLevelInner.value.setValue(value);
      }
    });

    SliderPanel levelOuterSlider = new SliderPanel(1.f, 64.f, 1.f, 3);
    levelOuterSlider.addInfoText("Tessellation level outer : ");
    levelOuterSlider.setSliderSize(new Dimension(200, 20));
    levelOuterSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_tessLevelOuter.value.setValue(value);
      }
    });

    JPanel sliderPanel = new JPanel();
    sliderPanel.setLayout(new BoxLayout(sliderPanel, BoxLayout.Y_AXIS));
    sliderPanel.add(levelInnerSlider);
    sliderPanel.add(levelOuterSlider);

    return sliderPanel;
  }

  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("Tessellation shader on Icosahedron");
  }

}
