/*-------------------------------------------------------------
 *  This is an example from the Inventor Mentor,
 *  volume 1, chapter 15, example 3.
 *
 *  Manipulator attachment example.
 *
 *  The scene graph has a cone, a cube and a sphere.
 *  When the user picks on the cone, the cone's
 *  associated transform is replaced by an SoTransformBoxManip.
 *  When the user picks the sphere, the
 *  sphere's associated transform is replaced by an
 *  SoHandleBoxManip.  Picking the cube causes an
 *  SoTrackballManip to replace the cube's transform.
 *
 *  Manipulator callbacks are used to change
 *  the color of the object being manipulated.
 *
 *  Note that for illustration purposes, the
 *  cube and cone already have transform nodes
 *  associated with them; the sphere does not. In all cases,
 *  the routine createTransformPath() is used to find the
 *  transform node that affects the picked object.
 *-----------------------------------------------------------*/

package inventor.mentor.attachManip;

import java.awt.BorderLayout;
import java.awt.Component;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.draggers.SoDragger;
import com.openinventor.inventor.manips.SoHandleBoxManip;
import com.openinventor.inventor.manips.SoTrackballManip;
import com.openinventor.inventor.manips.SoTransformBoxManip;
import com.openinventor.inventor.misc.callbacks.SoDraggerCB;
import com.openinventor.inventor.misc.callbacks.SoSelectionPathCB;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example {
  SoHandleBoxManip myHandleBox = new SoHandleBoxManip();
  SoTrackballManip myTrackball = new SoTrackballManip();
  SoTransformBoxManip myTransformBox = new SoTransformBoxManip();

  SoPath handleBoxPath = null;
  SoPath trackballPath = null;
  SoPath transformBoxPath = null;

  SoMaterial cubeMat = new SoMaterial();
  SoMaterial sphereMat = new SoMaterial();
  SoMaterial coneMat = new SoMaterial();
  private IViewerExaminer myViewer;

  public static void main(String[] args) {
    Main example = new Main();
    example.demoMain("Attaching Manipulators");
  }

  @Override
  public void start() {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    // create and set up the selection node
    SoSelection selectionRoot = new SoSelection();

    // Create a cone
    SoSeparator coneRoot = new SoSeparator();

    SoTransform coneTrans = new SoTransform();
    coneTrans.translation.setValue(3, 0, 0);
    coneMat.diffuseColor.setValue(0.8f, 0.8f, 0.8f);
    {
      coneRoot.addChild(coneTrans);
      coneRoot.addChild(coneMat);
      coneRoot.addChild(new SoCone());
    }

    // Create a cube with its own transform.
    SoSeparator cubeRoot = new SoSeparator();
    SoTransform cubeXform = new SoTransform();
    cubeXform.translation.setValue(-3.0f, 0.0f, 0.0f);
    cubeMat.diffuseColor.setValue(0.8f, 0.8f, 0.8f);
    {
      cubeRoot.addChild(cubeXform);
      cubeRoot.addChild(cubeMat);
      cubeRoot.addChild(new SoCube());
    }

    // add a sphere node without a transform
    // (one will be added when we attach the manipulator)
    SoSeparator sphereRoot = new SoSeparator();
    sphereMat.diffuseColor.setValue(0.8f, 0.8f, 0.8f);
    {
      sphereRoot.addChild(sphereMat);
      sphereRoot.addChild(new SoSphere());
    }


    // create the scene graph
    SoSeparator root = new SoSeparator();
    { // assemble scene graph
      root.addChild(coneRoot);
      root.addChild(cubeRoot);
      root.addChild(sphereRoot);
    }

    // Get the draggers and add callbacks to them. Note
    // that you don't put callbacks on manipulators. You put
    // them on the draggers which handle events for them.
    SoDragger myDragger = myTrackball.getDragger();
    myDragger.addStartCallback(new DragStartCallback(cubeMat), null);
    myDragger.addFinishCallback(new DragFinishCallback(cubeMat), null);

    myDragger = myHandleBox.getDragger();
    myDragger.addStartCallback(new DragStartCallback(sphereMat), null);
    myDragger.addFinishCallback(new DragFinishCallback(sphereMat), null);

    myDragger = myTransformBox.getDragger();
    myDragger.addStartCallback(new DragStartCallback(coneMat), null);
    myDragger.addFinishCallback(new DragFinishCallback(coneMat), null);

    { // Assemble scene graph
      selectionRoot.addSelectionCallback(new SelectionCallback(), null);
      selectionRoot.addDeselectionCallback(new DeselectionCallback(), null);
      selectionRoot.addChild(root);
    }

    myViewer.setSceneGraph(selectionRoot);
    myViewer.viewAll();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  // Is this node of a type that is influenced by transforms?
  boolean isTransformable(SoNode myNode) {
    if (myNode instanceof SoGroup
        || myNode instanceof SoShape
        || myNode instanceof SoCamera
        || myNode instanceof SoLight)
      return true;
    else
      return false;
  }


  SoPath createTransformPath(SoPath inputPath) {
    int pathLength = inputPath.regular.getLength();
    if (pathLength < 2) // Won't be able to get parent of tail
      return null;

    SoNode tail = inputPath.regular.getTail();
/*
    // CASE 1: The tail is a node kit.
    // Nodekits have built in policy for creating parts.
    // The kit copies inputPath, then extends it past the
    // kit all the way down to the transform. It creates the
    // transform if necessary.
    if (tail instanceof SoBaseKit) {
      SoBaseKit kit = (SoBaseKit)tail;
      return kit.createPathToPart("transform", true, inputPath);
    }
*/
    SoTransform editXf = null;
    SoGroup parent;
    // CASE 2: The tail is not a group.
    boolean isTailGroup;
    isTailGroup = tail instanceof SoGroup;
    if (!isTailGroup) {
      // 'parent' is node above tail. Search under parent right
      // to left for a transform. If we find a 'movable' node
      // insert a transform just left of tail.
      parent = (SoGroup)inputPath.regular.getNode(pathLength - 2);
      int tailIndx = parent.findChild(tail);

      for (int i = tailIndx; (i >= 0) && (editXf == null);i--){
        SoNode myNode = parent.getChild(i);
        if (myNode instanceof SoTransform)
          editXf = (SoTransform)myNode;
        else if (i != tailIndx && (isTransformable(myNode)))
          break;
      }
      if (editXf == null) {
        editXf = new SoTransform();
        parent.insertChild(editXf, tailIndx);
      }
    }
    // CASE 3: The tail is a group.
    else {
      // Search the children from left to right for transform
      // nodes. Stop the search if we come to a movable node.
      // and insert a transform before it.
      int i;

      parent = (SoGroup)tail;
      for (i = 0; (i < parent.getNumChildren()) && (editXf == null); i++) {
        SoNode myNode = parent.getChild(i);
        if (myNode instanceof SoTransform)
          editXf = (SoTransform )myNode;
        else if (isTransformable(myNode))
          break;
      }
      if (editXf == null) {
        editXf = new SoTransform();
        parent.insertChild(editXf, i);
      }
    }

    // Create 'pathToXform.' Copy inputPath, then make last
    // node be editXf.
    SoPath pathToXform = null;
    pathToXform = inputPath.regular.copy(0);

    if (!isTailGroup) // pop off the last entry.
      pathToXform.regular.pop();
      // add editXf to the end
    int xfIndex = parent.findChild(editXf);
    pathToXform.regular.append(xfIndex);

    return(pathToXform);
  }

  class SelectionCallback extends SoSelectionPathCB {
    @Override
    public void invoke(SoPath path) {
      // Attach the manipulator.
      // Use the convenience routine to get a path to
      // the transform that effects the selected object.
      SoPath xformPath = createTransformPath(path);
      if (xformPath == null)
        return;

      // Attach the handle box to the sphere,
      // the trackball to the cube
      // or the transformBox to the wrapperKit
      if (path.regular.getTail() instanceof SoSphere) {
        handleBoxPath = xformPath;
        myHandleBox.replaceNode(xformPath);
      }
      else if (path.regular.getTail() instanceof SoCube) {
        trackballPath = xformPath;
        myTrackball.replaceNode(xformPath);
      }
      else if (path.regular.getTail() instanceof SoCone) {
        transformBoxPath = xformPath;
        myTransformBox.replaceNode(xformPath);
      }
    }
  }

  class DeselectionCallback extends SoSelectionPathCB {
    @Override
    public void invoke(SoPath path) {
      if (path.regular.getTail() instanceof SoSphere)
        myHandleBox.replaceManip(handleBoxPath, new SoTransform());
      else if (path.regular.getTail() instanceof SoCube)
        myTrackball.replaceManip(trackballPath, new SoTransform());
      else if (path.regular.getTail() instanceof SoCone)
        myTransformBox.replaceManip(transformBoxPath, new SoTransform());
    }
  }

  // This is called when a manipulator is
  // about to begin manipulation.
  class DragStartCallback extends SoDraggerCB {
    SoMaterial m_material;

    public DragStartCallback(SoMaterial mat) {
      m_material = mat;
    }

    @Override
    public void invoke(SoDragger d) {
      m_material.diffuseColor.setValue(new SbColor(1.0f, 0.2f, 0.2f));
    }
  }

// This is called when a manipulator is
// done manipulating.
  class DragFinishCallback extends SoDraggerCB {
    SoMaterial m_material;

    public DragFinishCallback(SoMaterial mat) {
      m_material = mat;
    }

    @Override
    public void invoke(SoDragger d) {
      m_material.diffuseColor.setValue(new SbColor(0.8f, 0.8f, 0.8f));
    }
  }
}
