/*----------------------------------------------------------------
 *  This is an example from the Inventor Mentor
 *  chapter 14, example 2.
 *
 *  Use nodekits to create a scene with a sphere into an
 *  SoShapeKit.  Then, add a material editor for the sphere and
 *  a light editor on the light.
 *
 *  Once the scene is created, a material editor is attached to
 *  the appearanceKit's 'material' part, and a directional light editor
 *  is attached to the directional light.
 *----------------------------------------------------------------*/

package inventor.mentor.editors;

import java.awt.BorderLayout;
import java.awt.Component;

import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.actions.SoSearchAction;
import com.openinventor.inventor.nodekits.SoAppearanceKit;
import com.openinventor.inventor.nodekits.SoShapeKit;
import com.openinventor.inventor.nodes.SoDirectionalLight;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoSphere;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;

import util.Example;
import util.ViewerComponentsFactory;
import util.editors.DirectionalLightEditor;
import util.editors.MaterialEditor;

public class Main extends Example {

  private IRenderAreaInteractive myRenderArea;

  MaterialEditor mtlEditor;
  DirectionalLightEditor ltEditor;

  public static void main(String[] args) {
    Main example = new Main();
    example.demoMain("Main Window: Sphere In A Shape Kit");
  }

  @Override
  public void start() {
    myRenderArea = ViewerComponentsFactory.createRenderAreaInteractive();

    // Appearance kit
    SoAppearanceKit myAppearance = new SoAppearanceKit();
    // Give the sphere a good starting color
    myAppearance.set("material { diffuseColor .8 .3 .1 }");

    // OBJECT!! Create an SoShapeKit with a sphere
    SoShapeKit mySphere = new SoShapeKit();
    mySphere.setPart("shape", new SoSphere());

    // MATERIAL EDITOR!!  Attach it to myAppearance's material node.
    mtlEditor = new MaterialEditor();
    SoMaterial mtl = (SoMaterial)myAppearance.getPart("material", true);
    mtlEditor.setTitle("Material of Sphere");
    mtlEditor.setVisible(true);
    mtlEditor.attach(mtl);

    // DIRECTIONAL LIGHT EDITOR!!
    // Attach it to the SoDirectionalLight node
    SoSearchAction searchAction = new SoSearchAction();
    searchAction.setNodeClass(SoDirectionalLight.class);
    searchAction.apply(myRenderArea.getSceneInteractor());
    SoPath path = searchAction.getPath();
    if ( path == null )
    {
      System.err.println("Headlight not found");
      return;
    }

    ltEditor = new DirectionalLightEditor();
    ltEditor.setTitle("Lighting of Sphere");
    ltEditor.setVisible(true);
    ltEditor.attach(path);

    SoSeparator root = new SoSeparator();
    { // Assemble scene graph
      root.addChild(myAppearance);
      root.addChild(mySphere);
    }

    final Component canvas = myRenderArea.getComponent();

    myRenderArea.setSceneGraph(root);
    myRenderArea.viewAll(new SbViewportRegion(canvas.getSize()));

    canvas.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(canvas);
  }

  @Override
  public void stop()
  {
    mtlEditor.dispose();
    if ( ltEditor != null )
      ltEditor.dispose();
    myRenderArea.dispose();
  }
}
