/*-------------------------------------------------------------
 *  This is an example from The Inventor Mentor,
 *  chapter 10, example 7.
 *
 *  This example demonstrates the use of the pick filter
 *  callback to pick through manipulators.
 *
 *  The scene graph has several objects. Clicking the left
 *  mouse on an object selects it and adds a manipulator to
 *  it. Clicking again deselects it and removes the manipulator.
 *  In this case, the pick filter is needed to deselect the
 *  object rather than select the manipulator.
 *------------------------------------------------------------*/

package inventor.mentor.pickFilterManip;

import java.awt.BorderLayout;
import java.awt.Component;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.SoPickedPoint;
import com.openinventor.inventor.manips.SoHandleBoxManip;
import com.openinventor.inventor.manips.SoTransformManip;
import com.openinventor.inventor.misc.callbacks.SoSelectionPathCB;
import com.openinventor.inventor.misc.callbacks.SoSelectionPickCB;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example {

  private IViewerExaminer myViewer;

  public static void main(String[] args) {
    Main example = new Main();
    example.demoMain("Select Through Manips");
  }

  @Override
  public void start() {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    // Create a scene graph. Use the toggle selection policy.
    SoSelection sel = new SoSelection();
    sel.policy.setValue(SoSelection.Policies.TOGGLE);
    sel.addChild(buildScene());

    // Selection callbacks
    sel.addSelectionCallback(new SelectionCB());
    sel.addDeselectionCallback(new DeselectionCB());
    sel.setPickFilterCallback(new PickFilterCB());

    myViewer.setSceneGraph(sel);
    myViewer.viewAll();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  // Create a sample scene graph
  SoNode myText(String str, int i, SbColor color) {
    SoSeparator sep = new SoSeparator();
    SoBaseColor col = new SoBaseColor();
    SoTransform xf = new SoTransform();
    SoText3 text = new SoText3();

    col.rgb.setValue(color);
    xf.translation.setValue(6.0f * i, 0.0f, 0.0f);
    text.string.setValue(str);
    text.parts.setValue(SoText3.PartType.FRONT.getValue() | SoText3.PartType.SIDES.getValue());
    text.justification.setValue(SoText3.Justifications.CENTER);

    {
      sep.addChild(col);
      sep.addChild(xf);
      sep.addChild(text);
    }
    return sep;
  }

  SoNode buildScene() {
    SoSeparator scene = new SoSeparator();
    SoFont font = new SoFont();

    font.size.setValue(10);

    {
      scene.addChild(font);
      scene.addChild(myText("O", 0, new SbColor(0.0f, 0.0f, 1.0f)));
      scene.addChild(myText("p", 1, new SbColor(0.0f, 1.0f, 0.0f)));
      scene.addChild(myText("e", 2, new SbColor(0.0f, 1.0f, 1.0f)));
      scene.addChild(myText("n", 3, new SbColor(1.0f, 0.0f, 0.0f)));
      // Open Inventor is two words!
      scene.addChild(myText("I", 5, new SbColor(1.0f, 0.0f, 1.0f)));
      scene.addChild(myText("n", 6, new SbColor(1.0f, 1.0f, 0.0f)));
      scene.addChild(myText("v", 7, new SbColor(1.0f, 1.0f, 1.0f)));
      scene.addChild(myText("e", 8, new SbColor(0.0f, 0.0f, 1.0f)));
      scene.addChild(myText("n", 9, new SbColor(0.0f, 1.0f, 0.0f)));
      scene.addChild(myText("t", 10, new SbColor(0.0f, 1.0f, 1.0f)));
      scene.addChild(myText("o", 11, new SbColor(1.0f, 0.0f, 0.0f)));
      scene.addChild(myText("r", 12, new SbColor(1.0f, 0.0f, 1.0f)));
    }

    return scene;
  }

  class SelectionCB extends SoSelectionPathCB {
    @Override
    public void invoke(SoPath path) {
      if (path.regular.getLength() < 2)
        return;

      // Find the transform affecting this object
      SoPath xfPath = findXform(path);

      // Replace the transform with a manipulator
      SoHandleBoxManip manip = new SoHandleBoxManip();
      manip.replaceNode(xfPath);
    }

    // Returns path to xform left of the input path tail.
    // Inserts the xform if none found. In this example,
    // assume that the xform is always the node preceeding
    // the selected shape.
    SoPath findXform(SoPath p) {
      SoPath returnPath;

      // Copy the input path up to tail's parent.
      returnPath = p.regular.copy(0, p.regular.getLength() - 1);

      // Get the parent of the selected shape
      SoGroup g = (SoGroup) p.regular.getNodeFromTail(1);
      int tailNodeIndex = p.regular.getIndexFromTail(0);

      // Check if there is already a transform node
      if (tailNodeIndex > 0) {
        SoNode n = g.getChild(tailNodeIndex - 1);
        if (n instanceof SoTransform) {
          // Append to returnPath and return it.
          returnPath.regular.append(n);
          return returnPath;
        }
      }

      // Otherwise, add a transform node.
      SoTransform xf = new SoTransform();
      g.insertChild(xf, tailNodeIndex); // right before the tail
      // Append to returnPath and return it.
      returnPath.regular.append(xf);
      return returnPath;
    }
  }

  class DeselectionCB extends SoSelectionPathCB{
    @Override
    public void invoke(SoPath path) {
      if (path.regular.getLength() < 2)
        return;

      // Find the manipulator affecting this object
      SoPath manipPath = findManip(path);

      // Replace the manipulator with a transform
      SoTransformManip manip = (SoTransformManip)manipPath.regular.getTail();
      manip.replaceManip(manipPath, new SoTransform());
    }

    // Returns the manip affecting this path. In this example,
    // the manip is always preceeding the selected shape.
    SoPath findManip(SoPath p) {
      SoPath returnPath;

      // Copy the input path up to tail's parent.
      returnPath = p.regular.copy(0, p.regular.getLength() - 1);

      // Get the index of the last node in the path.
      int tailNodeIndex = p.regular.getIndexFromTail(0);

      // Append the left sibling of the tail to the returnPath
      returnPath.regular.append(tailNodeIndex - 1);
      return returnPath;
    }
  }

  //////////////////////////////////////////////////////////////
  // CODE FOR The Inventor Mentor STARTS HERE

  class PickFilterCB extends SoSelectionPickCB {
    @Override
    public SoPath invoke(SoPickedPoint pick) {
      SoPath filteredPath = null;

      // See if the picked object is a manipulator.
      // If so, change the path so it points to the object the manip
      // is attached to.
      SoPath p = pick.getPath();
      SoNode n = p.regular.getTail();
      if (n instanceof SoTransformManip) {
        // Manip picked! We know the manip is attached
        // to its next sibling. Set up and return that path.
        int manipIndex = p.regular.getIndex(p.regular.getLength() - 1);
        filteredPath = p.regular.copy(0, p.regular.getLength() - 1);
        filteredPath.regular.append(manipIndex + 1); // get next sibling
      } else
        filteredPath = p;

      return filteredPath;
    }
  }

  // CODE FOR The Inventor Mentor ENDS HERE
  ///////////////////////////////////////////////////////////////
}
