/*-----------------------------------------------------------
 *  This is an example from the Inventor Mentor,
 *  chapter 9, example 2.
 *
 *  Using the offscreen renderer to generate a texture map.
 *  Generate simple scene and grab the image to use as
 *  a texture map.
 *----------------------------------------------------------*/

package inventor.mentor.texture;

import java.awt.BorderLayout;
import java.awt.Component;

import com.openinventor.inventor.SbVec2s;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbColorRGBA;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.SoOffscreenRenderArea;
import com.openinventor.inventor.devices.SoCpuBufferObject;
import com.openinventor.inventor.fields.SoSFImage;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{

  private IViewerExaminer myViewer;

  public static void main(String[] args) {
    Main example = new Main();
    example.demoMain("Offscreen Rendered Texture");
  }

  @Override
  public void start() {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    // Make a scene from reading in a file
    SoSeparator texRoot = new SoSeparator();

    SoComplexity nodeComplexity = new SoComplexity();
    nodeComplexity.textureQuality.setValue(0.0f);

    SoInput in = new SoInput();
    in.openFile("$OIVJHOME/data/models/jumpyMan.iv");
    SoNode result = SoDB.readNode(in);

    SoPerspectiveCamera myCamera = new SoPerspectiveCamera();

    SoRotationXYZ rot = new SoRotationXYZ();
    rot.axis.setValue(SoRotationXYZ.AxisType.X);
    rot.angle.setValue((float)java.lang.Math.PI/2.f);
    myCamera.position.setValue(new SbVec3f( -0.2f, -0.2f, 2.0f));
    myCamera.scaleHeight(0.4f);
    {
      texRoot.addChild(nodeComplexity);
      texRoot.addChild(myCamera);
      texRoot.addChild(new SoDirectionalLight());
      texRoot.addChild(rot);
      texRoot.addChild(result);
    }

    // Generate the texture map
    SoTexture2 texture = new SoTexture2();
    if (generateTextureMap(texRoot, texture, (short)64, (short)64))
      System.out.println("Successfully generated texture map\n");
    else
      System.out.println("Could not generate texture map\n");

    // Make a scene with a cube and apply the texture to it
    SoSeparator root = new SoSeparator();

    { // Assemble scene graph
      root.addChild(texture);
      root.addChild(new SoCube());
    }

    myViewer.setSceneGraph(root);
    myViewer.viewAll();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  private boolean generateTextureMap(final SoNode root, SoTexture2 texture,
                         short textureWidth, short textureHeight)
  {
    SbViewportRegion myViewport = new SbViewportRegion(textureWidth, textureHeight);

    SoOffscreenRenderArea myRenderer = new SoOffscreenRenderArea();
    myRenderer.setSceneGraph(root);
    myRenderer.setViewportRegion(myViewport);
    myRenderer.setClearColor(new SbColorRGBA(0.3f, 0.3f, 0.3f, 1.0f));

    SoCpuBufferObject buffer = new SoCpuBufferObject();
    myRenderer.renderToBuffer(buffer);

    texture.image.setValue(new SbVec2s(textureWidth, textureHeight), 3,
      SoSFImage.DataTypes.UNSIGNED_BYTE, buffer);

    return true;
  }
}
