package inventor.sample.nurbsProperties;

import java.awt.BorderLayout;
import java.awt.Component;

import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.actions.SoSearchAction;
import com.openinventor.inventor.events.SoKeyboardEvent;
import com.openinventor.inventor.events.SoKeyboardEvent.Keys;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.SoEventCallback;
import com.openinventor.inventor.nodes.SoNurbsProperty;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  public class ProcessKeyEvents extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback cb)
    {
      int val;
      SoKeyboardEvent keyEvent = (SoKeyboardEvent) cb.getEvent();
      Keys key = keyEvent.getKey();

      if ( SoKeyboardEvent.isKeyPressEvent(cb.getEvent(), key) )
        return;

      switch ( key )
      {
      case B : // Boundaries
        // Show/hide Boundaries
        val = nurbsProperty.drawStyle.getValue();

        if ( (val & SoNurbsProperty.DrawStyles.BOUNDARIES_AND_TRIM_CURVES.getValue()) > 0 )
        {
          nurbsProperty.drawStyle.setValue(val & ~SoNurbsProperty.DrawStyles.BOUNDARIES_AND_TRIM_CURVES.getValue());
          System.out.println("hide Boundaries ");
        }
        else
        {
          nurbsProperty.drawStyle.setValue(val | SoNurbsProperty.DrawStyles.BOUNDARIES_AND_TRIM_CURVES.getValue());
          System.out.println("show Boundaries ");
        }
        break;
      case I : // IsoParametic
        // Show/hide Iso Parametric Curves
        val = nurbsProperty.drawStyle.getValue();

        if ( (val & SoNurbsProperty.DrawStyles.CENTRAL_ISO_PARAM_CURVES.getValue()) > 0 )
        {
          System.out.println("hide Iso Parametric Curves ");
          nurbsProperty.drawStyle.setValue(val & ~SoNurbsProperty.DrawStyles.CENTRAL_ISO_PARAM_CURVES.getValue());
        }
        else
        {
          System.out.println("show Iso Parametric Curves ");
          nurbsProperty.drawStyle.setValue(val | SoNurbsProperty.DrawStyles.CENTRAL_ISO_PARAM_CURVES.getValue());
        }
        break;
      case N : // Normal
        // Show/hide Nurbs Surface
        val = nurbsProperty.drawStyle.getValue();
        if ( (val & SoNurbsProperty.DrawStyles.NORMAL.getValue()) > 0 )
        {
          System.out.println("hide Nurbs Surface ");
          nurbsProperty.drawStyle.setValue(val & ~SoNurbsProperty.DrawStyles.NORMAL.getValue());
        }
        else
        {
          System.out.println("show Nurbs Surface ");
          nurbsProperty.drawStyle.setValue(val | SoNurbsProperty.DrawStyles.NORMAL.getValue());
        }
        break;
      case M : // More
        // Increase number of sample points on trimming curves
        val = nurbsProperty.numSamplePoints.getValue();
        nurbsProperty.numSamplePoints.setValue(val + 1);
        System.out.println("increase number of sample points on trimming curves: " + (val + 1));
        break;
      case L : // Less
        // Decrease number of sample points on trimming curves
        val = nurbsProperty.numSamplePoints.getValue();
        // -1 means to use the SoComplexity setting
        if ( val > -1 )
          nurbsProperty.numSamplePoints.setValue(val - 1);
        System.out.println("decrease number of sample points on trimming curves: " + (val - 1));
        break;
      default:
        // do nothing
        break;
      }
      cb.setHandled();
    }
  }

  private IViewerExaminer myViewer;

  SoNurbsProperty nurbsProperty;
  SoSeparator root;

  @Override
  public void start()
  {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    root = readSceneGraph("$OIVJHOME/data/pick2.iv");
    {
      // look for the first NurbsProperty in the scene graph.
      SoSearchAction search = new SoSearchAction();
      search.setNodeClass(SoNurbsProperty.class);
      search.setInterest(SoSearchAction.Interests.FIRST);
      search.apply(root);

      if ( search.getPath() == null )
      {
        // add one if does not exist.
        System.out.println(" no NurbsProperty node found, adding one");
        nurbsProperty = new SoNurbsProperty();
        nurbsProperty.color.setValue(1, 1, 1);
        nurbsProperty.isoParamCurvesPattern.setValue((short) 0xF0F0);
        root.insertChild(nurbsProperty, 0);
      }
      else
      { // use the first found
        nurbsProperty = (SoNurbsProperty) search.getPath().regular.getTail();
        // Set override to true to see the effect of our changes on the whole
        // scene graph.
        nurbsProperty.setOverride(true);
      }
    }

    // add a callback for managing user input
    SoEventCallback eventCB = new SoEventCallback();
    root.addChild(eventCB);

    System.out.println("displaying the scene graph...");
    // Open a viewer for the simplified scene:
    myViewer.setSceneGraph(root);
    myViewer.viewAll();

    System.out.println("\n Enter Esc to switch between viewing and picking mode ");
    System.out.println(" Enter: \n   B to toggle display of Nurbs and Trimming curves boundaries");
    System.out.println("   I to toggle display of Central Iso Parametric Curves");
    System.out.println("   N to toggle display of Nurbs Surface ");
    System.out.println("   M to increase the number of sample points on trimming curves");
    System.out.println("   L to decrease the number of sample points on trimming curves");

    eventCB.addEventCallback(SoKeyboardEvent.class, new ProcessKeyEvents());

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  private static SoSeparator readSceneGraph(String fileName)
  {
    SoInput input = new SoInput();

    if ( !input.openFile(fileName) )
    {
      System.err.println("Cannot open file " + fileName);
      return null;
    }

    SoSeparator node = SoDB.readAll(input);
    if ( node == null )
    {
      System.err.println("Problem reading file");
      input.closeFile();
      return null;
    }

    input.closeFile();
    return node;
  }

  public static void main(String argv[])
  {
    Main example = new Main();
    example.demoMain("NURBS properties viewer");
  }
}
