package launcher;

import java.awt.*;
import java.awt.event.*;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStreamReader;
import java.lang.reflect.Method;
import java.util.Vector;

import javax.swing.*;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.*;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeCellRenderer;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import com.openinventor.inventor.SoPreferences;

import util.Example;

public class DemosLauncher extends JFrame
{
  protected static final String ROOT_DIR =
      SoPreferences.getValue("OIVJHOME") + File.separator + "examples";

  // enabling multiple demos may cause Inventor.start & Inventor.stop methods
  // conflicts
  private static final boolean ENABLE_MULTIPLE_DEMOS = false;

  private static final Color BK_COLOR = new Color(242, 242, 242);
  private static final Color BK_SELECTION_COLOR = new Color(206, 214, 217);
  private static final Color SLIDER_TICKS_COLOR = new Color(56, 90, 137);

  private static final ImageIcon PREV_ICON = new ImageIcon(ROOT_DIR + "/launcher/images/previous.png");
  private static final ImageIcon PLAY_ICON = new ImageIcon(ROOT_DIR + "/launcher/images/play.png");
  private static final ImageIcon NEXT_ICON = new ImageIcon(ROOT_DIR + "/launcher/images/next.png");
  private static final ImageIcon STOP_ICON = new ImageIcon(ROOT_DIR + "/launcher/images/stop.png");
  private static final ImageIcon EMPTY_ICON = new ImageIcon(ROOT_DIR + "/launcher/images/empty.png");
  private static final ImageIcon LOGO_ICON =
      new ImageIcon(ROOT_DIR + "/launcher/images/DemosLauncher_SplashScreen.png");

  private static final int DELAY_INIT = 5;
  private static final int DELAY_MIN = 0;
  private static final int DELAY_MAX = 30;

  private String m_currentDirPath;
  private DefaultMutableTreeNode m_currentNode;

  private JTree m_dirTree;
  private DefaultTreeModel m_model;
  private JScrollPane m_spList;
  private JList<String> m_examplesList;
  private JButton m_nextButton;
  private JButton m_stopButton;
  private JButton m_launchButton;
  private JButton m_previousButton;
  private JCheckBox m_modeBox;
  private JSlider m_timeSlider;
  private JPanel m_tools_jp;

  private boolean m_singleMode;
  private boolean m_timerMode;
  private int m_timerDelay;
  private Timer timer;
  private Vector<Demo> m_runningExamples;
  private Point m_exampleFrameLocation;

  public DemosLauncher()
  {
    super();

    m_runningExamples = new Vector<Demo>();

    m_currentDirPath = ROOT_DIR;
    m_singleMode = true;
    m_timerMode = false;
    m_timerDelay = DELAY_INIT * 1000;
    m_exampleFrameLocation = new Point(0, 0);

    try
    {
      jbInit();
    }
    catch (Exception e)
    {
      e.printStackTrace();
    }
  }

  public static void main(String[] args)
  {
    DemosLauncher d = new DemosLauncher();
    d.setVisible(true);
  }

  private DefaultMutableTreeNode getTreeNode(TreePath path)
  {
    return (DefaultMutableTreeNode) (path.getLastPathComponent());
  }

  private DirTreeNode getDirTreeNode(DefaultMutableTreeNode node)
  {
    if ( node == null )
      return null;

    Object obj = node.getUserObject();
    if ( obj instanceof DirTreeNode )
      return (DirTreeNode) obj;

    return null;
  }

  private void launchDemo()
  {
    if ( (m_singleMode == true || m_timerMode == true) && m_runningExamples.size() == 1 )
      // close active demo
      closeDemo(m_runningExamples.firstElement());

    // start demo
    DirTreeNode current_tree_node = getDirTreeNode(m_currentNode);
    String selected_classname = current_tree_node.getCurrentExampleClass();
    Demo new_demo = new Demo(m_currentDirPath, selected_classname, current_tree_node.getCurrentExampleName());

    System.out.println("===> Launch " + m_currentDirPath + current_tree_node.getCurrentExampleName());

    new_demo.startDemo();
    m_runningExamples.add(new_demo);

    // update buttons
    if ( !m_timerMode )
    {
      m_previousButton.setEnabled(true);
      m_nextButton.setEnabled(true);
    }
    m_stopButton.setEnabled(true);
    m_launchButton.setEnabled(false);
  }

  private void closeDemo(Demo d)
  {
    // close demo
    d.stopDemo();
    m_runningExamples.remove(d);

    // update buttons
    if ( m_runningExamples.isEmpty() )
    {
      m_previousButton.setEnabled(false);
      m_nextButton.setEnabled(false);
      if ( !m_timerMode )
      {
        m_stopButton.setEnabled(false);
        m_launchButton.setEnabled(true);
      }
    }
  }

  private void closeAllDemos()
  {
    // close all running demos
    for ( int i = 0; i < m_runningExamples.size(); i++ )
      m_runningExamples.elementAt(i).stopDemo();
    m_runningExamples.removeAllElements();

    // update buttons
    m_previousButton.setEnabled(false);
    m_nextButton.setEnabled(false);
    m_stopButton.setEnabled(false);
    m_launchButton.setEnabled(true);
  }

  private void openPreviousDemo()
  {
    DirTreeNode current_dtn = getDirTreeNode(m_currentNode);
    int index_current = current_dtn.getCurrentExampleIndex();

    if ( current_dtn.getExamplesCount() != 0 && index_current > 0 )
    {
      // set index on the previous example
      current_dtn.setCurrentExampleIndex(index_current - 1);
      // update frame components
      m_dirTree.setSelectionPath(new TreePath(m_currentNode.getPath()));
      m_examplesList.setSelectedIndex(index_current - 1);
    }
    else if ( !setPreviousExamplesList() )
    { // search another examples list
      m_previousButton.setEnabled(false);
      return;
    }

    launchDemo();
  }

  private void openNextDemo()
  {
    DirTreeNode current_dtn = getDirTreeNode(m_currentNode);
    int index_current = current_dtn.getCurrentExampleIndex();

    if ( current_dtn.getExamplesCount() != 0 && index_current < current_dtn.getExamplesCount() - 1 )
    {
      // set index on the next example
      current_dtn.setCurrentExampleIndex(index_current + 1);
      // update frame components
      m_dirTree.setSelectionPath(new TreePath(m_currentNode.getPath()));
      m_examplesList.setSelectedIndex(index_current + 1);
    }
    else if ( !setNextExamplesList() )
    { // search another examples list
      // restart search at the beginning of JTree
      m_currentNode = (DefaultMutableTreeNode) m_dirTree.getPathForRow(0).getLastPathComponent();
      setNextExamplesList();
    }

    launchDemo();
  }

  private boolean setPreviousExamplesList()
  {
    DefaultMutableTreeNode previous_node;
    DefaultMutableTreeNode old_node = m_currentNode;

    while ( (previous_node = m_currentNode.getPreviousNode()) != null )
    {
      Object node_object = previous_node.getUserObject();

      if ( node_object instanceof Boolean )
      {
        // expand node
        DefaultMutableTreeNode parent = (DefaultMutableTreeNode) previous_node.getParent();
        if ( !getDirTreeNode(parent).expand(parent) )
          m_currentNode = parent;
      }
      else
      {
        DirTreeNode previous_dtn = getDirTreeNode(previous_node);
        if ( previous_dtn.getExamplesCount() != 0 )
        {
          // update frame components
          m_dirTree.setSelectionPath(new TreePath(previous_node.getPath()));
          m_examplesList.setListData(previous_dtn.getExamplesNameList());
          m_examplesList.setSelectedIndex(previous_dtn.getExamplesCount() - 1);
          // update data
          m_currentDirPath = previous_dtn.getFile().getPath() + File.separator;
          m_currentNode = previous_node;
          previous_dtn.setCurrentExampleIndex(previous_dtn.getExamplesCount() - 1);
          return true;
        }
        else
          m_currentNode = previous_node;
      }
    }
    m_currentNode = old_node;
    return false;
  }

  private boolean setNextExamplesList()
  {
    DefaultMutableTreeNode next_node;
    DefaultMutableTreeNode old_node = m_currentNode;

    while ( (next_node = m_currentNode.getNextNode()) != null )
    {
      Object node_object = next_node.getUserObject();

      if ( node_object instanceof Boolean )
      {
        // expand node
        if ( !getDirTreeNode(m_currentNode).expand(m_currentNode) )
          m_currentNode = next_node;
      }
      else
      {
        DirTreeNode next_dtn = getDirTreeNode(next_node);
        if ( next_dtn.getExamplesCount() != 0 )
        {
          // update frame components
          m_dirTree.setSelectionPath(new TreePath(next_node.getPath()));
          m_examplesList.setListData(next_dtn.getExamplesNameList());
          m_examplesList.setSelectedIndex(0);
          // update data
          m_currentDirPath = next_dtn.getFile().getPath() + File.separator;
          m_currentNode = next_node;
          next_dtn.setCurrentExampleIndex(0);
          return true;
        }
        else
          m_currentNode = next_node;
      }
    }
    m_currentNode = old_node;
    return false;
  }

  /********* Interface *********/

  private void makeJTree()
  {
    DirTreeNode f_node = new DirTreeNode(new File(m_currentDirPath));
    DefaultMutableTreeNode top = new DefaultMutableTreeNode(f_node);

    // build directories tree
    top.add(new DefaultMutableTreeNode(Boolean.valueOf(true)));
    f_node.expand(top);

    m_model = new DefaultTreeModel(top);
    m_dirTree = new JTree(m_model);
    m_dirTree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
    m_dirTree.setCellRenderer(new DirTreeCellRenderer());
    m_dirTree.addTreeSelectionListener(new DirTreeSelectionListener());
    m_dirTree.addTreeExpansionListener(new DirTreeExpansionListener());
    m_dirTree.setBackground(BK_COLOR);
  }

  private JToolBar makeButtonsToolBar()
  {
    // previous button
    m_previousButton = new JButton(PREV_ICON);
    m_previousButton.setBorder(BorderFactory.createRaisedBevelBorder());
    m_previousButton.setPressedIcon(EMPTY_ICON);
    m_previousButton.setEnabled(false);
    m_previousButton.setToolTipText("Launch previous demo");
    m_previousButton.addMouseListener(new MouseAdapter()
    {
      @Override
      public void mouseClicked(MouseEvent e)
      {
        if ( m_previousButton.isEnabled() )
          openPreviousDemo();
      }
    });

    // launch button
    m_launchButton = new JButton(PLAY_ICON);
    m_launchButton.setBorder(BorderFactory.createRaisedBevelBorder());
    m_launchButton.setPressedIcon(EMPTY_ICON);
    m_launchButton.setToolTipText("Launch demo");
    m_launchButton.setEnabled(true);
    m_launchButton.addMouseListener(new LaunchButtonMouseAdapter());

    // next button
    m_nextButton = new JButton(NEXT_ICON);
    m_nextButton.setBorder(BorderFactory.createRaisedBevelBorder());
    m_nextButton.setPressedIcon(EMPTY_ICON);
    m_nextButton.setEnabled(false);
    m_nextButton.setToolTipText("Launch next demo");
    m_nextButton.addMouseListener(new MouseAdapter()
    {
      @Override
      public void mouseClicked(MouseEvent e)
      {
        if ( m_nextButton.isEnabled() )
          openNextDemo();
      }
    });

    // stop button
    m_stopButton = new JButton(STOP_ICON);
    m_stopButton.setBorder(BorderFactory.createRaisedBevelBorder());
    m_stopButton.setPressedIcon(EMPTY_ICON);
    m_stopButton.setEnabled(false);
    m_stopButton.setToolTipText("Stop demo(s)");
    m_stopButton.addMouseListener(new MouseAdapter()
    {
      @Override
      public void mouseClicked(MouseEvent e)
      {
        if ( m_stopButton.isEnabled() )
        {
          closeAllDemos();
          if ( m_timerMode )
            timer.stop();
        }
      }
    });

    JToolBar buttons_tb = new JToolBar();
    buttons_tb.add(m_previousButton);
    buttons_tb.add(m_launchButton);
    buttons_tb.add(m_nextButton);
    buttons_tb.add(m_stopButton);
    buttons_tb.setFloatable(false);
    return buttons_tb;
  }

  private JPanel makeToolsPanel()
  {

    // mode checkbox
    m_modeBox = new JCheckBox("Single Example Launch");
    m_modeBox.setFont(m_dirTree.getFont());
    m_modeBox.setSelected(true);
    m_modeBox.setToolTipText("Enable single or multiple demos launches");
    m_modeBox.addMouseListener(new MouseAdapter()
    {
      @Override
      public void mouseClicked(MouseEvent e)
      {
        if ( m_singleMode == true )
          m_singleMode = false;
        else
        {
          if ( m_runningExamples.size() > 1 )
            closeAllDemos();
          m_singleMode = true;
        }
      }
    });

    // timer checkbox
    JCheckBox timer_box = new JCheckBox("Auto-launch");
    timer_box.setFont(m_dirTree.getFont());
    timer_box.setToolTipText("Enable demo auto-launch");
    timer_box.addMouseListener(new TimerBoxMouseAdapter());

    // timer slider
    m_timeSlider = new JSlider(DELAY_MIN, DELAY_MAX, DELAY_INIT);
    m_timeSlider.setMajorTickSpacing(5);
    m_timeSlider.setMinorTickSpacing(1);
    m_timeSlider.setPaintTicks(true);
    m_timeSlider.setPaintLabels(true);
    m_timeSlider.setEnabled(false);
    m_timeSlider.setForeground(SLIDER_TICKS_COLOR);
    m_timeSlider.setToolTipText("Time between two launches");
    m_timeSlider.addChangeListener(new ChangeListener()
    {
      @Override
      public void stateChanged(ChangeEvent e)
      {
        m_timerDelay = m_timeSlider.getValue() * 1000;
      }
    });
    m_timeSlider.addMouseListener(new MouseAdapter()
    {
      @Override
      public void mouseReleased(MouseEvent e)
      {
        if ( timer != null )
          timer.setDelay(m_timerDelay);
      }
    });

    // slider panel
    JPanel slider_panel = new JPanel(new GridBagLayout());
    GridBagConstraints c = new GridBagConstraints();
    c.gridx = 0;
    c.gridy = 0;
    slider_panel.add(new JLabel("Time (seconds)"), c);
    c.gridx = 0;
    c.gridy = 1;
    slider_panel.add(m_timeSlider, c);
    // timer panel
    JPanel timer_panel = new JPanel();
    timer_panel.add(timer_box);
    timer_panel.add(slider_panel);

    // tools panel
    JPanel jp = new JPanel();
    jp.setLayout(new GridBagLayout());
    c.weightx = 1.0;
    c.gridx = 0;
    if ( ENABLE_MULTIPLE_DEMOS )
      jp.add(m_modeBox, c);
    c.weightx = 1.0;
    c.gridx = 1;
    jp.add(timer_panel, c);
    c.weightx = 0;
    c.gridx = 2;
    jp.add(makeButtonsToolBar(), c);
    return jp;
  }

  private void initList()
  {
    // build empty list
    m_examplesList = new JList<>();
    m_examplesList.setBackground(BK_COLOR);
    m_examplesList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    m_examplesList.setCellRenderer(new ExamplesListCellRenderer());
    m_examplesList.addMouseListener(new ExamplesListMouseAdapter());
    m_examplesList.addListSelectionListener(new ListSelectionListener()
    {
      @Override
      public void valueChanged(ListSelectionEvent e)
      {
        m_examplesList.ensureIndexIsVisible(m_examplesList.getSelectedIndex());
      }
    });

    // add list to scrollpane
    m_spList.setViewportView(m_examplesList);
  }

  private void makeFrame()
  {

    /********* Directories Panel *********/
    // JTree
    makeJTree();

    // ScrollPane
    JScrollPane sp1 = new JScrollPane(m_dirTree);
    sp1.setPreferredSize(new Dimension(200, 500));

    JPanel dir_panel = new JPanel(new BorderLayout());
    dir_panel.add(sp1, BorderLayout.CENTER);

    /********* Tools Panel *********/
    m_tools_jp = makeToolsPanel();
    m_tools_jp.setVisible(false);

    /********* Start Image Label *********/
    JLabel jb = new JLabel(LOGO_ICON);
    jb.setVerticalAlignment(SwingConstants.TOP);
    jb.setOpaque(true);
    jb.setBackground(BK_COLOR);

    m_spList = new JScrollPane(jb);
    m_spList.setBorder(new TitledBorder(new EtchedBorder(), "Available Demos"));

    /********* SplitPane *********/
    JSplitPane jsp = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, true, dir_panel, m_spList);
    jsp.setOneTouchExpandable(true);
    jsp.setDividerLocation(200);

    /********* Frame *********/
    setTitle("Demos Launcher - " + m_currentDirPath);
    this.getContentPane().add(jsp, BorderLayout.CENTER);
    this.getContentPane().add(m_tools_jp, BorderLayout.NORTH);

    /*** Menu bar ***/
    JMenuItem exit_item = new JMenuItem("Exit");
    exit_item.addActionListener(new ExitMenuItemListener());

    JMenu file_menu = new JMenu("File");
    file_menu.add(exit_item);

    JCheckBoxMenuItem ctrl_item = new JCheckBoxMenuItem("Controls");
    ctrl_item.addItemListener(new BatchMenuItemListener());

    JMenu view_menu = new JMenu("View");
    view_menu.add(ctrl_item);

    JMenuBar menu_bar = new JMenuBar();
    menu_bar.add(file_menu);
    menu_bar.add(view_menu);

    this.setJMenuBar(menu_bar);
  }

  class ExitMenuItemListener extends AbstractAction
  {
    @Override
    public void actionPerformed(ActionEvent e)
    {
      processEvent(new WindowEvent(DemosLauncher.this, WindowEvent.WINDOW_CLOSING));
    }
  }

  class BatchMenuItemListener implements ItemListener
  {
    @Override
    public void itemStateChanged(ItemEvent e)
    {
      m_tools_jp.setVisible(e.getStateChange() == ItemEvent.SELECTED);
    }
  }

  private void jbInit() throws Exception
  {
    makeFrame();

    addWindowListener(new WindowAdapter()
    {
      @Override
      public void windowClosing(WindowEvent e)
      {
        closeAllDemos();
        System.exit(0);
      }
    });

    Toolkit.getDefaultToolkit().setDynamicLayout(true);
    pack();
    Dimension screen_dim = Toolkit.getDefaultToolkit().getScreenSize();
    setLocation((int) screen_dim.getWidth() - getWidth(), 0);
  }

  /********* Listener Classes *********/
  class DirTreeSelectionListener implements TreeSelectionListener
  {
    @Override
    public void valueChanged(TreeSelectionEvent e)
    {
      DefaultMutableTreeNode node = getTreeNode(e.getPath());
      DirTreeNode selected = getDirTreeNode(node);

      if ( selected != null )
      {
        m_currentDirPath = selected.getFile().getAbsolutePath() + File.separator;
        setTitle("Demos launcher - " + m_currentDirPath);

        if ( m_spList.getViewport().getView() instanceof JLabel )
          // build list component
          initList();

        // update examples list
        if ( selected.getExamplesCount() == 0 )
          m_examplesList.setListData(new String[] { "No available demo in this directory" });
        else
          m_examplesList.setListData(selected.getExamplesNameList());
      }
      else
        setTitle("Demos launcher - ");
    }
  }

  class DirTreeExpansionListener implements TreeExpansionListener
  {
    @Override
    public void treeExpanded(TreeExpansionEvent e)
    {
      final DefaultMutableTreeNode node = getTreeNode(e.getPath());
      final DirTreeNode selected = getDirTreeNode(node);
      m_currentDirPath = selected.getFile().getAbsolutePath() + File.separator;

      // directory expander thread
      Thread runner = new Thread()
      {
        @Override
        public void run()
        {
          if ( selected != null && selected.expand(node) )
          {
            Runnable runnable = new Runnable()
            {
              @Override
              public void run()
              {
                m_model.reload(node);
              }
            };
            SwingUtilities.invokeLater(runnable);
          }
        }
      };
      runner.start();
    }

    @Override
    public void treeCollapsed(TreeExpansionEvent e)
    {}
  }

  class ExamplesListMouseAdapter extends MouseAdapter
  {
    // launch selected demo in examples list on double mouse clicks
    @Override
    public void mouseClicked(MouseEvent e)
    {
      if ( e.getClickCount() == 2 )
      {
        m_currentNode = getTreeNode(m_dirTree.getSelectionPath());
        DirTreeNode current = getDirTreeNode(m_currentNode);

        if ( current.getExamplesCount() != 0 )
        {
          current.setCurrentExampleIndex(m_examplesList.getSelectedIndex());

          if ( m_timerMode )
          {
            if ( timer == null )
            {
              timer = new Timer(m_timerDelay, new LaunchAction());
              timer.setInitialDelay(0);
              timer.start();
            }
            else
            {
              ActionListener[] timer_actions = timer.getActionListeners();
              ((LaunchAction) timer_actions[0]).setFirstExampleSelected(true);
              timer.restart();
            }
          }
          else
            launchDemo();
        }
      }
    }
  }

  class LaunchButtonMouseAdapter extends MouseAdapter
  {
    @Override
    public void mouseClicked(MouseEvent e)
    {
      if ( m_launchButton.isEnabled() )
      {
        if ( m_dirTree.getSelectionCount() == 0 )
          m_dirTree.setSelectionRow(0);

        m_currentNode = getTreeNode(m_dirTree.getSelectionPath());
        if ( getDirTreeNode(m_currentNode).getExamplesCount() == 0 )
        {
          if ( !m_timerMode )
            // search for next examples directory
            openNextDemo();
          else
          {
            LaunchAction la = new LaunchAction();
            la.setFirstExampleSelected(false);
            timer = new Timer(m_timerDelay, la);
            timer.setInitialDelay(0);
            timer.start();
          }
        }
        else
        {
          int index_selected = m_examplesList.getSelectedIndex();
          if ( index_selected == -1 )
          {
            index_selected = 0;
            m_examplesList.setSelectedIndex(0);
          }
          getDirTreeNode(m_currentNode).setCurrentExampleIndex(index_selected);

          if ( m_timerMode )
          {
            timer = new Timer(m_timerDelay, new LaunchAction());
            timer.setInitialDelay(0);
            timer.start();
          }
          else
            launchDemo();
        }
      }
    }
  }

  class TimerBoxMouseAdapter extends MouseAdapter
  {
    @Override
    public void mouseClicked(MouseEvent e)
    {
      JCheckBox timer_box = (JCheckBox) e.getSource();
      if ( timer_box.isSelected() )
      {
        closeAllDemos();
        m_timerMode = true;
        m_modeBox.setEnabled(false);
        m_timeSlider.setEnabled(true);
      }
      else
      {
        closeAllDemos();
        m_timerMode = false;
        m_modeBox.setEnabled(true);
        m_timeSlider.setEnabled(false);
        if ( timer != null )
          timer.stop();
      }
    }
  }

  /********* Launch Action Class *********/
  class LaunchAction extends AbstractAction
  {
    private boolean m_firstExampleSelected;

    public LaunchAction()
    {
      m_firstExampleSelected = true;
    }

    public void setFirstExampleSelected(boolean b)
    {
      m_firstExampleSelected = b;
    }

    @Override
    public void actionPerformed(ActionEvent e)
    {
      if ( m_firstExampleSelected )
      {
        launchDemo();
        m_firstExampleSelected = false;
      }
      else
        openNextDemo();
    }
  }

  /********* Demo Class *********/
  class Demo extends WindowAdapter
  {

    public JFrame m_frame;
    public Example m_example;
    private String m_dirPath;
    private String m_classname;
    private String m_exampleName;

    public Demo(String dirPath, String classname, String exampleName)
    {
      m_dirPath = dirPath;
      m_classname = classname;
      m_exampleName = exampleName;
    }

    public String getDirPath()
    {
      return m_dirPath;
    }

    public void stopDemo()
    {
      if ( m_example != null && m_frame != null )
      {
        m_example.stop();
        m_frame.dispose();
      }
    }

    public void startDemo()
    {
      try
      {
        // path to main class file
        String exampleDir = m_dirPath + m_exampleName + File.separator;

        // load class
        Class<?> c = ClassLoader.getSystemClassLoader().loadClass(m_classname);
        m_example = (Example) c.getDeclaredConstructor().newInstance();

        // invoke "setPrefix" method
        try
        {
          Method setPrefix = m_example.getClass().getMethod("setPrefix", new Class[] { String.class });
          setPrefix.invoke(m_example, exampleDir);
        }
        catch (NoSuchMethodException e)
        {}

        // build example frame and start
        m_frame = new JFrame(m_exampleName);
        m_frame.addWindowListener(this);
        m_frame.addComponentListener(new ComponentAdapter()
        {
          @Override
          public void componentMoved(ComponentEvent e)
          {
            m_exampleFrameLocation = m_frame.getLocationOnScreen();
          }
        });
        m_frame.add(m_example);
        m_example.start();
        m_frame.pack();
        m_frame.setLocation(m_exampleFrameLocation);
        m_frame.setVisible(true);
      }
      catch (Exception e)
      {
        System.err.println(e);
      }
    }

    @Override
    public void windowClosing(WindowEvent e)
    {
      closeDemo(this);
    }
  }

  /********* DirTreeCellRenderer class *********/
  class DirTreeCellRenderer extends JLabel implements TreeCellRenderer
  {

    private final ImageIcon m_clsFoldIcon = new ImageIcon(ROOT_DIR + "/launcher/images/clsfold.png");
    private final ImageIcon m_openFoldIcon = new ImageIcon(ROOT_DIR + "/launcher/images/openfold.png");
    private final ImageIcon m_clsAppletFoldIcon = new ImageIcon(ROOT_DIR + "/launcher/images/clsfoldApplet.png");
    private final ImageIcon m_openAppletFoldIcon = new ImageIcon(ROOT_DIR + "/launcher/images/openfoldApplet.png");
    private final ImageIcon m_logoIcon = new ImageIcon(ROOT_DIR + "/launcher/images/logoOivJava.jpg");

    private final Color m_textColor = UIManager.getColor("Tree.TextForeground");

    protected boolean m_selected;

    public DirTreeCellRenderer()
    {
      super();
      setOpaque(false);
    }

    @Override
    public Component getTreeCellRendererComponent(JTree tree, Object value, boolean sel, boolean expanded, boolean leaf,
        int row, boolean hasFocus)
    {
      DefaultMutableTreeNode node = (DefaultMutableTreeNode) value;
      Object obj = node.getUserObject();
      setText(obj.toString());

      if ( obj instanceof Boolean )
      {
        // node expansion not finished
        setText("Retrieving data...");
        setIcon(null);
      }

      if ( obj instanceof DirTreeNode )
      {
        int examplesCount = ((DirTreeNode) obj).getExamplesCount();
        // set icon node
        if ( leaf )
        {
          if ( examplesCount != 0 )
            setIcon(m_logoIcon);
          else
            setIcon(null);
        }
        else if ( expanded )
        {
          if ( examplesCount != 0 )
            setIcon(m_openAppletFoldIcon);
          else
            setIcon(m_openFoldIcon);
        }
        else if ( examplesCount != 0 )
          setIcon(m_clsAppletFoldIcon);
        else
          setIcon(m_clsFoldIcon);
      }

      setFont(m_dirTree.getFont());
      setForeground(m_textColor);
      m_selected = sel;
      return this;
    }

    @Override
    public void paintComponent(Graphics g)
    {
      Icon icon = getIcon();
      int offset = 0;

      if ( icon != null )
        offset = icon.getIconWidth() + getIconTextGap();

      if ( m_selected )
      {
        // draw selection rectangle
        g.setColor(BK_SELECTION_COLOR);
        g.fillRect(offset, 0, getWidth() - offset, getHeight());
      }
      super.paintComponent(g);
    }
  }

  /********* ExamplesListCellRenderer class *********/
  class ExamplesListCellRenderer extends JPanel implements ListCellRenderer<Object>
  {
    private int TXT_LINES_NB = 5;
    private JLabel m_demoName;
    private JTextArea m_demoTxt;
    private JButton m_demoImage;

    public ExamplesListCellRenderer()
    {
      m_demoName = new JLabel();
      m_demoName.setOpaque(true);
      Font tree_font = m_dirTree.getFont();
      m_demoName.setFont(new Font(tree_font.getName(), Font.BOLD, tree_font.getSize()));

      m_demoTxt = new JTextArea();

      m_demoImage = new JButton();
      m_demoImage.setBackground(Color.white);
      m_demoImage.setPreferredSize(new Dimension(100, 100));
      m_demoImage.setBorder(BorderFactory.createMatteBorder(0, 0, 8, 8, Color.white));

      setLayout(new BorderLayout());
      // setBorder(BorderFactory.createMatteBorder(0, 0, 1, 0, Color.black));
    }

    @Override
    public Component getListCellRendererComponent(JList<?> list, Object value, int index, boolean isSelected,
        boolean cellHasFocus)
    {
      boolean has_image = false;
      boolean has_txt = false;

      String exampleName = value.toString();
      String exampleDir = m_currentDirPath + exampleName + File.separatorChar;

      // Example name
      m_demoName.setText(exampleName);
      m_demoName.setBackground(isSelected ? BK_SELECTION_COLOR : BK_COLOR);

      // Example text
      try
      {
        BufferedReader buffer =
            new BufferedReader(new InputStreamReader(new FileInputStream(exampleDir + exampleName + ".txt")));

        // limit text lines number
        m_demoTxt.setText(null);
        String s = buffer.readLine();
        for ( int i = 0; s != null && i < TXT_LINES_NB; i++, s = buffer.readLine() )
          m_demoTxt.append(s + '\n');
        has_txt = true;
        buffer.close();
      }
      catch (FileNotFoundException e)
      {
        m_demoTxt.setText("");
        has_txt = false;
      }
      catch (Exception e)
      {
        System.err.println(e);
      }

      // Example image
      File img_file = new File(exampleDir + exampleName + ".jpg");
      if ( img_file.exists() )
      {
        try
        {
          ImageIcon img = new ImageIcon(img_file.toURI().toURL());
          m_demoImage.setIcon(img);
          has_image = true;
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
      else
      {
        has_image = false;
        m_demoImage.setIcon(null);
      }

      add(m_demoName, BorderLayout.NORTH);
      if ( has_image )
      {
        add(m_demoImage, BorderLayout.EAST);
        add(m_demoTxt, BorderLayout.CENTER);
      }
      else if ( has_txt )
      {
        add(m_demoTxt, BorderLayout.CENTER);
        remove(m_demoImage);
      }
      else
      {
        remove(m_demoTxt);
        remove(m_demoImage);
      }

      return this;
    }
  }
}
