package medical.web.medicalremotermpr.service;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec2f;
import com.openinventor.inventor.SbVec2s;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbVec3i32;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.actions.SoGetBoundingBoxAction;
import com.openinventor.inventor.events.SoEvent;
import com.openinventor.inventor.events.SoLocation2Event;
import com.openinventor.inventor.events.SoMouseButtonEvent;
import com.openinventor.inventor.events.SoMouseWheelEvent;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.nodes.SoSeparator.FastEditings;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor;
import com.openinventor.ldm.nodes.SoDataRange;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.medical.nodes.Gnomon;
import com.openinventor.medical.nodes.TextBox;
import com.openinventor.remoteviz.rendering.RenderArea;
import com.openinventor.remoteviz.rendering.RenderAreaListener;
import com.openinventor.remoteviz.rendering.ServiceListener;
import com.openinventor.volumeviz.nodes.SoOrthoSlice;
import com.openinventor.volumeviz.nodes.SoSlice;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingQuality;

public class MedicalRemoteMprService extends ServiceListener
{
  private static final String VOLUME_FILENAME = "/medical/data/dicomSample/listOfDicomFiles.dcm";
  public static final String COLORMAPFILENAME = "/medical/data/resources/volrenGlow.am";

  private static final String[] AXIS_NAMES = { "Sagittal", "Coronal", "Axial" };
  private static final String ORTHO_SLICE_NAME = "orthoSlice";
  private static final String NAME_S_NUM_S_POS = "nameSNumSPos";

  private final static Logger LOGGER = Logger.getLogger(MedicalRemoteMprService.class.getName());

  // Common volume info
  private SoVolumeData m_volData = null;
  private SbVec3i32 m_volDim = new SbVec3i32(0, 0, 0);
  private SbBox3f m_volExt = new SbBox3f(0, 0, 0, 0, 0, 0);
  private SbVec3f m_voxelSize = new SbVec3f(0, 0, 0);

  // Volume rendering info
  private SoDataRange m_volRange = null;
  private SoTransferFunction m_volCmap = null;
  private SoVolumeRenderingQuality m_volQual = null;
  private SoVolumeRender m_volRend = null;

  // Slice rendering info (all slices share same data range etc)
  private SoDataRange m_sliceRange = null;
  private SoTransferFunction m_sliceCmap = null;
  private SoMaterial m_sliceMatl = null;

  private SoOrthoSlice m_sagittalSlice = null;
  private SoOrthoSlice m_coronalSlice = null;
  private SoOrthoSlice m_axialSlice = null;

  private SoSwitch[] m_sliceSBars;

  private SoSwitch[] m_referenceLineSwitch;
  private SoTranslation m_coronalOnAxialTrans = null;
  private SoTranslation m_coronalOnSagittalTrans = null;
  private SoTranslation m_axialOnCoronalTrans = null;
  private SoTranslation m_axialOnSagittalTrans = null;
  private SoTranslation m_sagittalOnAxialTrans = null;
  private SoTranslation m_sagittalOnCoronalTrans = null;

  // Text for CW in each viewer.
  private TextBox m_textCW_TR = null;
  private TextBox m_textCW_BR = null;
  private TextBox m_textCW_BL = null;
  // Upper left annotation (View name, slice number, slice position)
  private TextBox m_textUL_TR = null;
  private TextBox m_textUL_BR = null;
  private TextBox m_textUL_BL = null;

  // Mouse button state and enable display of voxel values
  enum MouseMode
  {
    MOUSE_NOOP, // Moving the mouse does nothing
    MOUSE_SHOW_VALUE, // Show voxel pos/value as mouse moves (default when no
                      // button)
    MOUSE_SCROLL_IMAGE, // Change image number as mouse moves up/down (bound to
                        // mouse 1)
    MOUSE_CHANGE_WINCW // Change window center/level as mouse moves (bound to
                       // mouse 2)
  };

  private SbVec2s m_mousePosition = new SbVec2s((short) 0, (short) 0);
  private MouseMode m_mouseMode = MouseMode.MOUSE_SHOW_VALUE;
  private String m_colormap = null;

  public MedicalRemoteMprService()
  {
    m_sliceSBars = new SoSwitch[3];
    m_referenceLineSwitch = new SoSwitch[3];

    // Global initialization -- Load the volume data
    m_volData = new SoVolumeData();
    m_volData.extent.setValue(0, 0, 0, 1, 1, 1); // Temporary until data loaded

    // Load example resources
    String dicom;
    try
    {
      dicom = new File(Main.class.getResource(VOLUME_FILENAME).toURI()).toString();
      m_colormap = new File(Main.class.getResource(COLORMAPFILENAME).toURI()).toString();
    }
    catch (Exception e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load resources", e);
      return;
    }

    m_volData.fileName.setValue(dicom);
    MedicalHelper.dicomAdjustVolume(m_volData, true);

    // Remember volume characteristics for user interface updates.
    // Note that volume extent min is the outer edge of the first voxel,
    // DICOM origin is the _center_ of the first voxel (difference of 1/2
    // voxel).
    m_volDim = m_volData.data.getSize();
    m_volExt = m_volData.extent.getValue();
    SbVec3f volSize = m_volExt.getSize();
    m_voxelSize.setValue(volSize.getX() / m_volDim.getX(), volSize.getY() / m_volDim.getY(),
        volSize.getZ() / m_volDim.getZ());
    SbVec3f dicomOrigin = m_volExt.getMin();
    dicomOrigin.setX(dicomOrigin.getX() + (0.5f * m_voxelSize.getX()));
    dicomOrigin.setY(dicomOrigin.getY() + (0.5f * m_voxelSize.getY()));
    dicomOrigin.setZ(dicomOrigin.getZ() + (0.5f * m_voxelSize.getZ()));

    m_coronalOnAxialTrans = new SoTranslation();
    m_coronalOnSagittalTrans = new SoTranslation();
    m_axialOnCoronalTrans = new SoTranslation();
    m_axialOnSagittalTrans = new SoTranslation();
    m_sagittalOnAxialTrans = new SoTranslation();
    m_sagittalOnCoronalTrans = new SoTranslation();

    // One Data Range for 3 ortho slices.
    m_sliceRange = new SoDataRange();
    m_sliceRange.min.setValue(0.0);
    m_sliceRange.max.setValue(256.0);
    m_sagittalSlice = new SoOrthoSlice();
    m_coronalSlice = new SoOrthoSlice();
    m_axialSlice = new SoOrthoSlice();

    // Shared slice rendering material (full intensity)
    m_sliceMatl = new SoMaterial();
    m_sliceMatl.diffuseColor.setValue(1, 1, 1);

    // Shared slice rendering colormap (opaque gray scale).
    // Note we will check the Photometric Interpretation value when we actually
    // load some data.
    m_sliceCmap = new SoTransferFunction();
    m_sliceCmap.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.INTENSITY);
  }

  @Override
  public void onInstantiatedRenderArea(RenderArea renderArea)
  {
    RenderAreaListener renderAreaListener = new RenderAreaListener();
    renderArea.addListener(renderAreaListener);
    renderArea.getTouchManager().addDefaultRecognizers();

    // 3D View (top left viewer)
    if ( renderArea.getId().equals("renderArea3D_TOP_LEFT") )
    {
      // Instantiate a sceneExaminer to interact with the camera
      SceneExaminer rootTL = new SceneExaminer();

      rootTL.setCameraMode(SceneInteractor.CameraMode.ORTHOGRAPHIC);
      renderArea.getSceneManager().getGLRenderAction()
          .setTransparencyType(SoGLRenderAction.TransparencyTypes.SORTED_PIXEL);
      renderArea.getSceneManager().setAntialiasing(0.9f);

      createVolumeView(rootTL);
      MedicalHelper.orientView(MedicalHelper.Axis.CORONAL, rootTL.getCameraInteractor().getCamera(), m_volData);

      // Apply the sceneExaminer node as renderArea scene graph
      renderArea.getSceneManager().setSceneGraph(rootTL);
      rootTL.viewAll(renderArea.getSceneManager().getViewportRegion());
    }

    // SAGITTAL View (top right viewer)
    if ( renderArea.getId().equals("renderArea2D_TOP_RIGHT") )
    {
      // Instantiate a sceneExaminer to interact with the camera
      SceneExaminer rootTR = new SceneExaminer();
      rootTR.enableOrbit(false);
      rootTR.setInteractionMode(SceneExaminer.InteractionMode.SELECTION);
      rootTR.setCameraMode(SceneInteractor.CameraMode.ORTHOGRAPHIC);

      // Create slice rendering scene graphs
      createSliceView(rootTR, MedicalHelper.Axis.SAGITTAL,
          m_volDim.getValueAt(MedicalHelper.Axis.SAGITTAL.getValue()) / 2);
      renderArea.getSceneManager().setSceneGraph(rootTR);
      rootTR.viewAll(renderArea.getSceneManager().getViewportRegion());
    }

    // CORONAL View (bottom left viewer)
    if ( renderArea.getId().equals("renderArea2D_BOTTOM_LEFT") )
    {
      // Instantiate a sceneExaminer to interact with the camera
      SceneExaminer rootBL = new SceneExaminer();
      rootBL.enableOrbit(false);
      rootBL.setInteractionMode(SceneExaminer.InteractionMode.SELECTION);
      rootBL.setCameraMode(SceneInteractor.CameraMode.ORTHOGRAPHIC);

      // Create slice rendering scene graphs
      createSliceView(rootBL, MedicalHelper.Axis.CORONAL,
          m_volDim.getValueAt(MedicalHelper.Axis.CORONAL.getValue()) / 2);
      renderArea.getSceneManager().setSceneGraph(rootBL);
      rootBL.viewAll(renderArea.getSceneManager().getViewportRegion());
    }

    // TRANSVERSE View (bottom right viewer)
    if ( renderArea.getId().equals("renderArea2D_BOTTOM_RIGHT") )
    {
      // Instantiate a sceneExaminer to interact with the camera
      SceneExaminer rootBR = new SceneExaminer();
      rootBR.enableOrbit(false);

      rootBR.setInteractionMode(SceneExaminer.InteractionMode.SELECTION);
      rootBR.setCameraMode(SceneInteractor.CameraMode.ORTHOGRAPHIC);

      // Create slice rendering scene graphs
      createSliceView(rootBR, MedicalHelper.Axis.AXIAL, m_volDim.getValueAt(MedicalHelper.Axis.AXIAL.getValue()) / 2);
      renderArea.getSceneManager().setSceneGraph(rootBR);
      rootBR.viewAll(renderArea.getSceneManager().getViewportRegion());
    }
  }

  /**
   * Create scene graph for volume rendering.
   */
  private void createVolumeView(SceneExaminer root)
  {
    SoSeparator volSep = new SoSeparator();
    root.addChild(volSep);

    // The volume is not pickable.
    SoPickStyle pickStyle = new SoPickStyle();
    pickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);
    volSep.addChild(pickStyle);

    // Add an instance of the volume data
    volSep.addChild(m_volData);

    // We already selected a range for the demo data set.
    m_volRange = new SoDataRange();
    m_volRange.min.setValue(176);
    m_volRange.max.setValue(476);
    volSep.addChild(m_volRange);

    // To render BorderSlice in the 3D viewer.
    SoOrthoSlice sagittalSlice = new SoOrthoSlice();
    sagittalSlice.sliceNumber.connectFrom(m_sagittalSlice.sliceNumber);
    sagittalSlice.axis.connectFrom(m_sagittalSlice.axis);
    sagittalSlice.borderColor.connectFrom(m_sagittalSlice.borderColor);
    sagittalSlice.borderWidth.connectFrom(m_sagittalSlice.borderWidth);
    sagittalSlice.enableBorder.connectFrom(m_sagittalSlice.enableBorder);
    sagittalSlice.enableImage.setValue(false);
    volSep.addChild(sagittalSlice);

    SoOrthoSlice coronalSlice = new SoOrthoSlice();
    coronalSlice.sliceNumber.connectFrom(m_coronalSlice.sliceNumber);
    coronalSlice.axis.connectFrom(m_coronalSlice.axis);
    coronalSlice.borderColor.connectFrom(m_coronalSlice.borderColor);
    coronalSlice.borderWidth.connectFrom(m_coronalSlice.borderWidth);
    coronalSlice.enableBorder.connectFrom(m_coronalSlice.enableBorder);
    coronalSlice.enableImage.setValue(false);
    volSep.addChild(coronalSlice);

    SoOrthoSlice axialSlice = new SoOrthoSlice();
    axialSlice.sliceNumber.connectFrom(m_axialSlice.sliceNumber);
    axialSlice.axis.connectFrom(m_axialSlice.axis);
    axialSlice.borderColor.connectFrom(m_axialSlice.borderColor);
    axialSlice.borderWidth.connectFrom(m_axialSlice.borderWidth);
    axialSlice.enableBorder.connectFrom(m_axialSlice.enableBorder);
    axialSlice.enableImage.setValue(false);
    volSep.addChild(axialSlice);

    // We already created a color map for the demo data set.
    m_volCmap = new SoTransferFunction();
    m_volCmap.loadColormap(m_colormap);
    m_volCmap.minValue.setValue(1);
    volSep.addChild(m_volCmap);

    // Volume rendering options (lighting, etc)
    m_volQual = new SoVolumeRenderingQuality();
    m_volQual.interpolateOnMove.setValue(true);
    m_volQual.preIntegrated.setValue(true);
    volSep.addChild(m_volQual);

    // The actual rendering node.
    // Note the lowResMode is only used when Open Inventor is in interactive
    // mode.
    m_volRend = new SoVolumeRender();
    m_volRend.numSlicesControl.setValue(SoVolumeRender.NumSlicesControls.AUTOMATIC);
    m_volRend.numSlices.setValue(-1);
    m_volRend.samplingAlignment.setValue(SoVolumeRender.SamplingAlignments.BOUNDARY_ALIGNED);
    m_volRend.opacityThreshold.setValue(0.01f);
    m_volRend.lowResMode.setValue(SoVolumeRender.LowResModes.DECREASE_SCREEN_RESOLUTION);
    m_volRend.lowScreenResolutionScale.setValue(2);
    volSep.addChild(m_volRend);

    // Orientation indicator
    volSep.addChild(new Gnomon());

    // OIV Logo
    // Add Open Inventor logo at the left-bottom corner
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }
    volSep.addChild(logoBackground);

  }

  /**
   * Create a specific slice view
   */
  private void createSliceView(SceneExaminer viewScene, MedicalHelper.Axis axis, int sliceNumber)
  {
    // Handle events specific to slices
    // Note that the SceneExaminer automatically converts 1-finger touch and
    // drag events to mouse button and mouse move events. So we can handle basic
    // touch interaction using the onMouseXXX methods.
    SoEventCallback eventNode = new SoEventCallback();
    eventNode.addEventCallback(SoMouseButtonEvent.class, new OnMouseButtonSlice(), null);
    eventNode.addEventCallback(SoLocation2Event.class, new OnMouseMoveSlice(axis), null);
    eventNode.addEventCallback(SoMouseWheelEvent.class, new OnMouseWheelSlice(axis), null);
    viewScene.addChild(eventNode);

    // Force bounding box to be extent of the volume.
    // Note that SceneExaminer does not automatically adjust near/far clipping
    // planes unless the camera is modified (unlike the legacy ExaminerViewer).
    // Changing to a different slice means the slice geometry is at a different
    // Z value and that could move the geometry outside the current clip planes.
    // Setting the bbox to the full volume extent ensures all slices are
    // visible.
    SoBBox bboxNode = new SoBBox();
    bboxNode.mode.setValue(SoBBox.Modes.USER_DEFINED);
    bboxNode.boundingBox.connectFrom(m_volData.extent);
    viewScene.addChild(bboxNode);

    // Keep the volume rendering separate from geometry rendering.
    SoSeparator sceneSep = new SoSeparator();
    sceneSep.setName(AXIS_NAMES[axis.getValue()]);
    viewScene.addChild(sceneSep);

    // Add an instance of the shared volume data
    sceneSep.addChild(m_volData);

    // Add shared slice rendering property nodes.
    sceneSep.addChild(m_sliceRange);

    sceneSep.addChild(m_sliceMatl);
    sceneSep.addChild(m_sliceCmap);

    SoOrthoSlice slice;
    SbColor borderColor;
    TextBox textUL;
    TextBox textCW;
    switch ( axis )
    {
    case SAGITTAL :
      slice = m_sagittalSlice;
      borderColor = new SbColor(0.9f, 0.2f, 0.5f);
      textUL = m_textUL_TR = new TextBox();
      textCW = m_textCW_TR = new TextBox();
      break;

    case CORONAL :
      slice = m_coronalSlice;
      borderColor = new SbColor(0.2f, 0.9f, 0.5f);
      textUL = m_textUL_BL = new TextBox();
      textCW = m_textCW_BL = new TextBox();
      break;

    case AXIAL :
    default:
      slice = m_axialSlice;
      borderColor = new SbColor(0.5f, 0.2f, 0.9f);
      textUL = m_textUL_BR = new TextBox();
      textCW = m_textCW_BR = new TextBox();
      break;
    }

    int axisValue = axis.getValue();

    slice.borderWidth.setValue(1.0f);
    slice.enableBorder.setValue(true);
    slice.axis.setValue(axisValue);
    slice.borderColor.setValue(borderColor);
    slice.setName(ORTHO_SLICE_NAME);
    slice.sliceNumber.setValue(sliceNumber);
    slice.interpolation.setValue(SoSlice.Interpolations.MULTISAMPLE_12);
    sceneSep.addChild(slice);
    // Add standard slice orientation markers
    viewScene.addChild(MedicalHelper.buildSliceOrientationMarkers(slice));

    textCW.position.setValue(-0.98f, -0.9f, 0);
    textCW.fontSize.setValue(12.f);
    SbVec2f winCW = MedicalHelper.dicomGetWindowCenterWidth(m_sliceRange);
    textCW.addLine("WL / WW: " + (int) winCW.getX() + " / " + (int) winCW.getY());
    viewScene.addChild(textCW);

    textUL.setName(NAME_S_NUM_S_POS);
    textUL.position.setValue(-0.98f, 0.98f, 0.0f);
    textUL.fontSize.setValue(12.f);
    textUL.addLine(AXIS_NAMES[axisValue]);
    textUL.addLine("Image: ");
    viewScene.addChild(textUL);
    updateSliceNumText(slice);

    // Adjust camera
    MedicalHelper.orientView(axis, viewScene.getCamera(), m_volData); // Standard
                                                                      // orientation
    viewScene.getCameraInteractor().pushCamera(); // Save default view

    // Add slice scale bars
    m_sliceSBars[axisValue] = new SoSwitch();
    // Turned off when viewport is small
    m_sliceSBars[axisValue].whichChild.setValue(0);
    m_sliceSBars[axisValue].addChild(MedicalHelper.buildSliceScaleBars(viewScene.getCamera()));
    viewScene.addChild(m_sliceSBars[axisValue]);

    // Add annotation to create reference lines
    m_referenceLineSwitch[axisValue] = new SoSwitch();
    m_referenceLineSwitch[axisValue].addChild(createReferenceLines(axis));
    m_referenceLineSwitch[axisValue].whichChild.setValue(0);
    viewScene.addChild(m_referenceLineSwitch[axisValue]);

    // Init first line pos:
    setVHLines(axisValue);
  }

  private void setVHLines(int axis)
  {
    SbViewportRegion vr = new SbViewportRegion();
    SoGetBoundingBoxAction bbAction = new SoGetBoundingBoxAction(vr);
    bbAction.apply(SoNode.getByName(AXIS_NAMES[axis]));

    float pos;
    if ( axis == MedicalHelper.Axis.CORONAL.getValue() )
    {
      pos = bbAction.getCenter().getValue()[1];
      m_coronalOnAxialTrans.translation.setValue(0, pos, 0);
      m_coronalOnSagittalTrans.translation.setValue(0, pos, 0);
    }
    else if ( axis == MedicalHelper.Axis.SAGITTAL.getValue() )
    {
      pos = bbAction.getCenter().getValue()[0];
      m_sagittalOnAxialTrans.translation.setValue(pos, 0, 0);
      m_sagittalOnCoronalTrans.translation.setValue(pos, 0, 0);
    }
    else if ( axis == MedicalHelper.Axis.AXIAL.getValue() )
    {
      pos = bbAction.getCenter().getValue()[2];
      m_axialOnCoronalTrans.translation.setValue(0, 0, pos);
      m_axialOnSagittalTrans.translation.setValue(0, 0, pos);
    }
  }

  /**
   * Update Window Center/Width based on change in event position. Called from
   * OnMouseMoveSlice.
   */
  private void updateWindowCW(SbVec2s newPos)
  {
    int deltaX = newPos.getX() - m_mousePosition.getX();
    int deltaY = newPos.getY() - m_mousePosition.getY();
    // Sometimes we get a stream of mouse move events with the same event
    // position.
    // This is probably a glitch in the mouse and/or Windows. Ignore this case.
    if ( deltaX != 0 || deltaY != 0 )
    {
      SbVec2f winCW = MedicalHelper.dicomGetWindowCenterWidth(m_sliceRange);
      winCW.setX(winCW.getX() + deltaY);
      winCW.setY(winCW.getY() + deltaX);
      if ( winCW.getY() < 1 )
        winCW.setY(1);
      MedicalHelper.dicomSetWindowCenterWidth(m_sliceRange, winCW);

      updateWinCtrWidthText(winCW.getX(), winCW.getY());
    }
  }

  /**
   * Update wherever the UI displays the window center/width.
   */
  private void updateWinCtrWidthText(float center, float width)
  {
    String str = "WL / WW: " + (int) center + " / " + (int) width;
    m_textCW_TR.setLine(str, 0);
    m_textCW_BR.setLine(str, 0);
    m_textCW_BL.setLine(str, 0);
  }

  /**
   * Update wherever the UI displays the slice number.
   *
   * NOTE! OIV numbers slices starting at 0, but medical applications typically
   * display the image number starting at 1.
   */
  private void updateSliceNumText(SoOrthoSlice sliceNode)
  {

    int sliceNum = sliceNode.sliceNumber.getValue();
    int sliceAxis = sliceNode.axis.getValue();
    int numSlices = m_volDim.getValueAt(sliceAxis);
    TextBox textBox;

    switch ( sliceAxis )
    {
    case 0 : // SAGITTAL
      textBox = m_textUL_TR;
      break;
    case 1 : // CORONAL
      textBox = m_textUL_BL;
      break;
    case 2 : // AXIAL
    default:
      textBox = m_textUL_BR;
      break;
    }

    // Update image number
    String str = "Image " + sliceNum + "  of  " + numSlices;
    textBox.setLine(str, 1);

    // Update image position
    float pos = m_volExt.getMin().getValueAt(sliceAxis) + (sliceNum * m_voxelSize.getValueAt(sliceAxis));
    str = String.format("%.3f mm", pos);
    textBox.setLine(str, 2);
  }

  private SoAnnotation createReferenceLines(MedicalHelper.Axis axis)
  {
    SoAnnotation annotation = new SoAnnotation();
    annotation.fastEditing.setValue(FastEditings.KEEP_ZBUFFER);

    SoDrawStyle lineDrawStyle = new SoDrawStyle();
    lineDrawStyle.lineWidth.setValue(0.5f);
    annotation.addChild(lineDrawStyle);

    // Create horizontal reference line
    float[][] horizontalLine = new float[2][3];
    SoSeparator hLineSep = new SoSeparator();
    annotation.addChild(hLineSep);
    {
      SbColor lineColor;

      switch ( axis )
      {
      case SAGITTAL :
        lineColor = new SbColor(0.2f, 0.5f, 0.9f);
        m_axialOnSagittalTrans.setName("axialOnSagittalTrans");
        hLineSep.addChild(m_axialOnSagittalTrans);

        horizontalLine[0][0] = 0.0f;
        horizontalLine[0][1] = -1000.0f;
        horizontalLine[0][2] = 0.0f;
        horizontalLine[1][0] = 0.0f;
        horizontalLine[1][1] = 1000.0f;
        horizontalLine[1][2] = 0.0f;
        break;
      case CORONAL :
        lineColor = new SbColor(0.2f, 0.5f, 0.9f);
        m_axialOnCoronalTrans.setName("axialOnCoronalTrans");
        hLineSep.addChild(m_axialOnCoronalTrans);

        horizontalLine[0][0] = -1000.0f;
        horizontalLine[0][1] = 0.0f;
        horizontalLine[0][2] = 0.0f;
        horizontalLine[1][0] = 1000.0f;
        horizontalLine[1][1] = 0.0f;
        horizontalLine[1][2] = 0.0f;
        break;
      case AXIAL :
      default:
        lineColor = new SbColor(0.2f, 0.9f, 0.5f);
        m_coronalOnAxialTrans.setName("coronalOnAxialTrans");
        hLineSep.addChild(m_coronalOnAxialTrans);

        horizontalLine[0][0] = -1000.0f;
        horizontalLine[0][1] = 0.0f;
        horizontalLine[0][2] = -100.0f;
        horizontalLine[1][0] = 1000.0f;
        horizontalLine[1][1] = 0.0f;
        horizontalLine[1][2] = -100.0f;
        break;
      }

      SoMaterial hLinemat = new SoMaterial();
      hLinemat.diffuseColor.setValue(lineColor);
      hLineSep.addChild(hLinemat);

      SoVertexProperty vp = new SoVertexProperty();
      vp.vertex.setValues(0, horizontalLine);
      SoLineSet line = new SoLineSet();
      line.numVertices.set1Value(0, 2);
      line.vertexProperty.setValue(vp);
      hLineSep.addChild(line);
    }

    // Create vertical reference line
    float[][] verticalLine = new float[2][3];
    SoSeparator vLineSep = new SoSeparator();
    annotation.addChild(vLineSep);
    {
      SbColor lineColor;

      switch ( axis )
      {
      case SAGITTAL :
        lineColor = new SbColor(0.2f, 0.9f, 0.5f);
        m_coronalOnSagittalTrans.setName("m_coronalOnSagittalTrans");
        vLineSep.addChild(m_coronalOnSagittalTrans);

        verticalLine[0][0] = 0.0f;
        verticalLine[0][1] = 0.0f;
        verticalLine[0][2] = -1000.0f;
        verticalLine[1][0] = 0.0f;
        verticalLine[1][1] = 0.0f;
        verticalLine[1][2] = 1000.0f;
        break;
      case CORONAL :
        lineColor = new SbColor(0.9f, 0.2f, 0.5f);
        m_sagittalOnCoronalTrans.setName("m_sagittalOnCoronalTrans");
        vLineSep.addChild(m_sagittalOnCoronalTrans);

        verticalLine[0][0] = 0.0f;
        verticalLine[0][1] = 0.0f;
        verticalLine[0][2] = -1000.0f;
        verticalLine[1][0] = 0.0f;
        verticalLine[1][1] = 0.0f;
        verticalLine[1][2] = 1000.0f;
        break;
      case AXIAL :
      default:
        lineColor = new SbColor(0.9f, 0.2f, 0.5f);
        m_sagittalOnAxialTrans.setName("m_sagittalOnAxialTrans");
        vLineSep.addChild(m_sagittalOnAxialTrans);

        verticalLine[0][0] = 0.0f;
        verticalLine[0][1] = -1000.0f;
        verticalLine[0][2] = -100.0f;
        verticalLine[1][0] = 0.0f;
        verticalLine[1][1] = 1000.0f;
        verticalLine[1][2] = -100.0f;
        break;
      }

      SoMaterial vLinemat = new SoMaterial();
      vLinemat.diffuseColor.setValue(lineColor);
      vLineSep.addChild(vLinemat);

      SoVertexProperty vp = new SoVertexProperty();
      vp.vertex.setValues(0, verticalLine);
      SoLineSet line = new SoLineSet();
      line.numVertices.set1Value(0, 2);
      line.vertexProperty.setValue(vp);
      vLineSep.addChild(line);
    }

    return annotation;
  }

  /**
   * Handle mouse button events for slices. Affects what happens when the mouse
   * cursor is moved.
   * <p>
   * SceneExaminer automatically converts 1-finger down/up touch events to mouse
   * button down/up events. So we are also handling touch here.
   */
  private class OnMouseButtonSlice extends SoEventCallbackCB
  {
    private boolean m_mouse1Down = false;
    private boolean m_mouse2Down = false;

    @Override
    public void invoke(SoEventCallback node)
    {
      SoEvent theEvent = node.getEvent();

      if ( SoMouseButtonEvent.isButtonPressEvent(theEvent, SoMouseButtonEvent.Buttons.BUTTON1) )
      {
        m_mouse1Down = true;
        if ( !m_mouse2Down ) // Don't change if we're already in a non-default
                             // mode.
          m_mouseMode = MouseMode.MOUSE_SCROLL_IMAGE;

        node.setHandled();
      }
      else if ( SoMouseButtonEvent.isButtonReleaseEvent(theEvent, SoMouseButtonEvent.Buttons.BUTTON1) )
      {
        m_mouse1Down = false;
        if ( !m_mouse2Down )
          m_mouseMode = MouseMode.MOUSE_SHOW_VALUE; // Default when no button
                                                    // pressed
        node.setHandled();
      }
      else if ( SoMouseButtonEvent.isButtonPressEvent(theEvent, SoMouseButtonEvent.Buttons.BUTTON2) )
      {
        m_mouse2Down = true;
        if ( !m_mouse1Down )
          m_mouseMode = MouseMode.MOUSE_CHANGE_WINCW;
        node.setHandled();
      }
      else if ( SoMouseButtonEvent.isButtonReleaseEvent(theEvent, SoMouseButtonEvent.Buttons.BUTTON2) )
      {
        m_mouse2Down = false;
        if ( !m_mouse1Down )
          m_mouseMode = MouseMode.MOUSE_SHOW_VALUE; // Default when no button
                                                    // pressed
        node.setHandled();
      }
      else if ( SoMouseButtonEvent.isButtonPressEvent(theEvent, SoMouseButtonEvent.Buttons.BUTTON3) )
      {
        node.setHandled();
      }
      else if ( SoMouseButtonEvent.isButtonReleaseEvent(theEvent, SoMouseButtonEvent.Buttons.BUTTON3) )
      {
        node.setHandled();
      }

      // Remember position
      m_mousePosition = theEvent.getPosition();
    }
  }

  /**
   * Handle mouse move events for slices. Behavior depends on which mouse
   * buttons are pressed (if any).
   */
  private class OnMouseMoveSlice extends SoEventCallbackCB
  {

    private MedicalHelper.Axis m_axis;

    public OnMouseMoveSlice(MedicalHelper.Axis axis)
    {
      m_axis = axis;
    }

    @Override
    public void invoke(SoEventCallback node)
    {
      SoLocation2Event theEvent = (SoLocation2Event) node.getEvent();

      // Check what mode we are in.
      if ( m_mouseMode == MouseMode.MOUSE_SCROLL_IMAGE )
      {
        // ---------------------------------------------------------------
        // This change only applies to the current slice window.
        SbVec2s newPos = theEvent.getPosition();
        int delta = newPos.getY() - m_mousePosition.getY();
        // Sometimes we get a stream of mouse move events with the same event
        // position.
        // This is probably a glitch in the mouse and/or Windows. Ignore this
        // case.
        if ( delta != 0 )
        {
          SoOrthoSlice sliceNode;
          switch ( m_axis )
          {
          case SAGITTAL :
            sliceNode = m_sagittalSlice;
            break;
          case CORONAL :
            sliceNode = m_coronalSlice;
            break;
          case AXIAL :
          default:
            sliceNode = m_axialSlice;
            break;
          }

          int curSlice = sliceNode.sliceNumber.getValue();
          int numSlices = m_volDim.getValueAt(sliceNode.axis.getValue());

          // If delta is negative, increment slice number, else decrement.
          // This arithmetic seems "backward" but it's the expected behavior.
          int newSlice = curSlice - delta;

          // Update if necessary.
          // SoOrthoSlice will "clamp" invalid values, so this check is not
          // strictly necessary, but it avoids triggering an unnecessary redraw.
          if ( newSlice < 0 )
            newSlice = 0;
          else if ( newSlice >= numSlices )
            newSlice = numSlices - 1;

          if ( newSlice != curSlice )
          {
            sliceNode.sliceNumber.setValue(newSlice);

            // Set Line pos
            setVHLines(sliceNode.axis.getValue());
          }
          updateSliceNumText(sliceNode);
        }
      }
      else if ( m_mouseMode == MouseMode.MOUSE_CHANGE_WINCW )
      {
        // ---------------------------------------------------------------
        // This change applies to all slice windows.

        updateWindowCW(theEvent.getPosition());
      }

      // Remember new position
      m_mousePosition = theEvent.getPosition();
    }

  }

  /**
   * Handle mouse wheel events for slices --> Change the image number.
   * <p>
   * For consistency with other applications:
   * <ul>
   * <li>Mouse wheel forward decreases image number.
   * <li>Mouse wheel backward increases image number.
   * </ul>
   */
  private class OnMouseWheelSlice extends SoEventCallbackCB
  {
    private MedicalHelper.Axis m_axis;

    public OnMouseWheelSlice(MedicalHelper.Axis axis)
    {
      m_axis = axis;
    }

    @Override
    public void invoke(SoEventCallback cb)
    {
      int MOUSE_WHEEL_DELTA = 120; // True for all known mice...

      SoMouseWheelEvent theEvent = (SoMouseWheelEvent) cb.getEvent();
      int delta = theEvent.getDelta() / MOUSE_WHEEL_DELTA; // Should be -1 or +1

      // Note that RemoteViz does not set event position for wheel events.
      SoOrthoSlice sliceNode;
      switch ( m_axis )
      {
      case SAGITTAL :
        sliceNode = m_sagittalSlice;
        break;
      case CORONAL :
        sliceNode = m_coronalSlice;
        break;
      case AXIAL :
      default:
        sliceNode = m_axialSlice;
        break;
      }

      int curSlice = sliceNode.sliceNumber.getValue();
      int numSlices = m_volDim.getValueAt(sliceNode.axis.getValue());

      // If delta is negative, increment slice number, else decrement.
      // This arithmetic seems "backward" but it's the expected behavior.
      int newSlice = curSlice - delta;

      // Update if necessary.
      // SoOrthoSlice will "clamp" invalid values, so this check is not
      // strictly necessary, but it avoids triggering an unnecessary redraw.
      if ( newSlice < 0 )
        newSlice = 0;
      else if ( newSlice >= numSlices )
        newSlice = numSlices - 1;

      if ( newSlice != curSlice )
      {
        sliceNode.sliceNumber.setValue(newSlice);
        setVHLines(sliceNode.axis.getValue());
      }

      updateSliceNumText(sliceNode);
    }
  }

}
