// WaterLevel.java
//
// This charting example illustrates various techniques :
//
// - using a customized dragger as a "level of water" indicator
// - constraining a dragger
// - customizing axis with post-rebuild callback
// - "on-the-fly" picking with tooltip display
// - using PoSceneView to set window-relative annotations or titles
// - using engines and Open Inventor Java exceptions
//
//
package meshviz.graph.advanced.waterLevel;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec2f;
import com.openinventor.inventor.SbVec2s;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.SoPickedPoint;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.actions.SoRayPickAction;
import com.openinventor.inventor.draggers.SoDragger;
import com.openinventor.inventor.draggers.SoTranslate1Dragger;
import com.openinventor.inventor.engines.SoConcatenate;
import com.openinventor.inventor.engines.SoDecomposeVec3f;
import com.openinventor.inventor.events.SoLocation2Event;
import com.openinventor.inventor.fields.SoMFString;
import com.openinventor.inventor.gui.view.PoSceneView;
import com.openinventor.inventor.misc.callbacks.SoDraggerCB;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.* ;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.InteractionMode;
import com.openinventor.meshviz.graph.PoAxis;
import com.openinventor.meshviz.graph.PoBase;
import com.openinventor.meshviz.graph.PoCartesianAxis;
import com.openinventor.meshviz.graph.PoCartesianAxis.Types;
import com.openinventor.meshviz.graph.PoCurve;
import com.openinventor.meshviz.graph.PoLinearAxis;
import com.openinventor.meshviz.graph.PoRectangle;
import com.openinventor.meshviz.misc.callbacks.PoRebuildCB;
import com.openinventor.meshviz.nodes.PoDomain;
import com.openinventor.meshviz.nodes.PoMiscTextAttr;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  IViewerExaminer viewer = null;
  SoSeparator root = null;
  PoCurve curve = null;
  SbVec3f domainSize = null;
  SoPickStyle unpickablePickStyle=null;
  SoPickStyle shapePickStyle=null;
  PoDomain domain = null;
  SbColor redColor = null;
  SbColor blueColor = null;
  SbColor greenColor = null;
  SbColor yellowColor = null;
  SoAnnoText3 bubleText = null;
  SoSwitch bubleSwitch;
  SoTranslation bubleTransl;

  public void showBuble(SbVec3f pos, String label) {
      bubleTransl.translation.setValue(pos);
      bubleText.string.setValue(label);
      bubleSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
  }

  public void hideBuble() {
      bubleSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
  }

  class ListenMove extends SoEventCallbackCB {
    // Display bubble when the mouse is over curve points

    @Override
    public void invoke(SoEventCallback eventNode)
    {
      if (viewer.getRenderArea().getInteractionMode() == InteractionMode.NAVIGATION)
        return;

      SbVec2s eventPos = eventNode.getEvent().getPosition();
      Dimension size = viewer.getRenderArea().getComponent().getSize();
      SoRayPickAction rayAction = new SoRayPickAction(new SbViewportRegion(size));
      rayAction.setPoint(eventPos);
      rayAction.apply(viewer.getRenderArea().getSceneInteractor());
      SoPickedPoint point = rayAction.getPickedPoint();
      if (point != null) {
        SoPath path = point.getPath();
        if (path !=  null) {
          SoNode node = path.full.getTail();
          if (node instanceof SoPointSet) {
            SoGroup group = (SoGroup)path.full.getNodeFromTail(1);
            int childIndex = group.findChild(node);
            // This group contains : Coord, PointSet, Coord, PointSet...
            showBuble(point.getPoint(),"point_" + childIndex/2
                      + " = " + point.getPoint().getY());
            return;
          }
        }
      } else
          hideBuble();
    }
  }


  class CustomizeAxisCB extends PoRebuildCB {
    // Nice-looking axis : this callback replaces axis body and arrow by cylinder and cone

    @Override
    public void invoke(PoBase base) {
      PoLinearAxis axis = (PoLinearAxis)base;
      axis.set("bodyApp.lightModel", "model PHONG");
				// ... activates lighting effects. Default is BASE_COLOR for dataviz axis
      axis.set("bodyApp.material", "specularColor 0.7 0.7 0.7 shininess 0.2");
				// ... some shining

      SoGroup group = (SoGroup)axis.getPart("body");
      float bodySize = axis.end.getValue();
      float domainFactor= 1.f; // initialized to avoid warning
      PoCartesianAxis.Types axistype = axis.type.getValue(PoCartesianAxis.Types.class);
      switch ( axistype )
      {
      case XY :
      case XZ :
        domainFactor  = domainSize.getX();
        bodySize -= axis.start.getValue().getX();
        axis.set("bodyApp.material", "diffuseColor 1 0 0");
        break;

      case YX :
      case YZ :
        domainFactor  = domainSize.getY();
        bodySize -= axis.start.getValue().getY();
        axis.set("bodyApp.material", "diffuseColor 0 1 0");
        break;

      case ZX :
      case ZY :
        domainFactor  = domainSize.getZ();
        bodySize -= axis.start.getValue().getZ();
        axis.set("bodyApp.material", "diffuseColor 0 0 1");
        break;

      default:
        break;
      }

      float radius = axis.getTick().mainLength * domainFactor;
				// ... !! tick length is defined as a percentage of the domain

      SoCylinder cyl = new SoCylinder();
      cyl.height.setValue(bodySize);
      cyl.radius.setValue(radius/2.f);

      SoTransform transform = new SoTransform();
      transform.rotation.setValue(new SbVec3f(0, 0, 1), (float)(-Math.PI/2));
      transform.translation.setValue(bodySize/2, 0, 0);

				// Replace axis body shape (line) by ours
      group.removeAllChildren();
      group.addChild(transform);
      group.addChild(cyl);

				// Replace arrow if needed
      if(axis.arrowVisibility.getValue() != PoAxis.Visibilities.VISIBILITY_OFF.getValue()) {
        float arrowSize = axis.getArrow().length * domainFactor;
        float arrowRadius= axis.getArrow().height * domainFactor;
        SoCone cone = new SoCone();
        cone.bottomRadius.setValue(arrowRadius);
        cone.height.setValue(arrowSize);
        SoTranslation coneTranslation = new SoTranslation();
        coneTranslation.translation.setValue(0, bodySize/2.f + arrowSize/2.f,  0);
        group.addChild(coneTranslation);
        group.addChild(cone);
        SoGroup arrowGroup = (SoGroup) axis.getPart("arrow");
        arrowGroup.removeAllChildren();
      }
    }
  }

  @Override
  public void start () {
    viewer = ViewerComponentsFactory.createViewerExaminer();

    redColor   = new SbColor(1, 0, 0);
    greenColor = new SbColor(0, 1, 0);
    blueColor  = new SbColor(0, 0, 1);
    yellowColor  = new SbColor(1, 1, 0);

    domain = new PoDomain();
    domain.setValues(new SbVec3f(0, 0, 0), new SbVec3f(10, 10, 10), PoDomain.BoundingBoxTypes.AS_IS);


    ///////////////////////////////////////////////////////////////////
    // Build a customized translate dragger to be used as a water level

    // We assume here that Y axis is "vertical"
    // Build a rectangle with the right size depending on the domain
    // (remember that it will lay on XZ plane)

    PoRectangle rectangle = new PoRectangle();
    domainSize = new SbVec3f();
    domainSize.setValue(domain.max.getValue());
    domainSize.substract(domain.min.getValue());
    rectangle.p.setValue(0, -domainSize.getX());
    rectangle.q.setValue(domainSize.getZ(), 0);

    SoSeparator draggerTranslatorSep = new  SoSeparator();

    SoMaterial blueMaterial = new SoMaterial();
    blueMaterial.diffuseColor.setValue(blueColor);

    SoSeparator draggerActiveTranslatorSep = new SoSeparator();
    SoMaterial transparentBlueMaterial = new SoMaterial();
    transparentBlueMaterial.diffuseColor.setValue(blueColor);
    transparentBlueMaterial.transparency.setValue(0.5f);

    SoRotation rectangleRotation = new SoRotation ();
    rectangleRotation.rotation.setValue(new SbVec3f(0, 1, 0), (float)(-Math.PI/2.));
    PoDomain resetDomain = new PoDomain(); // default domain does nothing

    draggerTranslatorSep.addChild(resetDomain); // avoid to apply domain twice !
    draggerTranslatorSep.addChild(transparentBlueMaterial);
    draggerTranslatorSep.addChild(rectangleRotation);
    draggerTranslatorSep.addChild(rectangle);

    SoDrawStyle lineDrawStyle = new SoDrawStyle();
    lineDrawStyle.style.setValue(SoDrawStyle.Styles.LINES);

    draggerActiveTranslatorSep.addChild(resetDomain); // avoid to apply domain twice !
    draggerActiveTranslatorSep.addChild(transparentBlueMaterial);
    draggerActiveTranslatorSep.addChild(rectangleRotation);
    draggerActiveTranslatorSep.addChild(rectangle);
    draggerActiveTranslatorSep.addChild(blueMaterial);
    draggerActiveTranslatorSep.addChild(lineDrawStyle);
    draggerActiveTranslatorSep.addChild(rectangle);

    SoTranslate1Dragger waterLevelDragger = new SoTranslate1Dragger();
    waterLevelDragger.setPart("translator", draggerTranslatorSep);
    waterLevelDragger.setPart("translatorActive", draggerActiveTranslatorSep);

    // Contrain the dragger within domain's vertical limits
    waterLevelDragger.addValueChangedCallback(new SoDraggerCB() {
        @Override
        public void invoke(SoDragger dragger) {
          //boolean wasEnabled = dragger.isValueChangedCallbacksEnabled();
          dragger.enableValueChangedCallbacks(false); // prevent recursion
          SoTranslate1Dragger theDragger = (SoTranslate1Dragger)dragger;
          float low = domain.min.getValue().getY();
          float high = domain.max.getValue().getY();
          if(theDragger.translation.getValue().getX() < low )
            theDragger.translation.setValue(low, 0, 0);
          if(theDragger.translation.getValue().getX() > high)
            theDragger.translation.setValue(high, 0, 0);
          dragger.enableValueChangedCallbacks(/*wasEnabled*/true); // restore previous
        }
      },
                                              null);

    ///////////////////////////////////////////////////////////////////
    // Positioning the dragger within the scene :
    // the default translate dragger is along X axis, we need to rotate it
    SoSeparator draggerSeparator = new SoSeparator();
    SoTransform draggerTransform= new SoTransform();
    draggerTransform.rotation.setValue(new SbVec3f(0, 0, 1), (float)(Math.PI/2.));
    draggerTransform.translation.setValue(domain.min.getValue());

    draggerSeparator.addChild(domain.getTransform());
    // ... apply the domain's transform to our dragger to stay in the same coordinate system
    // as dataviz nodes (curves...)
    draggerSeparator.addChild(draggerTransform);
    draggerSeparator.addChild(waterLevelDragger);

    ///////////////////////////////////////
    // Title annotation in a separate view

    SoSeparator titleSep= new SoSeparator();
    SoText2 title = new SoText2();
    title.string.setValue("Comparing curves - level : 0");
    title.justification.setValue(SoText2.Justifications.CENTER);
    SoFont titleFont = new SoFont();
    titleFont.size.setValue(14);
    titleFont.name.setValue("Arial");
    titleSep.addChild(titleFont);
    titleSep.addChild(title);
    // ... default is to have the text centered on the origin

    PoSceneView titleView = new PoSceneView() ;
    titleView.viewportOrigin.setValue(0, 0) ;
    titleView.viewportSize.setValue(1.0f, 0.1f) ;
    titleView.setPart("camera", new SoOrthographicCamera()); // use default camera
    titleView.setPart("scene", titleSep) ;

    /////////////////////////////////////////////////////////////////////////////
    // Automatic update of title depending on the waterLevelDragger's translation
    // using engines
    SoDecomposeVec3f decomposeEngine= new SoDecomposeVec3f();
    decomposeEngine.vector.connectFrom(waterLevelDragger.translation);

    SoConcatenate concatEngine = new SoConcatenate(SoMFString.class);
    ((SoMFString)concatEngine.input[0]).setValue("Comparing curves");
    concatEngine.input[1].connectFrom(decomposeEngine.x);
    title.string.connectFrom(concatEngine.output);

    PoMiscTextAttr textAttr = new PoMiscTextAttr() ;
    textAttr.fontName.setValue("Courier New") ;

    // Nodes to control which objects can be picked
    unpickablePickStyle = new SoPickStyle();
    unpickablePickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);

    shapePickStyle = new SoPickStyle();
    shapePickStyle.style.setValue(SoPickStyle.Styles.SHAPE);

    // Curve's coordinates :
    SbVec2f[] curvePts= {new SbVec2f(0.5F,0.5F), new SbVec2f(1.5F,3.8F), new SbVec2f(1.8F,1.2F),
                         new SbVec2f(2.4F,1.9F), new SbVec2f(3.2F,6.3F), new SbVec2f(4.5F,7.2F),
                         new SbVec2f(6.3F,9.3F), new SbVec2f(6.9F,7.2F), new SbVec2f(8,6),
                         new SbVec2f(8.5F,6.3F), new SbVec2f(9,7.5F),    new SbVec2f(9.5F,4.1F),
                         new SbVec2f(9.8F,3.5F), new SbVec2f(10F,1.5F)} ;
    curve = new PoCurve(curvePts) ;

    curve.isCurveFilled.setValue(true) ;
    curve.markerFilterType.setValue(PoCurve.FilterTypes.ALL_POINTS) ;

    curve.set("markerApp.material", "diffuseColor 0.8 0.8 0.8") ;
    curve.set("markerApp.drawStyle", "pointSize 5.0") ;
    curve.set("curvePointApp.material", "diffuseColor 0.8 0.8 0.8") ;

    // To make sure that only points can be picked, we will make the rest of the scene
    // unpickable (so that object traversed before the points wont be picked instead).
    // We need then to force curve's points to be pickable, inserting a pick style nodes
    // whenever the curve is rebuild.
    curve.addPostRebuildCallback(
                                 new PoRebuildCB() {
                                     @Override
                                    public void invoke(PoBase base) {
                                       SoGroup markerGroup = (SoGroup) curve.getPart("marker");
                                       markerGroup.insertChild(shapePickStyle, 0);
                                     }
                                   },
                                 null );

    //////////////////////////////////////////////////////////////
    // Axis system

    PoLinearAxis axisX = new PoLinearAxis(new SbVec3f(0,0,10), 10, Types.XY);
    PoLinearAxis axisY = new PoLinearAxis(new SbVec3f(0,0,0), 10, Types.YX);
    PoLinearAxis axisZ = new PoLinearAxis(new SbVec3f(0,0,0), 10, Types.ZX);

    // Make sure that gradutation are reasonably readable
    axisZ.gradPath.setValue(PoAxis.TextPaths.PATH_UP);
    axisZ.gradPosition.setValue(PoAxis.GradPositions.GRAD_BELOW);

    // Axis customization
    axisX.addPostRebuildCallback(new CustomizeAxisCB(), null);
    axisY.addPostRebuildCallback(new CustomizeAxisCB(), null);
    axisZ.addPostRebuildCallback(new CustomizeAxisCB(), null);


    // Offset between curves
    SoTranslation curveTranslation = new SoTranslation();
    curveTranslation.translation.setValue(0.f,
                                          -0.1f*domainSize.getY(),
                                          0.25f*domainSize.getZ());
    // .. warning : this translation must be in world coordinates

    SoMaterial curveMaterial1 = new SoMaterial();
    curveMaterial1.diffuseColor.setValue(redColor);

    SoMaterial curveMaterial2 = new SoMaterial();
    curveMaterial2.diffuseColor.setValue(greenColor);

    SoMaterial curveMaterial3 = new SoMaterial();
    curveMaterial3.diffuseColor.setValue(yellowColor);

    SoSeparator curvesSep = new SoSeparator();

    curvesSep.addChild(curveTranslation);
    curvesSep.addChild(curveMaterial1);
    curvesSep.addChild(curve);

    curvesSep.addChild(curveTranslation);
    curvesSep.addChild(curveMaterial2);
    curvesSep.addChild(curve);

    curvesSep.addChild(curveTranslation);
    curvesSep.addChild(curveMaterial3);
    curvesSep.addChild(curve);

    SoAnnotation bubleSep = new SoAnnotation();
    bubleTransl = new SoTranslation();
    SoMaterial bubleMat = new SoMaterial();
    bubleMat.diffuseColor.setValue(1,1,1);
    SoFont bubleFont = new SoFont();
    bubleFont.size.setValue(0.8f);
    SoAnnoText3Property bubleProp = new SoAnnoText3Property();
    bubleProp.renderPrintType.setValue(SoAnnoText3Property.RenderPrintTypes.RENDER2D_PRINT_RASTER);
    bubleProp.fontSizeHint.setValue(SoAnnoText3Property.FontSizeHints.ANNOTATION);
    bubleText = new SoAnnoText3();
    bubleSwitch = new SoSwitch();
    bubleSwitch.addChild(bubleText);
    bubleSep.addChild(bubleTransl);
    bubleSep.addChild(bubleMat);
    bubleSep.addChild(bubleFont);
    bubleSep.addChild(bubleProp);
    bubleSep.addChild(bubleSwitch);


    // Create the root of the scene graph
    root = new SoSeparator();
    { // assemble scene graph
      root.addChild(textAttr);
      root.addChild(domain);
      root.addChild(new SoPolygonOffset()); // fix depth testing artifacts
      root.addChild(unpickablePickStyle);
      root.addChild(curvesSep);
      root.addChild(axisX);
      root.addChild(axisY);
      root.addChild(axisZ);
      root.addChild(titleView);
      root.addChild(shapePickStyle);
      root.addChild(draggerSeparator);
      root.addChild(bubleSep);
    }

    viewer.setSceneGraph(root) ;
    // ... highest quality transparency in most cases
    viewer.getRenderArea().setTransparencyType (SoGLRenderAction.TransparencyTypes.SORTED_OBJECT);

    final SoEventCallback eventCallback = new SoEventCallback();
    eventCallback.addEventCallback(SoLocation2Event.class, new ListenMove());
    viewer.getRenderArea().getSceneInteractor().insertChild(eventCallback, 0);

    viewer.viewAll();

    final Component component = viewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    viewer.dispose();
  }

  public static void main(String[] argv) {
    Main example = new Main();
    example.demoMain("WaterLevel");
  }
}
