package meshviz.mesh.advanced.meshViewer;

import com.openinventor.inventor.*;
import com.openinventor.meshviz.data.*;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.draggers.*;
import com.openinventor.inventor.misc.callbacks.*;

public class MeshStreamNode extends VectorRepresentNode {
  private static final int WIDTH_FACTOR = 200;
  private static final int DEF_MAX_STEP_NUM = 10000;
  private static final SbColor[] PARTICLE_COLORS = {
      new SbColor(1, 1, 1),
      new SbColor(1, 0, 0),
      new SbColor(0, 1, 0),
      new SbColor(0, 0, 1),
      new SbColor(1, 1, 0),
      new SbColor(0, 1, 1),
      new SbColor(1, 0, 1)
  };

  PoStreamLine m_meshStreamLines;
  PoStreamPointMotion m_meshStreamPoint;
  PoStreamSphereMotion m_meshStreamSphere;
  PoStreamLineMotion m_meshStreamLineParticle;
  PoStreamTadpoleMotion m_meshStreamTadpole;
  PoStreamSurface m_meshStreamSurface;
  SoSeparator m_streamSource;
  SoSwitch m_meshStreamRepSwitch;
  SoSwitch m_streamSourceSwitch;
  SoSwitch m_meshStreamSwitch;
  SoSeparator m_meshStreamSep;
  SoJackDragger m_streamJackDragger;
  SoTransformBoxDragger m_streamBoxDragger;
  SoSwitch m_streamDraggerSwitch;

  private SoNode[] m_inventorNodes;
  private String m_name;
  private int m_number;
  private MeshStreamPanel m_panel;
  private VectorNode m_parent;
  private SbVec3f m_planeNormal;
  public float m_tadpoleLengthFactor;
  public float m_sourceSizeFactor;
  public int m_numStartPoints;
  public int m_viewFrame;
  public int m_whichShapeType;
  public float m_maxs;
  public float m_maxVec;

  SbVec3f m_draggerPos;
  SbVec3f m_draggerScale;
  SbRotation m_draggerRotation;


  public MeshStreamNode(PbMesh mesh, SbBox3f bbox) {
    m_number = m_streamVector.size() + 1;
    m_name = "Mesh Stream " + m_number;

    m_planeNormal = new SbVec3f(0, 0, 1);
    m_sourceSizeFactor = 0.25f;
    m_numStartPoints = 7;
    m_whichShapeType = 0; // circle
    m_viewFrame = 0;

    SbVec3f bounding_box_center = bbox.getCenter();
    float[] bbox_size = bbox.getSize().array;
    m_maxs = bbox_size[0];
    if (bbox_size[1] > m_maxs)
      m_maxs = bbox_size[1];
    if (bbox_size[2] > m_maxs)
      m_maxs = bbox_size[2];

    m_maxVec = mesh.getMaxVecsSet(0);
    float time_step = 0.05f * m_maxs / m_maxVec;
    m_tadpoleLengthFactor = 0.1f * m_maxs / m_maxVec;

    // define StreamLine
    m_meshStreamLines = new PoStreamLine();
    m_meshStreamLines.integrationMaxStepNumber.setValue(DEF_MAX_STEP_NUM);
    m_meshStreamLines.colors.setValues(0, PARTICLE_COLORS);

    // points
    m_meshStreamPoint = new PoStreamPointMotion();
    m_meshStreamPoint.vecsIndex.setValue(0);
    m_meshStreamPoint.startPoints.connectFrom(m_meshStreamLines.startPoints);
    m_meshStreamPoint.pulseFrequency.setValue(15);
    m_meshStreamPoint.timeStep.setValue(time_step);
    m_meshStreamPoint.colors.connectFrom(m_meshStreamLines.colors);
    m_meshStreamPoint.integrationMaxStepNumber.connectFrom(m_meshStreamLines.
        integrationMaxStepNumber);

    // spheres
    m_meshStreamSphere = new PoStreamSphereMotion();
    m_meshStreamSphere.vecsIndex.setValue(0);
    m_meshStreamSphere.startPoints.connectFrom(m_meshStreamLines.startPoints);
    m_meshStreamSphere.pulseFrequency.connectFrom(m_meshStreamPoint.
                                                  pulseFrequency);
    m_meshStreamSphere.timeStep.connectFrom(m_meshStreamPoint.timeStep);
    m_meshStreamSphere.sphereRadius.setValue(m_maxs / 150);
    m_meshStreamSphere.colors.connectFrom(m_meshStreamLines.colors);
    m_meshStreamSphere.integrationMaxStepNumber.connectFrom(m_meshStreamLines.
        integrationMaxStepNumber);

    // Lines and particles
    m_meshStreamLineParticle = new PoStreamLineMotion();
    m_meshStreamLineParticle.vecsIndex.setValue(0);
    m_meshStreamLineParticle.startPoints.connectFrom(m_meshStreamLines.
        startPoints);
    m_meshStreamLineParticle.pulseFrequency.connectFrom(m_meshStreamPoint.
        pulseFrequency);
    m_meshStreamLineParticle.timeStep.connectFrom(m_meshStreamPoint.timeStep);
    m_meshStreamLineParticle.integrationMaxStepNumber.connectFrom(
        m_meshStreamLines.integrationMaxStepNumber);

    // Tadpoles
    m_meshStreamTadpole = new PoStreamTadpoleMotion();
    m_meshStreamTadpole.vecsIndex.setValue(0);
    m_meshStreamTadpole.startPoints.connectFrom(m_meshStreamLines.startPoints);
    m_meshStreamTadpole.pulseFrequency.connectFrom(m_meshStreamPoint.
        pulseFrequency);
    m_meshStreamTadpole.timeStep.connectFrom(m_meshStreamPoint.timeStep);
    m_meshStreamTadpole.lengthFactor.setValue(m_tadpoleLengthFactor);
    m_meshStreamTadpole.integrationMaxStepNumber.connectFrom(m_meshStreamLines.
        integrationMaxStepNumber);

    // Surface
    m_meshStreamSurface = new PoStreamSurface();
    m_meshStreamSurface.vecsIndex.setValue(0);
    m_meshStreamSurface.startPoints.set1Value(0, bounding_box_center);
    m_meshStreamSurface.rakeOrientation.set1Value(0, new SbVec3f(1, 0, 0));
    m_meshStreamSurface.rakeLength.setValue(2 * m_maxs / WIDTH_FACTOR);
    m_meshStreamSurface.numLinesPerRake.setValue(5);
    m_meshStreamSurface.integrationMaxStepNumber.connectFrom(m_meshStreamLines.
        integrationMaxStepNumber);
    m_meshStreamSurface.colors.connectFrom(m_meshStreamLines.colors);

    { // Stream representations
      m_meshStreamRepSwitch = new SoSwitch();
      m_meshStreamRepSwitch.addChild(m_meshStreamLines);
      m_meshStreamRepSwitch.addChild(m_meshStreamLineParticle);
      m_meshStreamRepSwitch.addChild(m_meshStreamTadpole);
      m_meshStreamRepSwitch.addChild(m_meshStreamPoint);
      m_meshStreamRepSwitch.addChild(m_meshStreamSphere);
      m_meshStreamRepSwitch.addChild(m_meshStreamSurface);
      m_meshStreamRepSwitch.whichChild.setValue(0);
    }

    // define stream sources representation
    SoMaterial material = new SoMaterial();
    material.diffuseColor.set1Value(0, 1.f, 0.f, 0.f);
    SoDrawStyle draw_style = new SoDrawStyle();
    draw_style.pointSize.setValue(4);
    SoCoordinate3 source_coord = new SoCoordinate3();
    source_coord.point.connectFrom(m_meshStreamLines.startPoints);
    SoPointSet source_points = new SoPointSet();
    source_points.numPoints.setValue(-1); // use all points in source_coord
    {
      m_streamSource = new SoSeparator();
      m_streamSource.addChild(material);
      m_streamSource.addChild(draw_style);
      m_streamSource.addChild(source_coord);
      m_streamSource.addChild(source_points);

      m_streamSourceSwitch = new SoSwitch();
      m_streamSourceSwitch.addChild(m_streamSource);
      m_streamSourceSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
    }

    // draggers
    float scale_factor = m_maxs / 10;
    m_draggerScale = new SbVec3f(scale_factor, scale_factor, scale_factor);
    m_draggerPos = bounding_box_center;
    m_draggerRotation = new SbRotation(new SbVec3f(1, 0, 0),
                                       (float)java.lang.Math.PI/2.f);

    // Jack dragger
    m_streamJackDragger = new SoJackDragger();
    m_streamJackDragger.scaleFactor.setValue(m_draggerScale);
    m_streamJackDragger.translation.setValue(m_draggerPos);
    m_streamJackDragger.rotation.setValue(m_draggerRotation);

    // Box dragger
    m_streamBoxDragger = new SoTransformBoxDragger();
    m_streamBoxDragger.scaleFactor.setValue(m_draggerScale);
    m_streamBoxDragger.translation.setValue(m_draggerPos);
    m_streamBoxDragger.rotation.setValue(m_draggerRotation);

    {
      m_streamDraggerSwitch = new SoSwitch();
      m_streamDraggerSwitch.addChild(m_streamJackDragger);
      m_streamDraggerSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
    }

    {
      m_meshStreamSwitch = new SoSwitch();
      m_meshStreamSwitch.addChild(m_streamDraggerSwitch);
      m_meshStreamSwitch.addChild(m_pickSwitch);
      m_meshStreamSwitch.addChild(m_streamSourceSwitch);
      m_meshStreamSwitch.addChild(m_meshStreamRepSwitch);
      m_meshStreamSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);

      m_meshStreamSep = new SoSeparator();
      m_meshStreamSep.addChild(m_meshStreamSwitch);
    }

    m_panel = new MeshStreamPanel(this);

    m_streamJackDragger.addValueChangedCallback(new JackDraggerValueChangedCB(),
                                                null);
    m_streamBoxDragger.addMotionCallback(new BoxDraggerMotionCB(), null);
    updateSources();

    m_inventorNodes = new SoNode[] {
        m_meshStreamLines, m_meshStreamPoint, m_meshStreamSphere,
        m_meshStreamLineParticle, m_meshStreamTadpole, m_meshStreamSurface,
        m_streamBoxDragger, m_streamJackDragger, source_points
    };
  }

  public void decreaseNumber() {
    m_number--;
    m_name = "Mesh Stream " + m_number;
    m_panel.changeTitleName(m_name);
  }

  public String toString() {
    return m_name;
  }

  public RepresentNodePanel getPanel() {
    return m_panel;
  }

  public SoNode[] addToParent(DataNode parent, boolean is_drop_action) {
    m_parent = (VectorNode)parent;
    m_parent.addStream(this);

    if (! is_drop_action)
      m_streamVector.add(this);

    return m_inventorNodes;
  }

  public SoNode[] delete(boolean is_drop_action) {
    m_parent.removeStream(this);

    if (! is_drop_action) {
      for (int i = m_number; i < m_streamVector.size(); i++)
        ((MeshStreamNode)m_streamVector.get(i)).decreaseNumber();
      m_streamVector.remove(m_number-1);
    }

    return m_inventorNodes;
  }

  public void show(boolean show) {
    if (show)
      m_meshStreamSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
    else
      m_meshStreamSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
  }

  public DataNode getDataNodeParent() {
    return m_parent;
  }

  public void setAnimationFrame() {
    m_meshStreamPoint.viewFrame.setValue(m_viewFrame);
    m_meshStreamSphere.viewFrame.setValue(m_viewFrame);
    m_meshStreamLineParticle.viewFrame.setValue(m_viewFrame);
    m_meshStreamTadpole.viewFrame.setValue(m_viewFrame);
  }

  private void updateSources() {
  	m_planeNormal = m_draggerRotation.multVec(new SbVec3f(0, 1, 0));

    // move start-points of streamline
    SbVec3f n = m_planeNormal;
    float[] n_values = n.getValue();
    if (n_values[2] != 0)
      n.setValue(-n_values[2], 0, n_values[0]);
    else if (n_values[0] != 0)
      n.setValue(-n_values[2], n_values[0], 0);
    else
      n.setValue(1, 0, 0);

    // move stream surface
    m_meshStreamSurface.startPoints.set1Value(0, m_draggerPos);
    m_meshStreamSurface.rakeOrientation.set1Value(0, n);
    m_meshStreamSurface.rakeLength.setValue(m_draggerScale.getX() *
                                             m_sourceSizeFactor);

    // move stream lines, steam points & stream spheres
    moveStreamSource();
  }

  public void moveStreamSource() {
    // move stream lines, steam points & stream spheres
    SbVec3f start_point = new SbVec3f();
    SbMatrix m = new SbMatrix();
    m.setTransform(m_draggerPos, m_draggerRotation, m_draggerScale);
    m_meshStreamLines.startPoints.setNum(m_numStartPoints);

    switch (m_whichShapeType) {
      case 0: // circle
        double alpha = 0;
        double d_alpha = 2 * 3.1415927 / m_numStartPoints;
        for (int i = 0; i < m_numStartPoints; i++, alpha += d_alpha) {
          start_point.setValue( (float) (m_sourceSizeFactor * Math.cos(alpha)),
                               0,
                               (float) (m_sourceSizeFactor * Math.sin(alpha)));
          start_point = m.multVecMatrix(start_point);
          m_meshStreamLines.startPoints.set1Value(i, start_point);
        }
        break;

      case 1: // line
        float xc = -m_sourceSizeFactor;
        float dx = 2 * m_sourceSizeFactor / (float) (m_numStartPoints - 1);
        for (int i = 0; i < m_numStartPoints; i++, xc += dx) {
          start_point.setValue(xc, 0, 0);
          start_point = m.multVecMatrix(start_point);
          m_meshStreamLines.startPoints.set1Value(i, start_point);
        }
        break;

      case 2: // box
        for (int i = 0; i < m_numStartPoints; i++) {
          float d_x = (float) Math.random() * m_draggerScale.getX() * 2.f *
              m_sourceSizeFactor;
          float d_y = (float) Math.random() * m_draggerScale.getY() * 2.f *
              m_sourceSizeFactor;
          float d_z = (float) Math.random() * m_draggerScale.getZ() * 2.f *
              m_sourceSizeFactor;

          float x = m_draggerPos.getX() - m_draggerScale.getX() *
              m_sourceSizeFactor + d_x;
          float y = m_draggerPos.getY() - m_draggerScale.getY() *
              m_sourceSizeFactor + d_y;
          float z = m_draggerPos.getZ() - m_draggerScale.getZ() *
              m_sourceSizeFactor + d_z;

          start_point.setValue(x, y, z);
          m_meshStreamLines.startPoints.set1Value(i, start_point);
        }
        break;
    }
  }

  class JackDraggerValueChangedCB extends SoDraggerCB {
    public void invoke(SoDragger d) {
      SoJackDragger dragger = (SoJackDragger) d;
      m_draggerPos = dragger.translation.getValue();
      m_draggerScale = dragger.scaleFactor.getValue();
      m_draggerRotation = dragger.rotation.getValue();

      // update box dragger
      m_streamBoxDragger.scaleFactor.setValue(m_draggerScale);
      m_streamBoxDragger.translation.setValue(m_draggerPos);
      m_streamBoxDragger.rotation.setValue(m_draggerRotation);

      // update stream sources
      updateSources();
    }
  }

  class BoxDraggerMotionCB extends SoDraggerCB {
    public void invoke(SoDragger d) {
      SoTransformBoxDragger dragger = (SoTransformBoxDragger) d;
      m_draggerPos = dragger.translation.getValue();
      m_draggerScale = dragger.scaleFactor.getValue();
      m_draggerRotation = dragger.rotation.getValue();

      // update jack dragger
      m_streamJackDragger.enableValueChangedCallbacks(false);
      m_streamJackDragger.scaleFactor.setValue(m_draggerScale);
      m_streamJackDragger.translation.setValue(m_draggerPos);
      m_streamJackDragger.rotation.setValue(m_draggerRotation);
      m_streamJackDragger.enableValueChangedCallbacks(true);

      // update stream sources
      updateSources();
    }
  }
}
