package meshviz.mesh.advanced.meshViewer;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;

import javax.swing.*;
import javax.swing.border.TitledBorder;

import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;

public class MeshStreamPanel extends RepresentNodePanel {
  private static final String[] STYLE_BOX_ITEMS = {
      "Line",
      "Line and Particle",
      "Tadpole",
      "Point",
      "Sphere",
      "Surface"
  };
  private static final String[] SHAPE_TYPE_ITEMS = {
    "Circle",
    "Line",
    "Box"
  };
  private static final ImageIcon MORE_ICON =
      new ImageIcon(MeshViewer.m_filePrefix+"data/more.jpg");
  private static final ImageIcon LESS_ICON =
      new ImageIcon(MeshViewer.m_filePrefix+"data/less.jpg");
  private static final ImageIcon PREV_ICON =
      new ImageIcon(MeshViewer.m_filePrefix+"data/prev.jpg");
  private static final ImageIcon NEXT_ICON =
      new ImageIcon(MeshViewer.m_filePrefix+"data/next.jpg");
  private static final ImageIcon PAUSE_ICON =
      new ImageIcon(MeshViewer.m_filePrefix+"data/pause.jpg");
  private static final ImageIcon PLAY_ICON =
      new ImageIcon(MeshViewer.m_filePrefix+"data/play.jpg");

  private DraggerPanel m_draggerPanel;
  private JComboBox<String> m_styleBox;
  private JComboBox<String> m_shapeTypeBox;
  private JButton m_prevFrameButton;
  private JButton m_blinkerButton;
  private JButton m_nextFrameButton;
  private JPanel m_sourcesPanel;
  private JPanel m_animPanel;
  private JButton m_moreButton;
  private SliderPanel m_tadpoleLength;
  private SliderPanel m_sphereRadius;
  private SliderPanel m_shapeSize;
  private MeshStreamNode m_node;

  private boolean m_animPanelVisible = false;

  public MeshStreamPanel(MeshStreamNode node) {
    super(node, node.toString());
    m_node = node;

    try {
      File help_file = new File(MeshViewer.m_filePrefix + "data/MeshStreamHelp.htm");
      m_helpPage = help_file.toURI().toURL();

      jbInit();
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  private void jbInit() throws Exception {
    m_visibilityButton.setToolTipText("Mesh Stream visibility");

    /********* Style *********/
    JLabel style_label = new JLabel("Style");
    m_styleBox = new JComboBox<String>(STYLE_BOX_ITEMS);
    m_styleBox.addActionListener(new StreamStyleListener());

    /********* Dragger panel *********/
    m_draggerPanel = new DraggerPanel(m_node.m_streamJackDragger,
                                     m_node.m_streamDraggerSwitch,
                                     m_node.m_draggerScale.getX());
    m_draggerPanel.m_draggerOn.setSelected(true);

    /********* Sources panel *********/
    Dimension label_dim = new Dimension(120, 20);
    // Shape type
    JLabel shape_type_label = new JLabel("Shape type");
    shape_type_label.setPreferredSize(new Dimension(label_dim.width-10,
                                                    label_dim.height));
    m_shapeTypeBox = new JComboBox<String>(SHAPE_TYPE_ITEMS);
    m_shapeTypeBox.addActionListener(new ShapeTypeListener());
    // Shape size
    m_shapeSize = new SliderPanel(0, 10.f, m_node.m_sourceSizeFactor, 3);
    m_shapeSize.addInfoText("Shape size");
    m_shapeSize.setInfoTextSize(label_dim);
    m_shapeSize.setBoxSelected(false);
    m_shapeSize.setTextFieldColumns(4);
    m_shapeSize.addSliderPanelListener(new SliderPanel.Listener() {
      public void stateChanged(float value) {
        m_node.m_sourceSizeFactor = value;
        m_node.moveStreamSource();
      }
    });
    // Shape number
    SliderPanel shape_number = new SliderPanel(0, 50, m_node.m_numStartPoints);
    shape_number.addInfoText("Start points number");
    shape_number.setInfoTextSize(label_dim);
    shape_number.setBoxSelected(false);
    shape_number.setTextFieldColumns(4);
    shape_number.addSliderPanelListener(new SliderPanel.Listener() {
       public void stateChanged(float value) {
         m_node.m_numStartPoints = (int)value;
         m_node.moveStreamSource();
       }
    });

    SpringLayout sp_layout = new SpringLayout();
    m_sourcesPanel = new JPanel(sp_layout);
    TitledBorder sp_border = new TitledBorder(BorderFactory.createRaisedBevelBorder(),
                                        "Sources");
    sp_border.setTitleJustification(TitledBorder.LEFT);
    m_sourcesPanel.setBorder(sp_border);
    m_sourcesPanel.add(shape_type_label);
    m_sourcesPanel.add(m_shapeTypeBox);
    m_sourcesPanel.add(m_shapeSize);
    m_sourcesPanel.add(shape_number);
    sp_layout.putConstraint(SpringLayout.NORTH, shape_type_label,
                            NORTH_INNER_PANEL_CONSTRAINT+3,
                            SpringLayout.NORTH, m_sourcesPanel);
    sp_layout.putConstraint(SpringLayout.WEST, shape_type_label, 5,
                            SpringLayout.WEST, m_sourcesPanel);
    sp_layout.putConstraint(SpringLayout.NORTH, m_shapeTypeBox,
                            NORTH_INNER_PANEL_CONSTRAINT,
                            SpringLayout.NORTH, m_sourcesPanel);
    sp_layout.putConstraint(SpringLayout.WEST, m_shapeTypeBox, 19,
                            SpringLayout.EAST, shape_type_label);
    sp_layout.putConstraint(SpringLayout.NORTH, m_shapeSize, 3,
                            SpringLayout.SOUTH, m_shapeTypeBox);
    sp_layout.putConstraint(SpringLayout.NORTH, shape_number, 3,
                            SpringLayout.SOUTH, m_shapeSize);
    sp_layout.putConstraint(SpringLayout.EAST, m_sourcesPanel, 27,
                            SpringLayout.EAST, m_shapeSize);
    sp_layout.putConstraint(SpringLayout.SOUTH, m_sourcesPanel, SOUTH_CONSTRAINT,
                            SpringLayout.SOUTH, shape_number);

    /********* Particle Animation panel *********/
    // Random start
    JCheckBox random_start = new JCheckBox("Random start");
    random_start.addItemListener(new RandomStartListener());
    random_start.setSelected(true);
    // Animation buttons
    m_prevFrameButton = new JButton(PREV_ICON);
    m_prevFrameButton.setBorder(BorderFactory.createRaisedBevelBorder());
    m_prevFrameButton.setToolTipText("Previous frame");
    m_prevFrameButton.setEnabled(false);
    m_prevFrameButton.addMouseListener(new PrevFrameListener());

    m_blinkerButton = new JButton(PAUSE_ICON);
    m_blinkerButton.setBorder(BorderFactory.createRaisedBevelBorder());
    m_blinkerButton.setToolTipText("Stop Animation");
    m_blinkerButton.addMouseListener(new BlinkerListener());

    m_nextFrameButton = new JButton(NEXT_ICON);
    m_nextFrameButton.setBorder(BorderFactory.createRaisedBevelBorder());
    m_nextFrameButton.setToolTipText("Next frame");
    m_nextFrameButton.setEnabled(false);
    m_nextFrameButton.addMouseListener(new NextFrameListener());

    JToolBar anim_buttons = new JToolBar();
    anim_buttons.add(m_prevFrameButton);
    anim_buttons.add(m_blinkerButton);
    anim_buttons.add(m_nextFrameButton);
    anim_buttons.setFloatable(false);

    // Particle time step
    float time_step = 0.05f * m_node.m_maxs / m_node.m_maxVec;
    SliderPanel time_step_slider = new SliderPanel(0.3f * time_step,
                                                   3.f * time_step,
                                                   time_step, 6);
    time_step_slider.addInfoText("Particle Time Step");
    time_step_slider.setInfoTextSize(label_dim);
    time_step_slider.setBoxSelected(false);
    time_step_slider.setTextFieldColumns(6);
    time_step_slider.addSliderPanelListener(new SliderPanel.Listener() {
      public void stateChanged(float value) {
        m_node.m_meshStreamPoint.timeStep.setValue(value);
      }
    });
    // Tadpole length
    m_tadpoleLength = new SliderPanel(m_node.m_tadpoleLengthFactor / 4.f,
                                      m_node.m_tadpoleLengthFactor * 7.f / 4.f,
                                      m_node.m_tadpoleLengthFactor, 6);
    m_tadpoleLength.addInfoText("Tadpole Length");
    m_tadpoleLength.setInfoTextSize(label_dim);
    m_tadpoleLength.setBoxSelected(false);
    m_tadpoleLength.setEnabled(false);
    m_tadpoleLength.setTextFieldColumns(6);
    m_tadpoleLength.addSliderPanelListener(new SliderPanel.Listener() {
      public void stateChanged(float value) {
        m_node.m_meshStreamTadpole.lengthFactor.setValue(value);
        m_node.m_tadpoleLengthFactor = value;
      }
    });
    // Sphere radius
    float init_radius = m_node.m_meshStreamSphere.sphereRadius.getValue();
    m_sphereRadius = new SliderPanel(init_radius / 4.f,
                                     init_radius * 7.f / 4.f,
                                     init_radius, 6);
    m_sphereRadius.addInfoText("Sphere Radius");
    m_sphereRadius.setInfoTextSize(label_dim);
    m_sphereRadius.setBoxSelected(false);
    m_sphereRadius.setEnabled(false);
    m_sphereRadius.setTextFieldColumns(6);
    m_sphereRadius.addSliderPanelListener(new SliderPanel.Listener() {
      public void stateChanged(float value) {
        m_node.m_meshStreamSphere.sphereRadius.setValue(value);
      }
    });

    SpringLayout ap_layout = new SpringLayout();
    m_animPanel = new JPanel(ap_layout);
    TitledBorder ap_border = new TitledBorder(BorderFactory.createRaisedBevelBorder(),
                                        "Particle Animation");
    ap_border.setTitleJustification(TitledBorder.LEFT);
    m_animPanel.setBorder(ap_border);
    m_animPanel.add(anim_buttons);
    m_animPanel.add(random_start);
    m_animPanel.add(time_step_slider);
    m_animPanel.add(m_tadpoleLength);
    m_animPanel.add(m_sphereRadius);
    ap_layout.putConstraint(SpringLayout.NORTH, anim_buttons,
                            NORTH_INNER_PANEL_CONSTRAINT-2,
                            SpringLayout.NORTH, m_animPanel);
    ap_layout.putConstraint(SpringLayout.WEST, anim_buttons, 3,
                            SpringLayout.WEST, m_animPanel);
    ap_layout.putConstraint(SpringLayout.NORTH, random_start,
                            NORTH_INNER_PANEL_CONSTRAINT,
                            SpringLayout.NORTH, m_animPanel);
    ap_layout.putConstraint(SpringLayout.WEST, random_start, 51,
                            SpringLayout.EAST, anim_buttons);
    ap_layout.putConstraint(SpringLayout.NORTH, time_step_slider, 3,
                            SpringLayout.SOUTH, anim_buttons);
    ap_layout.putConstraint(SpringLayout.NORTH, m_tadpoleLength, 3,
                            SpringLayout.SOUTH, time_step_slider);
    ap_layout.putConstraint(SpringLayout.NORTH, m_sphereRadius, 3,
                            SpringLayout.SOUTH, m_tadpoleLength);
    ap_layout.putConstraint(SpringLayout.EAST, m_animPanel, 5,
                            SpringLayout.EAST, m_tadpoleLength);
    ap_layout.putConstraint(SpringLayout.SOUTH, m_animPanel, SOUTH_CONSTRAINT,
                            SpringLayout.SOUTH, m_sphereRadius);

    // More Button
    m_moreButton = new JButton(MORE_ICON);
    m_moreButton.setPreferredSize(new Dimension(452, 20));

    m_moreButton.addMouseListener(new MouseAdapter() {
      public void mouseReleased(MouseEvent e) {
        if (!m_animPanelVisible) {
          m_moreButton.setIcon(LESS_ICON);
          addAnimPanel();
        }
        else {
          m_moreButton.setIcon(MORE_ICON);
          removeAnimPanel();
        }
      }
    });

    /********* Main panel *********/
    add(style_label);
    add(m_styleBox);
    add(m_draggerPanel);
    add(m_sourcesPanel);
    m_spLayout.putConstraint(SpringLayout.NORTH, style_label,
                             NORTH_CONSTRAINT+3,
                             SpringLayout.NORTH, this);
    m_spLayout.putConstraint(SpringLayout.WEST, style_label, 10,
                             SpringLayout.WEST, this);
    m_spLayout.putConstraint(SpringLayout.NORTH, m_styleBox, NORTH_CONSTRAINT,
                             SpringLayout.NORTH, this);
    m_spLayout.putConstraint(SpringLayout.WEST, m_styleBox, 20,
                             SpringLayout.EAST, style_label);
    m_spLayout.putConstraint(SpringLayout.NORTH, m_draggerPanel, 15,
                             SpringLayout.SOUTH, m_styleBox);
    m_spLayout.putConstraint(SpringLayout.NORTH, m_sourcesPanel, 5,
                             SpringLayout.SOUTH, m_draggerPanel);
    m_spLayout.putConstraint(SpringLayout.SOUTH, this, SOUTH_CONSTRAINT,
                             SpringLayout.SOUTH, m_sourcesPanel);
  }

  private void addMoreButton() {
    add(m_moreButton);
    m_spLayout.putConstraint(SpringLayout.NORTH, m_moreButton, 5,
                             SpringLayout.SOUTH, m_sourcesPanel);
    m_spLayout.putConstraint(SpringLayout.SOUTH, this, SOUTH_CONSTRAINT,
                             SpringLayout.SOUTH, m_moreButton);

    update();
  }

  private void removeMoreButton() {
    remove(m_moreButton);
    m_spLayout.putConstraint(SpringLayout.SOUTH, this, SOUTH_CONSTRAINT,
                             SpringLayout.SOUTH, m_sourcesPanel);

    update();
  }

  private void addAnimPanel() {
    add(m_animPanel);
    m_spLayout.putConstraint(SpringLayout.NORTH, m_animPanel, 5,
                             SpringLayout.SOUTH, m_moreButton);
    m_spLayout.putConstraint(SpringLayout.SOUTH, this, SOUTH_CONSTRAINT,
                             SpringLayout.SOUTH, m_animPanel);
    m_animPanelVisible = true;

    update();
  }

  private void removeAnimPanel() {
    remove(m_animPanel);
    m_spLayout.putConstraint(SpringLayout.SOUTH, this, SOUTH_CONSTRAINT,
                             SpringLayout.SOUTH, m_moreButton);
    m_moreButton.setIcon(MORE_ICON);
    m_animPanelVisible = false;

    update();
  }

  private void update() {
    validate();
    repaint();
    JSplitPane pane = (JSplitPane)getParent().getParent().getParent().getParent();
    pane.resetToPreferredSizes();
  }

  class ShapeTypeListener implements ActionListener {
    public void actionPerformed(ActionEvent e) {
      int selected_index = m_shapeTypeBox.getSelectedIndex();
      m_node.m_whichShapeType = selected_index;
      if (selected_index == 2) {
        m_node.m_streamDraggerSwitch.replaceChild(0,
                                                  m_node.m_streamBoxDragger);
        m_shapeSize.setSliderValue(1.f);
        m_draggerPanel.m_draggerOrient.setEnabled(false);
      }
      else {
        m_node.m_streamDraggerSwitch.replaceChild(0,
                                                  m_node.m_streamJackDragger);
        m_shapeSize.setSliderValue(0.25f);
        m_draggerPanel.m_draggerOrient.setEnabled(true);
      }

      m_node.moveStreamSource();
    }
  }

  class PrevFrameListener extends MouseAdapter {
    public void mouseReleased(MouseEvent e) {
      if (m_prevFrameButton.isEnabled()) {
        m_node.m_viewFrame--;
        if (m_node.m_viewFrame == 0)
          m_prevFrameButton.setEnabled(false);
        m_node.setAnimationFrame();
      }
    }
  }

  class NextFrameListener extends MouseAdapter {
    public void mouseReleased(MouseEvent e) {
      if (m_nextFrameButton.isEnabled()) {
        m_node.m_viewFrame++;
        if (m_node.m_viewFrame == 1)
          m_prevFrameButton.setEnabled(true);
        m_node.setAnimationFrame();
      }
    }
  }

  class BlinkerListener extends MouseAdapter {
    private boolean m_isBlinking = true;

    public void mouseReleased(MouseEvent e) {
      m_isBlinking = !m_isBlinking;
      m_node.m_meshStreamPoint.isBlinking.setValue(m_isBlinking);
      m_node.m_meshStreamSphere.isBlinking.setValue(m_isBlinking);
      m_node.m_meshStreamLineParticle.isBlinking.setValue(m_isBlinking);
      m_node.m_meshStreamTadpole.isBlinking.setValue(m_isBlinking);
      if (!m_isBlinking) {
        m_blinkerButton.setIcon(PLAY_ICON);
        m_blinkerButton.setToolTipText("Play Animation");
        m_node.setAnimationFrame();
        if (m_node.m_viewFrame > 0)
          m_prevFrameButton.setEnabled(true);
        m_nextFrameButton.setEnabled(true);
      }
      else {
        m_blinkerButton.setIcon(PAUSE_ICON);
        m_blinkerButton.setToolTipText("Stop Animation");
        m_prevFrameButton.setEnabled(false);
        m_nextFrameButton.setEnabled(false);
      }
    }
  }

  class StreamStyleListener implements ActionListener {
    public void actionPerformed(ActionEvent e) {
      int selected_index = m_styleBox.getSelectedIndex();
      m_node.m_meshStreamRepSwitch.whichChild.setValue(selected_index);
      if (selected_index == 2)
        m_tadpoleLength.setEnabled(true);
      else
        m_tadpoleLength.setEnabled(false);

      if (selected_index == 4)
        m_sphereRadius.setEnabled(true);
      else
        m_sphereRadius.setEnabled(false);

      if (selected_index != 0 && selected_index != 5)
        addMoreButton();
      else {
        removeAnimPanel();
        removeMoreButton();
      }
    }
  }

  class RandomStartListener implements ItemListener {
    public void itemStateChanged(ItemEvent e) {
      boolean state = (e.getStateChange() == ItemEvent.SELECTED);
      m_node.m_meshStreamPoint.isStartRandomized.setValue(state);
      m_node.m_meshStreamSphere.isStartRandomized.setValue(state);
      m_node.m_meshStreamLineParticle.isStartRandomized.setValue(state);
      m_node.m_meshStreamTadpole.isStartRandomized.setValue(state);
    }
  }
}
