package meshviz.mesh.advanced.meshViewer;

import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.dnd.*;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeCellRenderer;
import javax.swing.tree.TreePath;

import com.openinventor.inventor.SbVec2s;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoPickedPoint;
import com.openinventor.inventor.actions.SoRayPickAction;
import com.openinventor.inventor.events.SoEvent;
import com.openinventor.inventor.events.SoMouseButtonEvent;
import com.openinventor.inventor.fields.SoField;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.SoEventCallback;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.meshviz.nodes.PoCartesianGrid2D;
import com.openinventor.meshviz.nodes.PoIndexedMesh2D;
import com.openinventor.meshviz.nodes.PoParalCartesianGrid2D;
import com.openinventor.meshviz.nodes.PoPolarGrid2D;
import com.openinventor.meshviz.nodes.PoQuadrangleMesh2D;
import com.openinventor.meshviz.nodes.PoRegularCartesianGrid2D;
import com.openinventor.meshviz.nodes.PoTriangleMesh2D;

public class MeshViewTree extends JTree {
  private Hashtable<SoNode, TreePath> m_hashtable;

  private MeshViewerPopupMenus m_popupMenus;
  private MeshNode m_root;
  private JPanel m_settingsPanel;
  private int m_initDatasetIndex;

  String[] m_datasetNames;
  String[] m_vecsetNames;
  MeshViewer m_meshViewer;
  MeshViewTreeModel m_treeModel;
  boolean m_legendVisible;

  public MeshViewTree(MeshViewer mesh_viewer, JPanel settings_panel) {
    super();
    m_meshViewer = mesh_viewer;
    m_settingsPanel = settings_panel;
    m_initDatasetIndex = mesh_viewer.m_initDatasetIndex;
    m_legendVisible = true;
    m_hashtable = new Hashtable<SoNode, TreePath>();

    // set mouse event callback for pick action
    m_meshViewer.m_mouseEventCB.addEventCallback(SoMouseButtonEvent.class,
                                                 new MousePressCB());

    try {
      jbInit();
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  private void jbInit() throws Exception {
    boolean is_2D = false;

    // init dataset names
    m_datasetNames = new String[m_meshViewer.m_mesh.getNumValuesSet()];
    for (int i = 0; i < m_datasetNames.length; i++) {
      String name = m_meshViewer.m_mesh.getValuesSetName(i);
      if (name.length() == 0)
        m_datasetNames[i] = "Dataset " + i;
      else
        m_datasetNames[i] = name;
    }

    boolean has_vecs_set = (m_meshViewer.m_mesh.getNumVecsSet() > 0);
    if (has_vecs_set) {
      // init vecset names
      m_vecsetNames = new String[m_meshViewer.m_mesh.getNumVecsSet()];
      for (int i = 0; i < m_vecsetNames.length; i++) {
        String name = m_meshViewer.m_mesh.getVecsSetName(i);
        if (name.length() == 0)
          m_vecsetNames[i] = "Vecset " + i;
        else
          m_vecsetNames[i] = name;
      }
    }

    if (m_meshViewer.m_meshNode instanceof PoCartesianGrid2D ||
        m_meshViewer.m_meshNode instanceof PoParalCartesianGrid2D ||
        m_meshViewer.m_meshNode instanceof PoRegularCartesianGrid2D ||
        m_meshViewer.m_meshNode instanceof PoPolarGrid2D ||
        m_meshViewer.m_meshNode instanceof PoIndexedMesh2D ||
        m_meshViewer.m_meshNode instanceof PoQuadrangleMesh2D ||
        m_meshViewer.m_meshNode instanceof PoTriangleMesh2D)
      is_2D = true;

    // Drag and Drop
    DragSource drag_source = new DragSource();
    DragSource.getDefaultDragSource();
    drag_source.createDefaultDragGestureRecognizer(
        this, DnDConstants.ACTION_MOVE, new NodeDragGestureListener());

    DropTarget dropTarget = new DropTarget(this, new NodeDropTargetListener());

    // Init tree
    if (is_2D)
      m_root = new Mesh2DNode(m_meshViewer);
    else
      m_root = new Mesh3DNode(m_meshViewer.m_scene3D);

    m_treeModel = new MeshViewTreeModel(m_root);
    this.setModel(m_treeModel);
    this.setCellRenderer(new MeshViewTreeCellRenderer());
    this.addMouseListener(new PopupTrigger());
    this.addKeyListener(new DeleteListener());
    this.addTreeSelectionListener(new MeshViewTreeSelectionListener());

    // Popup menus
    m_popupMenus = new MeshViewerPopupMenus(this, is_2D, has_vecs_set);

    DataNode.init();
    if (is_2D)
      init2DTree(has_vecs_set);
    else
      init3DTree(has_vecs_set);
  }

  private void init2DTree(boolean has_vecs_set) {
    Scalar2DRepresentNode.init();
    // add scalar datasets
    for (int i = 0; i < m_datasetNames.length; i++) {
      Scalar2DNode scalar2d_node = new Scalar2DNode(m_meshViewer,
                                                    m_datasetNames[i], i);
      m_root.add(scalar2d_node);
    }

    if (has_vecs_set) {
      Vector2DRepresentNode.init();
        // add vector datasets
        for (int i = 0; i < m_vecsetNames.length; i++) {
          Vector2DNode vector2d_node = new Vector2DNode(m_meshViewer,
                                                        m_vecsetNames[i], i);
          m_root.add(vector2d_node);
        }
      }

    // add Mesh Filled and Mesh Contouring
    expandRow(0);
    setSelectionRow(m_initDatasetIndex+1);
    Scalar2DNode selected_node = (Scalar2DNode)getLastSelectedPathComponent();
    m_treeModel.addNode(selected_node, new MeshFilledNode(), false);
    m_treeModel.addNode(selected_node,
                        new MeshContouringNode(m_meshViewer.m_isoValuesSwitch),
                        false);

    expandAllTree();
    m_meshViewer.m_viewer.viewAll();
  }

  private void init3DTree(boolean has_vecs_set) {
    Scalar3DRepresentNode.init();
    // add scalar datasets
    for (int i = 0; i < m_datasetNames.length; i++) {
      Scalar3DNode scalar3d_node = new Scalar3DNode(m_meshViewer,
                                                    m_datasetNames[i], i);
      m_root.add(scalar3d_node);
    }

    if (has_vecs_set) {
      Vector3DRepresentNode.init();
      // vector datasets
      for (int i = 0; i < m_vecsetNames.length; i++) {
        Vector3DNode vector3d_node = new Vector3DNode(m_meshViewer,
                                                      m_vecsetNames[i], i);
        m_root.add(vector3d_node);
      }
    }

    // add Mesh skin, Mesh Level surface and Mesh cut plane
    expandRow(0);
    setSelectionRow(m_initDatasetIndex+1);
    Scalar3DNode selected_node = (Scalar3DNode)getLastSelectedPathComponent();
    m_treeModel.addNode(selected_node, new MeshSkinNode(), false);
    m_treeModel.addNode(selected_node,
                        new MeshLevelSurfNode(m_meshViewer.m_isoValuesSwitch,
                                              m_datasetNames),
                        false);
    m_treeModel.addNode(selected_node,
                        new MeshCutPlaneNode(m_meshViewer.m_bbox),
                        false);

    expandAllTree();
    m_meshViewer.m_viewer.viewAll();
  }

  public void expandAllTree() {
    for (int i = 0; i < this.getRowCount(); i++)
      this.expandRow(i);
  }

  // enable / disable connections for each Inventor nodes in hashtable
  public void enableConnections(boolean enable) {
    for (Enumeration<SoNode> e = m_hashtable.keys(); e.hasMoreElements();) {
      SoNode node = (SoNode)e.nextElement();
      SoField[] fields = node.getAllFields();
      int num_fields = fields.length;
      for (int i = 0; i < num_fields; i++) {
        SoField field = fields[i];
        if (field.isConnected())
          field.enableConnection(enable);
      }
    }
  }

  // PopupTrigger
  class PopupTrigger extends MouseAdapter {
    @Override
    public void mousePressed(MouseEvent e)
    {
      if ( e.isPopupTrigger() )
        // Motif case
        showPopupMenu(e);
    }

    @Override
    public void mouseReleased(MouseEvent e)
    {
      if ( e.isPopupTrigger() )
        // Windows case
        showPopupMenu(e);
    }

    private void showPopupMenu(MouseEvent e)
    {
      int x = e.getX();
      int y = e.getY();
      TreePath path = getPathForLocation(x, y);
      if ( path != null )
      {
        setSelectionPath(path);
        // show menu
        m_popupMenus.show((DefaultMutableTreeNode) path.getLastPathComponent(), x, y);
      }
    }
  }

  // Delete key
  class DeleteListener extends KeyAdapter {
    public void keyReleased(KeyEvent e) {
      if(e.getKeyCode() == KeyEvent.VK_DELETE) {
        Object node = getLastSelectedPathComponent();
        if (node instanceof RepresentNode) {
          m_treeModel.deleteNode((RepresentNode)node, false);
          expandAllTree();
        }
      }
    }
  }

  /*** Tree classes **/

  // Tree model
  class MeshViewTreeModel extends DefaultTreeModel {
    DefaultMutableTreeNode m_root;

    public MeshViewTreeModel(DefaultMutableTreeNode root) {
      super(root);
      m_root = root;
    }

    public void addNode(DataNode parent_node, RepresentNode new_node,
                        boolean is_drop_action) {
      int node_index;
      if (parent_node == m_root)
        node_index = 0;
      else
        node_index = parent_node.getChildCount();

      // add new node in scene graph
      SoNode[] inventor_nodes = new_node.addToParent(parent_node, is_drop_action);
      // add new node in tree
      insertNodeInto(new_node, parent_node, node_index);

      // update menu actions
      updateActions(new_node);

      // update tree
      reload();
      expandAllTree();
      setSelectionPath(new TreePath(new_node.getPath()));

      // put Inventor nodes and their tree path in hashtable
      for (int i = 0; inventor_nodes != null && i < inventor_nodes.length; i++)
        m_hashtable.put(inventor_nodes[i], new TreePath(new_node.getPath()));

    }

    public void deleteNode(RepresentNode node, boolean is_drop_action) {
      if (node == null || node.getParent() == null)
        return;

      DefaultMutableTreeNode previous_node = node.getPreviousNode();

      // delete node in scene graph
      SoNode[] inventor_nodes = node.delete(is_drop_action);
      // delete node in tree
      removeNodeFromParent(node);

      // update menu actions
      updateActions(node);

      // update tree
      reload();

      if (! is_drop_action) {
        setSelectionPath(new TreePath(previous_node.getPath()));

        // remove Inventor nodes from hashtable
        for (int i = 0; inventor_nodes != null && i < inventor_nodes.length; i++)
          m_hashtable.remove(inventor_nodes[i]);
      }
    }

    // update menu actions
    private void updateActions(RepresentNode node) {
      if (node instanceof MeshSkeletonNode) {
        m_popupMenus.m_skeletonAction.
            setEnabled(!m_popupMenus.m_skeletonAction.isEnabled());
        return;
      }
      if (node instanceof Mesh2DVecFieldNode) {
        m_popupMenus.m_mesh2DVecFieldAction.
            setEnabled(!m_popupMenus.m_mesh2DVecFieldAction.isEnabled());
        return;
      }
      if (node instanceof AltitudeNode) {
        m_popupMenus.m_altitudeAction.
            setEnabled(!m_popupMenus.m_altitudeAction.isEnabled());
        return;
      }
      if (node instanceof MeshLinesNode) {
        m_popupMenus.m_meshLinesAction.
            setEnabled(!m_popupMenus.m_meshLinesAction.isEnabled());
        return;
      }
      if (node instanceof MeshContouringNode) {
        m_popupMenus.m_meshContouringAction.
            setEnabled(!m_popupMenus.m_meshContouringAction.isEnabled());
        return;
      }
      if (node instanceof MeshFilledNode) {
        m_popupMenus.m_meshFilledAction.
            setEnabled(!m_popupMenus.m_meshFilledAction.isEnabled());
        return;
      }
      if (node instanceof MeshLimitNode) {
        m_popupMenus.m_meshLimitAction.
            setEnabled(!m_popupMenus.m_meshLimitAction.isEnabled());
        return;
      }
      if (node instanceof MeshSidesNode) {
        m_popupMenus.m_meshSidesAction.
            setEnabled(!m_popupMenus.m_meshSidesAction.isEnabled());
        return;
      }
    }
  }

  // Entry tree display
  class MeshViewTreeCellRenderer extends JLabel implements TreeCellRenderer {

    private final Color BK_COLOR = new Color(204, 204, 255);
    private final Color m_textColor = UIManager.getColor("Tree.TextForeground");

    protected boolean m_selected;

    public MeshViewTreeCellRenderer() {
      super();
      setOpaque(false);
    }

    public Component getTreeCellRendererComponent(JTree tree, Object value,
                                                  boolean sel, boolean expanded,
                                                  boolean leaf, int row, boolean hasFocus) {
      DefaultMutableTreeNode node = (DefaultMutableTreeNode)value;
      setText(node.toString());
      Font tree_font = tree.getFont();
      if(node instanceof RepresentNode)
        setFont(tree_font);
      else // DataNode
        setFont(new Font(tree_font.getName(), Font.BOLD, tree_font.getSize()));
      setForeground(m_textColor);
      m_selected = sel;
      return this;
    }

    public void paintComponent(Graphics g) {
      if (m_selected) {
        // draw selection rectangle
        g.setColor(BK_COLOR);
        g.fillRect(0, 0, getWidth(), getHeight());
      }
      super.paintComponent(g);
    }
  }

  // Tree Selection Listener
  class MeshViewTreeSelectionListener implements TreeSelectionListener {
    public void valueChanged(TreeSelectionEvent e) {
      DefaultMutableTreeNode selected_node =
          (DefaultMutableTreeNode) e.getPath().getLastPathComponent();

      int which_dataset;
      if (selected_node instanceof RepresentNode) {
        // change settings panel
        RepresentNode rn = (RepresentNode) selected_node;
        JPanel node_panel = rn.getPanel();
        m_settingsPanel.removeAll();
        m_settingsPanel.add(node_panel);

        // get dataset for legend
        which_dataset = rn.getDataNodeParent().getWhichDataset();

      } else { // DataNode
        // remove settings panel
        m_settingsPanel.removeAll();

        // get dataset for legend
        DataNode dn = (DataNode) selected_node;
        which_dataset = dn.getWhichDataset();
      }

      // update legend
      if (which_dataset == -1)
        m_meshViewer.m_dataLegendSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
      else {
        m_meshViewer.m_dataSwitch.whichChild.setValue(which_dataset);
        if (m_legendVisible)
          m_meshViewer.m_dataLegendSwitch.whichChild.setValue(SoSwitch.
              SO_SWITCH_ALL);
      }
    }
  }

  /*** Drag and Drop classes ***/
  class NodeDragSourceAdapter extends DragSourceAdapter {
    public void dragOver(DragSourceDragEvent e) {
      Point p = e.getLocation();
      Point tree_pt = getLocationOnScreen();
      TreePath path = getPathForLocation(p.x - tree_pt.x, p.y - tree_pt.y);

      if (path != null
          && path.getLastPathComponent() instanceof DataNode)
        setSelectionPath(path);
    }
  }

  class NodeDragGestureListener implements DragGestureListener {
    public void dragGestureRecognized(DragGestureEvent e) {
      // cursor
      Toolkit tk = Toolkit.getDefaultToolkit();
      Cursor cursor = tk.createCustomCursor(Toolkit.getDefaultToolkit().
                                            createImage(MeshViewer.m_filePrefix +
          "data/hand.gif"), new Point(), "DragCursor");

      // get selected paths
      TreePath[] selected_paths = getSelectionPaths();
      if (selected_paths == null)
        return;

      Vector<RepresentNode> transferable_nodes_vector = new Vector<RepresentNode>();
      for (int i = 0; i < selected_paths.length; i++) {
        Object selected_node = selected_paths[i].getLastPathComponent();
        if (selected_node != null) {
          // move only RepresentNode
          if (selected_node instanceof RepresentNode)
            transferable_nodes_vector.addElement((RepresentNode)selected_node);
          else
            removeSelectionPath(selected_paths[i]);
        }
      }

      if (transferable_nodes_vector.isEmpty())
        return;

      RepresentNode[] transferable_nodes = new RepresentNode[transferable_nodes_vector.size()];
      Object[] tmp = transferable_nodes_vector.toArray();
      for (int j = 0; j < tmp.length; j++)
        transferable_nodes[j] = (RepresentNode)tmp[j];

      Transferable transferable = new NodeTransferable(transferable_nodes);
      e.startDrag(cursor, transferable, new NodeDragSourceAdapter());
    }
  }

  class NodeDropTargetListener extends DropTargetAdapter {

    // Check compatibility between transfered node and his new parent
    private boolean checkCompatibility(DataNode parent_node,
                                       RepresentNode new_node) {
      if (new_node instanceof AltitudeNode ||
          new_node instanceof MeshSidesNode ||
          new_node instanceof MeshLimitNode)
        return false;
      if (parent_node instanceof ScalarNode &&
          ! (new_node instanceof ScalarRepresentNode))
        return false;
      if (parent_node instanceof VectorNode &&
          ! (new_node instanceof VectorRepresentNode))
        return false;
      if (parent_node instanceof MeshNode &&
          (new_node instanceof MeshContouringNode ||
           new_node instanceof VectorRepresentNode))
        return false;

      return true;
    }

    public void drop(DropTargetDropEvent e) {
      try {
        Transferable transferable = e.getTransferable();
        if (transferable.isDataFlavorSupported(NodeTransferable.getNodeFlavor())) {
          // get transferable nodes
          RepresentNode[] nodes = NodeTransferable.getNodes();
          // get target
          Point p = e.getLocation();
          TreePath target = getPathForLocation(p.x, p.y);
          // check target
          final String message = checkDropTarget(target);
          if (message != null) {
            e.rejectDrop();

            SwingUtilities.invokeLater(new Runnable() {
              public void run() {
                JOptionPane.showMessageDialog(null, message, "Error Dialog",
                    JOptionPane.ERROR_MESSAGE);
              }
            });
            return;
          }

          DataNode new_parent = (DataNode)target.getLastPathComponent();
          MeshViewTreeModel model = (MeshViewTreeModel)getModel();

          // move nodes
          for (int i = 0; i < nodes.length; i++) {
            if (checkCompatibility(new_parent, nodes[i])) {
              // delete node
              model.deleteNode(nodes[i], true);
              // add nodes
              model.addNode(new_parent, nodes[i], true);
            }
          }

          e.acceptDrop(DnDConstants.ACTION_MOVE);
          e.getDropTargetContext().dropComplete(true);
          expandAllTree();
        }
      }
      catch (Exception exc) {
        exc.printStackTrace();
        e.rejectDrop();
      }
    }

    private String checkDropTarget(TreePath target) {
      if (target == null)
        return "No target selected";
      if (target.getLastPathComponent() instanceof RepresentNode)
        return "Not allowed operation";

      return null;
    }
  }

  class MousePressCB extends SoEventCallbackCB {
    public void invoke (SoEventCallback cb) {
      SoEvent event = cb.getEvent();

      // Check for mouse button being pressed
      if (SoMouseButtonEvent.isButtonPressEvent(event, SoMouseButtonEvent.Buttons.ANY)) {
        // SoRayPickAction to get point in the scene graph.
        SoRayPickAction picked_action =
            new SoRayPickAction(new SbViewportRegion(m_meshViewer.m_viewer.getRenderArea().getComponent().getSize()));
        SbVec2s event_pos = event.getPosition();
        picked_action.setPoint(event_pos);
        picked_action.apply(m_meshViewer.m_scene3D);
        SoPickedPoint picked_point = picked_action.getPickedPoint(0);

        if (picked_point != null) { // There is something.
          SoNode selected_node = (picked_point.getPath()).regular.getTail();

          // get tree path in hashtable and select it
          TreePath path = (TreePath)m_hashtable.get(selected_node);
          if (path != null)
            setSelectionPath(path);
        }
      }
    }
  }
} // Fin MyTree

// Node Transfer operation for drag and drop
class NodeTransferable implements Transferable {

  static final DataFlavor nodeFlavor = new DataFlavor(RepresentNode.class,
      "Representation Node");

  // DataFlavor supported
  static DataFlavor m_flavors[] = {
      nodeFlavor
  };

  // transferable nodes
  static RepresentNode[] m_nodes;

  public NodeTransferable(RepresentNode[] nodes) {
    m_nodes = nodes;
  }

  public DataFlavor[] getTransferDataFlavors() {
    return m_flavors;
  }

  public boolean isDataFlavorSupported(DataFlavor df) {
    return df.equals(nodeFlavor);
  }

  public Object getTransferData(DataFlavor df) throws
      UnsupportedFlavorException, IOException {

    if (df.equals(nodeFlavor)) {
      return m_nodes;
    }
    else {
      throw new UnsupportedFlavorException(df);
    }
  }

  public static DataFlavor getNodeFlavor() {
    return nodeFlavor;
  }

  public static RepresentNode[] getNodes() {
    return m_nodes;
  }
}
