package meshviz.mesh.advanced.meshViewer;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ContainerEvent;
import java.awt.event.ContainerListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.File;
import java.net.URL;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;
import javax.swing.filechooser.FileFilter;
import javax.swing.text.html.HTMLDocument;
import javax.swing.text.html.HTMLFrameHyperlinkEvent;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.actions.SoGLRenderAction.TransparencyTypes;
import com.openinventor.inventor.nodes.SoSwitch;

public class MeshViewTreeFrame extends JFrame {
  public static HelpFrame m_helpFrame;

  private MeshViewTree m_tree;
  private JPanel m_settingsPanel;
  private JFrame m_settingsFrame;
  private JMenuItem m_splitItem;
  private JMenuItem m_mergeItem;
  private JSplitPane m_splitPane;
  private JScrollPane m_treeScrollPane;
  private JScrollPane m_panelScrollPane;
  private BackgroundColorDialog m_bkgColorChooser;

  MeshViewer m_meshViewer;

  public MeshViewTreeFrame(MeshViewer mesh_viewer) {
    m_meshViewer = mesh_viewer;
    m_settingsFrame = null;
    m_helpFrame = new HelpFrame();
    m_bkgColorChooser = null;

    try {
      jbInit();
    }
    catch (Exception e) {
      e.printStackTrace();
    }

    Toolkit toolkit = Toolkit.getDefaultToolkit();
    toolkit.setDynamicLayout(true);
    Dimension screen_size = toolkit.getScreenSize();

    pack();
    setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    setLocation(screen_size.width-getWidth(), 0);
    setTitle("Mesh Viewer");
  }

  private void jbInit() throws Exception {
    // settings panel
    m_settingsPanel = new JPanel();
    m_panelScrollPane = new JScrollPane(m_settingsPanel);

    // tree
    m_tree = new MeshViewTree(m_meshViewer, m_settingsPanel);
    m_treeScrollPane = new JScrollPane(m_tree);

    // split pane
    m_splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT, true,
                                 m_treeScrollPane, m_panelScrollPane);
    m_splitPane.setOneTouchExpandable(true);
    m_splitPane.setResizeWeight(0.25);

    m_settingsPanel.addContainerListener(new ContainerListener() {
      public void componentAdded(ContainerEvent e) {
        m_splitPane.resetToPreferredSizes();
      }

      public void componentRemoved(ContainerEvent e) {
        m_splitPane.resetToPreferredSizes();
      }
    });

    this.getContentPane().add(m_splitPane);
    this.setJMenuBar(createMenuBar());
  }

  /*** Menu bar ***/
  private JMenuBar createMenuBar() {
    JMenu file_menu = createFileMenu();
    JMenu options_menu = createOptionsMenu();
    JMenu help_menu = createHelpMenu();

    JMenuBar menu_bar = new JMenuBar();
    menu_bar.add(file_menu);
    menu_bar.add(options_menu);
    menu_bar.add(help_menu);

    return menu_bar;
  }

  private JMenu createFileMenu() {
    JMenuItem open_file = new JMenuItem("Open file (*.iv)");
    open_file.addActionListener(new OpenIvFileListener());

    JMenuItem save_ascii = new JMenuItem("Save as ASCII");
    save_ascii.addActionListener(new SaveIvFileListener(false));

    JMenuItem save_binary = new JMenuItem("Save as binary");
    save_binary.addActionListener(new SaveIvFileListener(true));

    JMenuItem export_vector = new JMenuItem("Export Vector");
    export_vector.addActionListener(new ExportVectorListener());

    JMenu file_menu = new JMenu("File");
    file_menu.add(open_file);
    file_menu.add(save_ascii);
    file_menu.add(save_binary);
    file_menu.add(export_vector);

    return file_menu;
  }

  private JMenu createOptionsMenu() {
    // Transparency type submenu
    JRadioButtonMenuItem blend =
        new JRadioButtonMenuItem(new TransparencyAction("Blend", SoGLRenderAction.TransparencyTypes.NO_SORT));
    blend.setSelected(true);
    JRadioButtonMenuItem sorted_object_blend =
        new JRadioButtonMenuItem(new TransparencyAction("Sorted object blend", SoGLRenderAction.TransparencyTypes.SORTED_OBJECT));
    JRadioButtonMenuItem sorted_pixels_blend =
        new JRadioButtonMenuItem(new TransparencyAction("Sorted pixels blend", SoGLRenderAction.TransparencyTypes.SORTED_PIXEL));

    ButtonGroup trans_type_group = new ButtonGroup();
    trans_type_group.add(blend);
    trans_type_group.add(sorted_object_blend);
    trans_type_group.add(sorted_pixels_blend);

    JMenu transparency_type = new JMenu("Transparency type");
    transparency_type.add(blend);
    transparency_type.add(sorted_object_blend);
    transparency_type.add(sorted_pixels_blend);

    // Legend
    JCheckBoxMenuItem legend_item = new JCheckBoxMenuItem("Legend");
    legend_item.setSelected(true);
    legend_item.addItemListener(new LegendListener());

    // background
    JMenuItem background_item = new JMenuItem("Change background");
    background_item.addActionListener(new BackgroundListener());

    // Split / Merge
    m_splitItem = new JMenuItem("Split Window");
    m_mergeItem = new JMenuItem("Merge Window");
    m_splitItem.addActionListener(new SplitListener(this));
    m_mergeItem.setEnabled(false);
    m_mergeItem.addActionListener(new MergeListener(this));

    // Options menu
    JMenu options_menu = new JMenu("Options");
    options_menu.add(legend_item);
    options_menu.add(background_item);
    options_menu.addSeparator();
    options_menu.add(transparency_type);
    options_menu.addSeparator();
    options_menu.add(m_splitItem);
    options_menu.add(m_mergeItem);

    return options_menu;
  }

  private JMenu createHelpMenu() {
    JMenuItem help_item = new JMenuItem("Help");
    help_item.addActionListener(new HelpListener());

    JMenu help_menu = new JMenu("?");
    help_menu.add(help_item);

    return help_menu;
  }

  public void dispose() {
    super.dispose();
    if (m_settingsFrame != null)
      m_settingsFrame.dispose();
    if (m_helpFrame.isVisible())
      m_helpFrame.dispose();
  }

  class SaveIvFileListener extends AbstractAction {
    private boolean m_binary;

    public SaveIvFileListener (boolean binary) {
      m_binary = binary;
    }

    public void actionPerformed(ActionEvent e) {
      JFileChooser file_chooser = new JFileChooser(new File("."));
      file_chooser.setMultiSelectionEnabled(false);
      file_chooser.setFileFilter(new IvFileFilter());
      int return_val = file_chooser.showSaveDialog(MeshViewTreeFrame.this);
      if (return_val == JFileChooser.APPROVE_OPTION) {
        // disables the connection of each field of the viewers. This is done to
        // allow the SoDB::readAll to read correctly this iv file. In fact, when a
        // field is connected to a field of a MeshViz class derived from SoBaseKit
        // (eg PoMeshSkin), SoDB::readAll do not run correctly.
        m_tree.enableConnections(false);
        m_meshViewer.saveIvFile(file_chooser.getSelectedFile().getAbsolutePath(),
                                m_binary);
        m_tree.enableConnections(true);
      }
    }
  }

  class OpenIvFileListener extends AbstractAction {
    public void actionPerformed(ActionEvent e) {
      JFileChooser file_chooser = new JFileChooser(new File("."));
      file_chooser.setMultiSelectionEnabled(false);
      file_chooser.setFileFilter(new IvFileFilter());
      int return_val = file_chooser.showOpenDialog(MeshViewTreeFrame.this);

      if (return_val == JFileChooser.APPROVE_OPTION) {
        if (m_meshViewer.loadIvFile(file_chooser.getSelectedFile().getAbsolutePath())) {
          // build new tree
          m_tree = new MeshViewTree(m_meshViewer, m_settingsPanel);
          m_treeScrollPane = new JScrollPane(m_tree);
          m_splitPane.setLeftComponent(m_treeScrollPane);
          validate();
          repaint();
        }
      }
    }
  }

  class ByExtensionFileFilter extends FileFilter {
    protected String m_extension;
    protected String m_description;

    public ByExtensionFileFilter(String extension, String description) {
      m_extension = extension;
      m_description = description;
    }

    public boolean accept(File f) {
      if (f.isDirectory())
        return true;

      String file_name = f.getName();
      int point_index = file_name.indexOf('.');
      if (point_index != -1) {
        String extension = file_name.substring(point_index,
                                               file_name.length());
        if (extension.toLowerCase().equals(m_extension))
          return true;
      }
      return false;
    }

    public String getDescription() {
      return m_description;
    }
  }

  class IvFileFilter extends ByExtensionFileFilter {
    public IvFileFilter() {
      super(".iv", "Inventor File (*.iv)");
    }
  }

  class ExportVectorListener extends AbstractAction {
    public void actionPerformed(ActionEvent e) {
      JFileChooser file_chooser = new JFileChooser(new File("."));
      file_chooser.setMultiSelectionEnabled(false);

      file_chooser.addChoosableFileFilter(new VectorizeCGMFileFilter());
      file_chooser.addChoosableFileFilter(new VectorizeHPGLFileFilter());
      file_chooser.addChoosableFileFilter(new VectorizePSFileFilter());

      int return_val = file_chooser.showOpenDialog(MeshViewTreeFrame.this);

      if (return_val == JFileChooser.APPROVE_OPTION) {
        ( (VectorizeFileFilter) file_chooser.getFileFilter()).vectorize(
            file_chooser.getSelectedFile().getAbsolutePath());
      }
    }
  }

  abstract class VectorizeFileFilter extends ByExtensionFileFilter {
    public VectorizeFileFilter(String extension, String description) {
      super(extension, description);
    }
    abstract public void vectorize(String filename);
  }

  class VectorizeCGMFileFilter extends VectorizeFileFilter {
    public VectorizeCGMFileFilter() {
      super(".cgm", "CGM Files (*.cgm)");
    }
    public void vectorize(String filename) {
      m_meshViewer.vectorizeCGM(filename);
    }
  }

  class VectorizeHPGLFileFilter extends VectorizeFileFilter {
    public VectorizeHPGLFileFilter() {
      super(".hpgl", "HPGL/2 Files (*.hpgl)");
    }
    public void vectorize(String filename) {
      m_meshViewer.vectorizeHPGL(filename);
    }
  }

  class VectorizePSFileFilter extends VectorizeFileFilter {
    public VectorizePSFileFilter() {
      super(".ps", "PostScript Files (*.ps)");
    }
    public void vectorize(String filename) {
      m_meshViewer.vectorizePS(filename);
    }
  }

  class TransparencyAction extends AbstractAction {
    private TransparencyTypes m_whichType;
    public TransparencyAction(String name, TransparencyTypes screenDoor) {
      super(name);
      m_whichType = screenDoor;
    }

    public void actionPerformed(ActionEvent e) {
      m_meshViewer.m_viewer.getRenderArea().setTransparencyType(m_whichType);
    }
  }

  class LegendListener implements ItemListener {
    public void itemStateChanged(ItemEvent e) {
      if (e.getStateChange() == ItemEvent.SELECTED) {
        m_meshViewer.m_dataLegendSwitch.whichChild.
            setValue(SoSwitch.SO_SWITCH_ALL);
        m_tree.m_legendVisible = true;
      }
      else {
        m_meshViewer.m_dataLegendSwitch.whichChild.
            setValue(SoSwitch.SO_SWITCH_NONE);
        m_tree.m_legendVisible = false;
      }
    }
  }

  class SplitListener implements ActionListener {
    private JFrame m_treeFrame;

    public SplitListener(JFrame tree_frame) {
      m_treeFrame = tree_frame;
    }

    public void actionPerformed(ActionEvent e) {
      m_treeFrame.remove(m_splitPane);
      m_treeFrame.getContentPane().add(m_treeScrollPane);
      m_treeFrame.setSize(new Dimension(m_treeScrollPane.getWidth()+5,
                                        m_treeScrollPane.getHeight()+40));
      m_treeFrame.validate();
      m_treeFrame.repaint();

      m_settingsFrame = new JFrame("Settings");
      m_settingsFrame.getContentPane().add(m_panelScrollPane);
      m_settingsFrame.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
      m_settingsFrame.pack();
      m_settingsFrame.setLocation(m_treeFrame.getX(),
                                  m_treeFrame.getY() + m_treeFrame.getHeight()+5);
      m_settingsFrame.setVisible(true);

      m_mergeItem.setEnabled(true);
      m_splitItem.setEnabled(false);
    }
  }

  class MergeListener implements ActionListener {
    private JFrame m_treeFrame;

    public MergeListener(JFrame tree_frame) {
      m_treeFrame = tree_frame;
    }

    public void actionPerformed(ActionEvent e) {
      m_settingsFrame.dispose();
      m_settingsFrame = null;

      m_treeFrame.remove(m_treeScrollPane);
      m_splitPane.setLeftComponent(m_treeScrollPane);
      m_splitPane.setRightComponent(m_panelScrollPane);
      m_treeFrame.getContentPane().add(m_splitPane);
      m_treeFrame.pack();
      m_treeFrame.validate();
      m_treeFrame.repaint();

      m_mergeItem.setEnabled(false);
      m_splitItem.setEnabled(true);
    }
  }

  class HelpListener implements ActionListener {
    private URL m_helpPage;

    public HelpListener() {
      File help_file = new File(MeshViewer.m_filePrefix + "data/MeshViewerHelp.htm");
      try {
        m_helpPage = help_file.toURI().toURL();
      }
      catch (Exception e) {
        e.printStackTrace();
      }
    }

    public void actionPerformed(ActionEvent e) {
      if (m_helpPage != null)
        MeshViewTreeFrame.m_helpFrame.setPage(m_helpPage);
    }
  }

  class BackgroundListener implements ActionListener {
    public void actionPerformed(ActionEvent e) {
      if (m_bkgColorChooser == null)
        m_bkgColorChooser = new BackgroundColorDialog();

      m_bkgColorChooser.m_colorChooserDialog.setVisible(true);
    }
  }

  // Help Frame
  class HelpFrame extends JFrame {
    private JEditorPane m_editorPane;

    public HelpFrame() {
      super();

      m_editorPane = new JEditorPane();
      m_editorPane.setEditable(false);

      JScrollPane scroll_pane = new JScrollPane(m_editorPane);
      scroll_pane.setPreferredSize(new Dimension(550, 250));
      this.getContentPane().add(scroll_pane);

      Toolkit toolkit = Toolkit.getDefaultToolkit();
      toolkit.setDynamicLayout(true);
      Dimension screen_size = toolkit.getScreenSize();

      setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
      setLocation(screen_size.width/4, screen_size.height/2);
      setTitle("Help");

      m_editorPane.addHyperlinkListener(new Hyperactive());
    }

    public void setPage(URL page) {
      if (page != null) {
        try {
          this.m_editorPane.setPage(page);
          if (!this.isVisible()) {
            this.pack();
            this.setVisible(true);
          } else
            this.requestFocus();
        }
        catch (Exception e) {
          e.printStackTrace();
        }
      }
    }
  }

  class Hyperactive implements HyperlinkListener {

    public void hyperlinkUpdate(HyperlinkEvent e) {
      if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
        JEditorPane pane = (JEditorPane) e.getSource();
        if (e instanceof HTMLFrameHyperlinkEvent) {
          HTMLFrameHyperlinkEvent evt = (HTMLFrameHyperlinkEvent) e;
          HTMLDocument doc = (HTMLDocument) pane.getDocument();
          doc.processHTMLFrameHyperlinkEvent(evt);
        }
        else {
          try {
            pane.setPage(e.getURL());
          }
          catch (Throwable t) {
            t.printStackTrace();
          }
        }
      }
    }
  }


  // Background Color Chooser dialog
  class BackgroundColorDialog {
    private JColorChooser m_backgroundChooser;
    private PreviewPanel m_previewPanel;
    private Color m_initColor0;
    private Color m_initColor1;

    public JDialog m_colorChooserDialog;

    public BackgroundColorDialog() {
      m_initColor0 = m_meshViewer.m_background.color0.getValue().toAWTColor();
      m_initColor1 = m_meshViewer.m_background.color1.getValue().toAWTColor();

      m_backgroundChooser = new JColorChooser();
      m_previewPanel = new PreviewPanel();
      m_backgroundChooser.setPreviewPanel(new JPanel()); // remove default preview panel
      m_backgroundChooser.setColor(m_initColor0);

      m_colorChooserDialog = JColorChooser.createDialog(MeshViewTreeFrame.this,
          "Background color",
          true, m_backgroundChooser,
          m_previewPanel, m_previewPanel);

      // add preview panel
      m_colorChooserDialog.getContentPane().add(m_previewPanel,
                                                BorderLayout.NORTH);
      m_colorChooserDialog.setSize(new Dimension(440, 450));
    }

    class PreviewPanel
        extends JPanel
        implements ChangeListener, ActionListener {
      private JCheckBox m_previewBox;
      private JCheckBox m_color0Box;
      private JCheckBox m_color1Box;
      private JPanel m_color0Preview;
      private JPanel m_color1Preview;

      public PreviewPanel() {
        m_previewBox = new JCheckBox("Preview");
        m_previewBox.addItemListener(new PreviewListener());

        // color 0 panel
        m_color0Box = new JCheckBox("Color 0");
        m_color0Box.setSelected(true);
        m_color0Preview = new JPanel();
        m_color0Preview.setBorder(BorderFactory.createLoweredBevelBorder());
        m_color0Preview.setPreferredSize(new Dimension(100, 10));
        m_color0Preview.setOpaque(true);
        m_color0Preview.setBackground(m_initColor0);

        JPanel color0_panel = new JPanel(new BorderLayout());
        color0_panel.add(m_color0Box, BorderLayout.WEST);
        color0_panel.add(m_color0Preview, BorderLayout.CENTER);

        // color 1 panel
        m_color1Box = new JCheckBox("Color 1");
        m_color1Preview = new JPanel();
        m_color1Preview.setBorder(BorderFactory.createLoweredBevelBorder());
        m_color1Preview.setPreferredSize(new Dimension(100, 10));
        m_color1Preview.setOpaque(true);
        m_color1Preview.setBackground(m_initColor1);

        JPanel color1_panel = new JPanel(new BorderLayout());
        color1_panel.add(m_color1Box, BorderLayout.WEST);
        color1_panel.add(m_color1Preview, BorderLayout.CENTER);

        ButtonGroup colors_group = new ButtonGroup();
        colors_group.add(m_color0Box);
        colors_group.add(m_color1Box);

        // preview panel
        SpringLayout sp_layout = new SpringLayout();
        setLayout(sp_layout);
        add(m_previewBox);
        add(color0_panel);
        add(color1_panel);
        sp_layout.putConstraint(SpringLayout.NORTH, m_previewBox, 5,
                                SpringLayout.NORTH, this);
        sp_layout.putConstraint(SpringLayout.NORTH, color0_panel, 5,
                                SpringLayout.SOUTH, m_previewBox);
        sp_layout.putConstraint(SpringLayout.NORTH, color1_panel, 5,
                                SpringLayout.SOUTH, color0_panel);
        sp_layout.putConstraint(SpringLayout.SOUTH, this, 15,
                                SpringLayout.SOUTH, color1_panel);

        m_backgroundChooser.getSelectionModel().addChangeListener(this);
      }

      // ColorChooser color changed
      public void stateChanged(ChangeEvent e) {
        Color color = m_backgroundChooser.getColor();
        if (m_color0Box.isSelected()) {
          // change panel color
          m_color0Preview.setBackground(color);
          if (m_previewBox.isSelected())
            // change background color
            m_meshViewer.m_background.color0.setValue(new SbColor(color));
        }
        else {
          // change panel color
          m_color1Preview.setBackground(color);
          if (m_previewBox.isSelected())
            // change background color
            m_meshViewer.m_background.color1.setValue(new SbColor(color));
        }
      }

      public void actionPerformed(ActionEvent e) {
        if (e.getActionCommand() == "OK") {
          if (!m_previewBox.isSelected()) {
            // change background color
            m_previewBox.setSelected(true);
            m_previewBox.setSelected(false);
          }
          m_initColor0 = m_color0Preview.getBackground();
          m_initColor1 = m_color1Preview.getBackground();
        }
        if (e.getActionCommand() == "cancel") {
          // restore panels color
          m_color0Preview.setBackground(m_initColor0);
          m_color1Preview.setBackground(m_initColor1);
          if (m_previewBox.isSelected()) {
            // restore background color
            m_meshViewer.m_background.color0.setValue(new SbColor(m_initColor0));
            m_meshViewer.m_background.color1.setValue(new SbColor(m_initColor1));
          }
        }
      }

      class PreviewListener
          implements ItemListener {
        public void itemStateChanged(ItemEvent e) {
          if (e.getStateChange() == ItemEvent.SELECTED) {
            Color color0 = m_color0Preview.getBackground();
            Color color1 = m_color1Preview.getBackground();
            m_meshViewer.m_background.color0.setValue(new SbColor(color0));
            m_meshViewer.m_background.color1.setValue(new SbColor(color1));
          }
        }
      }
    }
  } // end BackgroundColorDialog
}
