package meshviz.mesh.sample.dialog;

import java.awt.*;
import java.awt.event.*;

/**
  *  PoAwtChoiceButtonData
  *
  * Class to define a choice button. This element is composed of a label on the
  * left and by a choice button on the right. By default the text of the label is
  * an empty string, the number of items and the selected item are equal to zero.
  *
  * @author Rodolphe Albout
  * @author Laurent ISTIN
  * @author Loic Vigneras
  * @author Patrick Vigneras
  */
public class PoAwtChoiceButtonData extends PoAwtElementData {

  // Constants
  // ---------

  // Number of pixels between the label and the choice button
  private static final int MARGIN_OFFSET = 6;

  // Constructors
  // ------------

  /** Construct a new PoAwtChoiceButtonData with no label and no items.
    */
  public PoAwtChoiceButtonData () {
    this (null, null, 0);
  }

  /** Construct a new PoAwtChoiceButtonData with the given label and no items.
    */
  public PoAwtChoiceButtonData (String label) {
    this (label, null, 0);
  }

  /** Construct a new PoAwtChoiceButtonData with no label and the given items.
    * The current item is the first one (indexed as 0).
    */
  public PoAwtChoiceButtonData (String[] itemNames) {
    this (null, itemNames, 0);
  }

  /** Construct a new PoAwtChoiceButtonData with the given label and the given
    * items. The current item is the first one (indexed as 0).
    */
  public PoAwtChoiceButtonData (String label, String[] itemNames) {
    this (label, itemNames, 0);
  }

  /** Construct a new PoAwtChoiceButtonData with no label, the given items,
    * and define the current item. If this one is out of bounds, it is set to zero.
    */
  public PoAwtChoiceButtonData (String[] itemNames, int selectedItem) {
    this (null, itemNames, selectedItem);
  }

  /** Construct a new PoAwtChoiceButtonData with the given label, the given
    * items, and define the current item. If this one is out of bounds, it is
    * set to zero.
    */
  public PoAwtChoiceButtonData (String label, String[] itemNames, int selectedItem) {
    super ();
    _label = new PoAwtLabelData (label, PoAwtLabelData.ALIGNMENT_END);
    //    _choice = new Choice ();
    //    linkInnerComponents ();
    setItems (itemNames);    // initialize also _choice, and link inner components
    setSelectedItem (selectedItem);
  }

  /** Construct a new PoAwtChoiceButtonData with the same values as the given one.
    */
  protected PoAwtChoiceButtonData (PoAwtChoiceButtonData choice) {
    super ();
    _label = new PoAwtLabelData (choice.getLabel (), PoAwtLabelData.ALIGNMENT_END);
    //    _choice = new Choice ();
    //    linkInnerComponents ();
    setItems (choice.getItems ());   // initialize also _choice, and link components
    setSelectedItem (choice.getSelectedItem ());
    setID (choice.getID ());
  }

  // Methods
  // -------

  /** Set the text of the label on the left of choice button.
    */
  public void setLabel (String label) {
    _label.setLabel (label);
  }

  /** Get the text of the label on the left of choice button.
    */
  public String getLabel () {
    return _label.getLabel ();
  }

  /** Set items names of the choice button. Items must be not null. If an item
    * is null, it is not inserted.
    */
  public void setItems (String[] itemNames) {
    // to avoid NullPointerException
    if (itemNames == null)
      itemNames = new String[0];

    _choice = new Choice ();
    //    _choice.removeAll ();    CAUTION: not existing in AWT 1.0
    for (int i = 0; i < itemNames.length; i++) {
      try {
	_choice.addItem (itemNames[i]);
      }
      catch (NullPointerException e) {
      }
    }
    linkInnerComponents ();   // rebuild element, with the new _choice
  }
  /** Get items names of the choice button.
    */
  public String[] getItems () {
    String[] result = new String[_choice.getItemCount()];
    for (int i = 0; i < result.length; i++)
      result[i] = _choice.getItem (i);
    return result;
  }

  /** Set the selected item number. If it is out of bounds, it is set to zero.
    */
  public void setSelectedItem (int itemNumber) {
    if (_choice.getItemCount() != 0) {
      try {
	_choice.select (itemNumber);
      }
      catch (IllegalArgumentException e) {
      }
    }
  }

  /** Get the selected item number.
    */
  public int getSelectedItem () {
    return _choice.getSelectedIndex ();
  }

  /** Returns the type of the element.
    */
  public int getType () {
    return CHOICE_BUTTON;
  }

  /** Creates and returns a new element which is a copy of this element.
    */
  protected Object clone () {
    return new PoAwtChoiceButtonData (this);
  }

  /** Affect the internal data awtElement to current object.
    * The current object and awtElement have to be the same object type.
    * This method should be called by PoAwtDialog, and not by user code
    * directly.
    */
  public void updateValues (PoAwtElementData element) {
    if (element.getType () == CHOICE_BUTTON) {
      PoAwtChoiceButtonData choice = (PoAwtChoiceButtonData) element;

      setLabel        (choice.getLabel ());
      setItems        (choice.getItems ());
      setSelectedItem (choice.getSelectedItem ());
      setID           (choice.getID ());
    }
  }

  /** Minimum width for the label of the element.
    * The mimimum width for a elementid with no label will be 0 (Default behaviour).
    * The width is the size in pixel.
    * This method should be called by PoAwtDialog, and not by user code
    * directly.
    */
  public int getMinLabelWidth () {
    return _label.getLength ();
  }

  /** This is call when the dialog is resized, It's use for resize some awtElement.
    * This method should be called by PoAwtDialog, and not by user code
    * directly.
    */
  public void setMinLabelWidth (int minWidth) {
    _label.setLength (minWidth);
  }

  // Layout and link inner components to the panel of this element
  private void linkInnerComponents () {
    this.removeAll ();
    GridBagLayout gridbag = new GridBagLayout ();
    this.setLayout (gridbag);
    GridBagConstraints c = new GridBagConstraints ();
    c.fill = GridBagConstraints.BOTH;
    c.weightx = 0.0;
    c.weighty = 1.0;
    gridbag.setConstraints (_label, c);
    this.add (_label);
    c.weightx = 1.0;
    c.insets = new Insets (0, MARGIN_OFFSET, 0, MARGIN_OFFSET);
    gridbag.setConstraints (_choice, c);
    this.add (_choice);
    _choice.addItemListener(new ItemListener() {
      public void itemStateChanged(ItemEvent e) {
        invokeValueChanged();
      }
    });
  }

  // Fields
  // ------

  // Inner components
  private PoAwtLabelData _label;
  private Choice _choice;      // includes the items and the current item

}
