package meshviz.mesh.sample.dialog;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.LayoutManager;
import java.awt.Panel;
import java.util.ArrayList;
import java.util.List;

/**
  *  PoAwtDialog
  *
  * Class to build a generalized dialog box which can contain real sliders,
  * integer sliders, toggle buttons, choice buttons, trigger buttons, labels,
  * and editable texts. The type of the elements (called Awt elements) contained
  * in the dialog box are specified by the application and all these elements
  * are disposed underneath each other. The Awt elements in the dialog box are
  * indexed from 0 (at the top of the dialog box) to N - 1 (at the bottom of the
  * dialog box), N is the number of Awt elements. One or several callback
  * objects can be attached to this dialog box. These callbacks are called each
  * time one of its Awt elements receives a keyboard or a mouse event.
  *
  * @author Rodolphe Albout
  * @author Laurent ISTIN
  * @author Loic Vigneras
  * @author Patrick Vigneras
  */
public class PoAwtDialog extends Panel {

  // Constants
  // ---------

  // Increment for _maxWidthLabel
  private static final int WIDTH_LABEL_OFFSET = 15;

  // Constructors
  // ------------

  /** Construct a new PoAwtDialog with no name and no Awt elements.
    */
  public PoAwtDialog () {
    this (null, null);
  }

  /** Construct a new PoAwtDialog with the given name, but no Awt elements.
    */
  public PoAwtDialog (String name) {
    this (name, null);
  }

  /** Construct a new PoAwtDialog with the given name and the given Awt elements.
    * @param name Name of this dialog box.
    * @param elts List of the Awt elements to be built inside the dialog box.
    */
  public PoAwtDialog (String name, PoAwtElementData[] elts) {
    super ();
    //    setName (name);
    _dialogPanel = this;    // valid if no scrollbar;
                            // otherwise '_dialogPanel' is included into 'this'
    // _vbar = null;   NOT USED currently
    _listElements = new PoAwtElementData[0];
    _elementListeners = new ElementDataListener[0];
    _listeners = new ArrayList<Listener>();
    _maxWidthLabel = new int[1];
    _maxWidthLabel[0] = 0;
    setValues (elts);
  }

  // Methods
  // -------

  // Manipulation of values

  /** Set the list of Awt elements to build inside the dialog box.
    */
  public void setValues (PoAwtElementData[] elts) {
    int i;
    // to avoid NullPointerException
    if (elts == null)
      elts = new PoAwtElementData[0];

    // Remove old listeners
    for (i = 0; i < _listElements.length; i++)
      _listElements[i].removeListener(_elementListeners[i]);

    // Create new elements
    _listElements = new PoAwtElementData [elts.length];
    _elementListeners = new ElementDataListener[elts.length];

    for (i = 0; i < elts.length; i++) {
      _listElements[i] = elts[i].copy();

      // Element listener
      _elementListeners[i] = new ElementDataListener(i);
      _listElements[i].addListener(_elementListeners[i]);
    }

    buildDialog ();
  }

  /** Get the list of Awt elements built inside the dialog box. The number of Awt
    * elements can be get by using the <CODE>length</CODE> field of the returned
    * array.
    */
  public PoAwtElementData[] getValues () {
    return _listElements;
  }

  /** Set the <CODE>index</CODE>'th Awt element in the dialog box.
    */
  public void set1Value (int index, PoAwtElementData elt) {

    // Check the index
    if ((index < 0) || (index >= _listElements.length))
      return;

    // Remove old listener, replace element, and add new listener
    _listElements[index].removeListener(_elementListeners[index]);
    _listElements[index] = elt.copy ();
    _elementListeners[index] = new ElementDataListener(index);
    _listElements[index].addListener(_elementListeners[index]);

    // Reset the dialog
    buildDialog ();
  }

  /** Get the <CODE>index</CODE>'th Awt element in the dialog box. The application
    * can call all the methods of PoAwtElementData class (and its derived class),
    * but be careful. Changing a label by using the method setLabel can cause the
    * dialog box to no longer be aligned. In this case, use the set1Value method to
    * force the dialog box to be re-aligned (see example 2 below).
    * <PRE>
    *
    *    Example 1:
    *    // Example to change the selected item of a choice button to 2
    *    // using the method set1Value
    *    ...
    *    PoAwtChoiceButtonData choice = (PoAwtChoiceButtonData)
    *      dialog.get1Value (CHOICE_INDEX);
    *    choice.setSelectedItem (2);
    *    ...
    *    <P>
    *
    *    Example 2:
    *    // Example to change the selected item of a choice button to 2
    *    // using the method set1Value
    *    ...
    *    PoAwtChoiceButtonData choice = (PoAwtChoiceButtonData)
    *      dialog.get1Value (CHOICE_INDEX).copy ();
    *    choice.setSelectedItem (2);
    *    dialog.set1Value (CHOICE_INDEX, (PoAwtElementData) choice);
    *    ...
    * </PRE>
    */
  public PoAwtElementData get1Value (int index) {
    if ((index < 0) || (index >= _listElements.length))
      return null;
    else
      return _listElements[index];
  }

  /** Inserts the specified Awt element at the index <CODE>index</CODE>.
    */
  public void insert1Value (int index, PoAwtElementData elt) {
    PoAwtElementData[] elts = new PoAwtElementData[1];
    elts[0] = elt;
    insertNValues (index, elts);
  }

  /** Inserts a list of Awt elements from the index <CODE>index</CODE>.
    */
  public void insertNValues (int index, PoAwtElementData[] elts) {
    int i;
    PoAwtElementData[] oldListElements = _listElements;
    ElementDataListener[] oldEltListeners = _elementListeners;

    // if no elements to insert, there's nothing to do !!!
    if (elts == null)
      return;

    // Check the index
    if (index < 0)
      index = 0;
    else
      if (index > _listElements.length)
	index = _listElements.length;

    // Create new arrays
    _listElements = new PoAwtElementData [oldListElements.length + elts.length];
    _elementListeners = new ElementDataListener [oldListElements.length + elts.length];

    // Copy first elements
    for (i = 0; i < index; i++) {
      _listElements[i] = oldListElements[i];
      _elementListeners[i] = oldEltListeners[i];
    }
    // Insert new elements
    for ( ; i < index + elts.length; i++) {
      _listElements[i] = elts[i - index].copy ();
      _elementListeners[i] = new ElementDataListener(i);
      _listElements[i].addListener(_elementListeners[i]);
    }
    // Copy and shift forward last elements (listeners element index must be corrected)
    // NB: old_i is the current index in the old array
    for ( int old_i = index; i < _listElements.length; i++, old_i++ )
    {
      _listElements[i] = oldListElements[old_i];
      oldEltListeners[old_i].setElementIndex(i);
      _elementListeners[i] = oldEltListeners[old_i];
    }

    buildDialog ();
  }

  /** Deletes the Awt element at the index <CODE>index</CODE>.
    */
  public void delete1Value (int index) {
    deleteNValues (index, 1);
  }
  /** Deletes <CODE>numElt</CODE> Awt elements from the index <CODE>index</CODE>.
    */
  public void deleteNValues (int index, int numElt) {
    int i;
    PoAwtElementData[] oldListElements = _listElements;
    ElementDataListener[] oldEltListeners = _elementListeners;
    // NB: _listElement and _elementListeners should be rebuild
    // because the size of Java arrays can't be changed directly.

    // Check index and numElt
    if ((index < 0) || (index >= _listElements.length) || (numElt <= 0))
      return;

    // Create new arrays
    _listElements = new PoAwtElementData [oldListElements.length - numElt];
    _elementListeners = new ElementDataListener[oldListElements.length - numElt];

    // Copy first elements
    for (i = 0; i < index; i++) {
      _listElements[i] = oldListElements[i];
      _elementListeners[i] = oldEltListeners[i];
    }
    // Copy and shift backward last elements (listeners element index must be corrected)
    for ( int old_i = index + numElt; (i < _listElements.length) && (old_i < oldListElements.length); i++, old_i++ )
    {
      _listElements[i] = oldListElements[old_i];
      oldEltListeners[old_i].setElementIndex(i);
      _elementListeners[i] = oldEltListeners[old_i];
    }

    buildDialog ();
  }

  // Listeners
  public void addListener(Listener listener)
  {
    if ( listener != null )
      _listeners.add(listener);
  }

  public void removeListener(Listener listener)
  {
    _listeners.remove(listener);
  }

  private void invokeValueChanged(int elementIndex)
  {
    for ( Listener listener : _listeners )
      listener.valueChanged(elementIndex);
  }

  // Misc

  private void buildDialog () {
    GridBagLayout gridbag = new GridBagLayout ();
    GridBagConstraints c = new GridBagConstraints ();

    // Remove old components and set the layout
    _dialogPanel.removeAll ();
    _dialogPanel.setLayout (gridbag);
    c.fill = GridBagConstraints.BOTH;
    c.weightx = 1.0;
    c.weighty = 1.0;
    c.gridwidth = GridBagConstraints.REMAINDER;

    // Determine _maxWidthLabel
    setMaxWidthLabel ();

    // Link elements to _dialogPanel
    for (int i = 0; i < _listElements.length; i++) {
      /* _listElements[i].setMinLabelWidth (_maxWidthLabel); */
      // The first component of each element (its label) must have the same
      // width, which is _maxWidthLabel.
      LayoutManager layout = _listElements[i].getLayout();
      if (layout instanceof GridBagLayout)
	((GridBagLayout) layout).columnWidths = _maxWidthLabel;

      gridbag.setConstraints (_listElements[i], c);
      _dialogPanel.add (_listElements[i]);
    }
    this.repaint ();
  }

  public void addNotify () {
    super.addNotify ();
    setMaxWidthLabel ();
  }

  private void setMaxWidthLabel () {
    _maxWidthLabel[0] = 0;
    for (int i = 0; i < _listElements.length; i++) {
      int widthLabel = _listElements[i].getMinLabelWidth ();
      if (_maxWidthLabel[0] < widthLabel)
	_maxWidthLabel[0] = widthLabel;
    }
    _maxWidthLabel[0] += WIDTH_LABEL_OFFSET;
  }

  public String getDefaultName () {
    return "PoAwtDialog";
  }

  public String getDefaultTitle () {
    return "PoAwtDialog";
  }

  // Fields
  // ------

  private Panel _dialogPanel;  // set to 'this' if no scrollbar, or as independant
                               // panel beside the scrollbar.
  //private Scrollbar _vbar;     // set to 'null' if no scrollbar. Not used currently.
  private PoAwtElementData[] _listElements;
  private ElementDataListener[] _elementListeners;
  private List<Listener> _listeners;

  private int[] _maxWidthLabel;  // will be used as a shared 'columnWidths' in
  // GridBagLayout (see this class) of each element. By this way, their first
  // components (the labels) will all have the same width.

  // Internal classes
  //=================

  class ElementDataListener extends PoAwtElementData.Listener
  {
    private int _elementIndex;

    /**
     * @param elementIndex index of the element in this dialog that has invoke the listener.
     */
    public ElementDataListener(int elementIndex)
    {
      _elementIndex = elementIndex;
    }

    public void setElementIndex(int elementIndex)
    {
      _elementIndex = elementIndex;
    }

    @Override
    public void valueChanged()
    {
      invokeValueChanged(_elementIndex);
    }
  }

  public static class Listener
  {
    public void valueChanged(int elementIndex)
    {};
  }

}

