package meshviz.mesh.sample.meshCellFilter;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.net.URL;

import javax.swing.JCheckBox;
import javax.swing.JPanel;
import javax.swing.SwingConstants;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.nodes.SoGroup;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;
import com.openinventor.meshviz.data.PbMesh;
import com.openinventor.meshviz.data.PoMesh;
import com.openinventor.meshviz.data.PoMeshSkeleton;
import com.openinventor.meshviz.data.PoMeshSkin;
import com.openinventor.meshviz.nodes.PoIntervalCellFilter;
import com.openinventor.meshviz.nodes.PoNonLinearDataMapping2;
import com.openinventor.meshviz.nodes.PoTetrahedronMesh3D;
import com.openinventor.util.AsciiStreamReader;

import meshviz.mesh.sample.MeshViewer;
import util.Example;
import util.ViewerComponentsFactory;

/**
 * Draws the skin of a mesh made up of tetrahedrons the skin is colored, using a
 * color mapping. Use a cell filter depending on cell value.
 */
public class Main extends Example
{

  private IViewerExaminer m_viewer;
  private float m_vmin;
  private float m_vmax;
  private float m_minThreshold;
  private float m_maxThreshold;
  PoIntervalCellFilter m_cellFilter;
  PoMeshSkin m_MeshSkin;
  SoSwitch m_filterSwitch;

  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("Mesh Cell Filter");
  }

  @Override
  public void start()
  {
    m_viewer = ViewerComponentsFactory.createViewerExaminer();

    PoTetrahedronMesh3D mesh = readFile(SoPreferences.getValue("OIVJHOME") + "/data/mesh/MESH_TETR.DAT");
    if (mesh == null)
      return;

    PbMesh pb_mesh = mesh.getMesh();

    // get the min-max value of this set
    m_vmin = pb_mesh.getMinValuesSet(0);
    m_vmax = pb_mesh.getMaxValuesSet(0);
    m_minThreshold = m_vmin + 0.2f * (m_vmax - m_vmin);
    m_maxThreshold = m_vmin + 0.8f * (m_vmax - m_vmin);

    // define a data-mapping associated to this data-set
    SbColor[] colors =
    { new SbColor(0, 0, 1), new SbColor(0, 1, 1), new SbColor(0, 1, 0),
        new SbColor(1, 1, 0), new SbColor(1, 0, 0) };
    float[] val = new float[5];
    val[0] = m_vmin;
    for (int j = 1; j < 5; j++)
      val[j] = val[j - 1] + (m_vmax - m_vmin) / 4;

    PoNonLinearDataMapping2 v_DataMapping = new PoNonLinearDataMapping2();
    v_DataMapping.type.setValue(PoNonLinearDataMapping2.Types.LINEAR_PER_LEVEL);
    v_DataMapping.color.setValues(0, colors);
    v_DataMapping.value.setValues(0, val);

    // define the mesh skin
    m_MeshSkin = new PoMeshSkin();
    m_MeshSkin.valuesIndex.setValue(0);
    m_MeshSkin.coloringType.setValue(PoMesh.ColoringTypes.COLOR_MAPPING);
    m_MeshSkin.valuesIndexForCellFilter.setValue(0);

    PoMeshSkeleton meshSkeleton = new PoMeshSkeleton();

    m_cellFilter = new PoIntervalCellFilter();
    m_cellFilter.min.setValue(m_minThreshold);
    m_cellFilter.max.setValue(m_maxThreshold);
    m_cellFilter.in.setValue(false);

    m_filterSwitch = new SoSwitch();
    m_filterSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);

    // build the scene-graph root
    SoGroup root = new SoGroup();
    {
      root.addChild(v_DataMapping);
      root.addChild(mesh);
      root.addChild(meshSkeleton);
      root.addChild(m_filterSwitch);
      {
        m_filterSwitch.addChild(m_cellFilter);
      }
      root.addChild(m_MeshSkin);
    }

    m_viewer.setSceneGraph(root);
    m_viewer.viewAll();

    guiInit();
  }

  @Override
  public void stop()
  {
    m_viewer.dispose();
  }

  private void guiInit()
  {
    // checkBox panel
    final JCheckBox cellFilterBox = new JCheckBox("active");
    cellFilterBox
        .setSelected(m_filterSwitch.whichChild.getValue() == SoSwitch.SO_SWITCH_ALL);
    cellFilterBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        boolean selected = (e.getStateChange() == ItemEvent.SELECTED);
        m_filterSwitch.whichChild.setValue((selected) ? SoSwitch.SO_SWITCH_ALL
            : SoSwitch.SO_SWITCH_NONE);
      }
    });

    final JCheckBox cellFilterInBox = new JCheckBox("between min/max");
    cellFilterInBox.setSelected(m_cellFilter.in.getValue());
    cellFilterInBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        boolean selected = (e.getStateChange() == ItemEvent.SELECTED);
        m_cellFilter.in.setValue(selected);
      }
    });

    // min threshold slider
    SliderPanel minThresholdSliderPanel = new SliderPanel(m_vmin, m_vmax,
        m_minThreshold, 1);
    minThresholdSliderPanel.setSliderSize(new Dimension(300, 20));
    minThresholdSliderPanel.addInfoText("min");
    minThresholdSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_minThreshold = value;
        m_cellFilter.min.setValue(m_minThreshold);
      }
    });

    // max threshold slider
    SliderPanel maxThresholdSliderPanel = new SliderPanel(m_vmin, m_vmax,
        m_maxThreshold, 1);
    maxThresholdSliderPanel.setSliderSize(new Dimension(300, 20));
    maxThresholdSliderPanel.addInfoText("max");
    maxThresholdSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_maxThreshold = value;
        m_cellFilter.max.setValue(m_maxThreshold);
      }
    });

    // gui panel
    JPanel box_panel = new JPanel();
    box_panel.add(cellFilterBox, SwingConstants.CENTER);
    box_panel.add(cellFilterInBox, SwingConstants.CENTER);

    JPanel guiPanel = new JPanel(new GridLayout(3, 1));
    guiPanel.setBorder(new TitledBorder(new EtchedBorder(), "Cell filter: "));
    guiPanel.add(box_panel);
    guiPanel.add(minThresholdSliderPanel);
    guiPanel.add(maxThresholdSliderPanel);

    // viewer panel
    final Component component = m_viewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(guiPanel, BorderLayout.SOUTH);
  }

  private synchronized PoTetrahedronMesh3D readFile(String dataFile)
  {
    URL data_url = MeshViewer.getURL(dataFile);
    if (data_url == null)
      return null;

    PoTetrahedronMesh3D mesh = null;
    AsciiStreamReader stream = null;

    try
    {
      int i;

      stream = new AsciiStreamReader(data_url.openStream());

      // Read number of nodes and tetahedrons
      int numNodes = stream.readNextInt();
      int numTetrahedrons = stream.readNextInt();
      System.out.println("Reading " + numTetrahedrons + " tetahedrons and "
          + numNodes + " nodes");

      // Read nodes
      float[] x = new float[numNodes];
      float[] y = new float[numNodes];
      float[] z = new float[numNodes];

      for (i = 0; i < numNodes; i++)
      {
        x[i] = stream.readNextFloat();
        y[i] = stream.readNextFloat();
        z[i] = stream.readNextFloat();
      }

      System.out.println("reading nodes vectors");
      SbVec3f[] vec = new SbVec3f[numNodes];
      float vx, vy, vz;
      for (i = 0; i < numNodes; i++)
      {
        vx = stream.readNextFloat();
        vy = stream.readNextFloat();
        vz = stream.readNextFloat();
        vec[i] = new SbVec3f(vx, vy, vz);
      }

      System.out.println("reading value");
      float[][] v = new float[4][numNodes];

      for (i = 0; i < numNodes; i++)
        v[0][i] = stream.readNextFloat();
      for (i = 0; i < numNodes; i++)
        v[1][i] = stream.readNextFloat();
      for (i = 0; i < numNodes; i++)
        v[2][i] = stream.readNextFloat();
      for (i = 0; i < numNodes; i++)
        v[3][i] = stream.readNextFloat();

      for (i = 0; i < numNodes; i++)
        stream.readNextInt();

      System.out.println("reading tetahedrons nodes indices");
      int[] tetrahedronNode = new int[numTetrahedrons * 4];
      for (i = 0; i < (numTetrahedrons * 4); i += 4)
      {
        tetrahedronNode[i] = stream.readNextInt();
        tetrahedronNode[i + 1] = stream.readNextInt();
        tetrahedronNode[i + 2] = stream.readNextInt();
        tetrahedronNode[i + 3] = stream.readNextInt();
      }

      System.out.println("End reading mesh.");

      mesh = new PoTetrahedronMesh3D();
      mesh.setGeometry(numNodes, x, y, z, numTetrahedrons, tetrahedronNode);
      mesh.addValuesSet(0, v[0]);

    } catch (Exception e)
    {
      System.err.println(e.toString());
      mesh = null;
    }

    return mesh;
  }
}
