package meshvizxlm.eclipsemeshviz.pillargrid;

import meshvizxlm.eclipsemeshviz.DemoSettings;

public abstract class PillarGrid
{
  public static final float UNDEF = -999.f;

  // number of vertices at K position for a given I and J
  // (equals 1 if not storing Horizontal Faults, 2 otherwise).
  protected int m_numHFacesAtKLevel;

  public static PillarGrid getNewInstance()
  {
    switch ( DemoSettings.GRID_TYPE )
    {
    case E_GRID :
      return new PillarEGrid();
    case JIK_GRID :
      return new PillarGridJIK();
    default:
      throw new UnsupportedOperationException("Grid type not supported " + DemoSettings.GRID_TYPE);
    }
  }

  // size of mesh
  protected int m_dimI;
  protected int m_dimJ;
  protected int m_dimK;
  protected int m_dimIxJ;

  // mesh data
  protected float[] m_coord; // 6*(nj+1)*(ni+1)
  protected float[] m_zcorn; // (2*ni)*(2*nj)*(2*nk)
  protected boolean[] m_actnum; // ni*nj*nk
  protected float[] m_poro; // ni*nj*nk

  protected int m_2DimI;

  protected int m_4DimIDimJ;

  protected long m_numZcorns;

  protected long m_numNonActiveCells;

  protected boolean m_availablePorosity;

  protected boolean m_minMaxUpdated;
  protected float[] m_min = { UNDEF, UNDEF, UNDEF };
  protected float[] m_max = { UNDEF, UNDEF, UNDEF };

  protected PillarGrid()
  {
    m_coord = null;
    m_zcorn = null;
    m_actnum = null;
    m_poro = null;

    m_dimI = 0;
    m_dimJ = 0;
    m_dimK = 0;
    m_dimIxJ = 0;
    m_numNonActiveCells = 0;
    m_minMaxUpdated = false;
    m_numHFacesAtKLevel = DemoSettings.HORIZONTAL_FAULTS ? 2 : 1;
  }

  public void clear()
  {
    m_coord = null;
    m_zcorn = null;
    m_actnum = null;
    m_poro = null;

    m_dimI = 0;
    m_dimJ = 0;
    m_dimK = 0;
    m_dimIxJ = 0;
    m_numZcorns = 0;
    m_numNonActiveCells = 0;
  }

  public void setCoord(int index, float x, float y, float z)
  {
    int i = index;
    m_coord[i++] = x;
    m_coord[i++] = y;
    m_coord[i] = z;
  }

  public boolean getActnum(int cellId)
  {
    return m_actnum[cellId];
  }

  public void setActnum(int cellId, boolean val)
  {
    m_actnum[cellId] = val;
  }

  public int getDimI()
  {
    return m_dimI;
  }

  public int getDimJ()
  {
    return m_dimJ;
  }

  public int getDimK()
  {
    return m_dimK;
  }

  public abstract float getProperty(int i, int j, int k);

  public abstract float getPorosity(int i, int j, int k);

  public float getNonIndexedPorosity(int i)
  {
    return m_poro[i];
  }

  public abstract float getDepth(int i, int j, int k);

  public void setDepth(int id, float val)
  {
    m_zcorn[id] = val;
  }

  public void setPorosity(int id, float val)
  {
    m_poro[id] = val;
  }

  public boolean isEmpty()
  {
    return m_dimI == 0 || m_dimJ == 0 || m_dimK == 0;
  }

  public boolean isValid(int i, int j, int k)
  {
    return m_actnum[k * m_dimIxJ + j * m_dimI + i];
  }

  public boolean isValid(int cellIndex)
  {
    return m_actnum[cellIndex];
  }

  public abstract void setDim(int numCellI, int numCellJ, int numCellK);

  public boolean isPorosityAvailable()
  {
    return m_availablePorosity;
  }

  public void setAvailablePorosity(boolean bool)
  {
    m_availablePorosity = bool;
  }

  protected boolean generatePillarVertex(float[] P12, float zcorn, double[] xyz)
  {
    float x1 = P12[0];
    float y1 = P12[1];
    float z1 = P12[2];
    float x2 = P12[3];
    float y2 = P12[4];
    float z2 = P12[5];
    float x;
    float y;
    float z = zcorn;

    // check geometry
    if ( x1 == UNDEF || x2 == UNDEF || y1 == UNDEF || y2 == UNDEF )
      return false;

    if ( x1 == x2 && y1 == y2 )
    {
      x = x1;
      y = y1;
    }
    else
    {
      if ( z1 == z2 )
        return false;
      float a = (z - z1) / (z2 - z1);
      float b = (1 - a);
      x = b * x1 + a * x2;
      y = b * y1 + a * y2;
    }

    xyz[0] = x;
    xyz[1] = y;
    xyz[2] = z;

    return true;
  }

  public long getNumZCorns()
  {
    return m_numZcorns;
  }

  public final boolean hasNonActiveCells()
  {
    return m_numNonActiveCells > 0;
  }

  public long getNumNonActiveCells()
  {
    return m_numNonActiveCells;
  }

  public float[] getMin()
  {
    if ( !m_minMaxUpdated )
      updateMinMax();
    return m_min;
  }

  public float getXMin()
  {
    if ( !m_minMaxUpdated )
      updateMinMax();
    return m_min[0];
  }

  public float getYMin()
  {
    if ( !m_minMaxUpdated )
      updateMinMax();
    return m_min[1];
  }

  public float getZMin()
  {
    if ( !m_minMaxUpdated )
      updateMinMax();
    return m_min[2];
  }

  public float[] getMax()
  {
    if ( !m_minMaxUpdated )
      updateMinMax();
    return m_max;
  }

  public float getXMax()
  {
    if ( !m_minMaxUpdated )
      updateMinMax();
    return m_max[0];
  }

  public float getYMax()
  {
    if ( !m_minMaxUpdated )
      updateMinMax();
    return m_max[1];
  }

  public float getZMax()
  {
    if ( !m_minMaxUpdated )
      updateMinMax();
    return m_max[2];
  }

  public long getMemoryFootPrint()
  {
    return (4 * (m_coord.length + m_zcorn.length) + m_actnum.length);
  }

  protected void updateMinMax()
  {
    if ( !m_minMaxUpdated )
    {
      m_min[0] = m_min[1] = m_min[2] = Float.POSITIVE_INFINITY;
      m_max[0] = m_max[1] = m_max[2] = Float.NEGATIVE_INFINITY;

      for ( int np = 0; np < m_coord.length; ++np )
      {
        m_min[0] = Math.min(m_min[0], m_coord[np++]);
        m_min[1] = Math.min(m_min[1], m_coord[np++]);
        m_min[2] = Math.min(m_min[2], m_coord[np++]);
        m_max[0] = Math.max(m_max[0], m_coord[np++]);
        m_max[1] = Math.max(m_max[1], m_coord[np++]);
        m_max[2] = Math.max(m_max[2], m_coord[np]);
      }
      m_minMaxUpdated = true;
    }
  }

  public abstract double[] computePillarVertex(int zcornId);

  public abstract int getNextUpperCellFaceZCornId(int zcornId);

  public abstract int getNextUpperFaceZCornId(int zcornId);

}
