package meshvizxlm.eclipsemeshviz.pillargrid;

import java.util.Arrays;

import meshvizxlm.eclipsemeshviz.DemoSettings;

/**
 * This type of PillarGrid must store 8 zcorn values per level at each pillar.
 * Consecutive zcorn values are along the k dimension. The total number of zcorn
 * is (8*(ni+1)*(nj+1)*(nk+1)).
 */
public class PillarGridJIK extends PillarGrid
{
  private int m_upperFaceZCornIdShift;

  public PillarGridJIK()
  {}

  // return 8 zcorns if HORIZONTAL_FAULTS, 4 zcorns otherwise
  // return a pointer to the first zcorn and its index in the table
  public int getPillarZCorns(int i, int j, int k, float[] indexedZcorns)
  {
    int zcorn_index = computeZCornId(i, j, k);
    if ( !DemoSettings.HORIZONTAL_FAULTS )
    {
      if ( k == m_dimK && m_numCorners == 8 )
        zcorn_index += 4;
      System.arraycopy(m_zcorn, zcorn_index, indexedZcorns, 0, 4);
    }
    else
      System.arraycopy(m_zcorn, zcorn_index, indexedZcorns, 0, 8);
    return zcorn_index;
  }

  @Override
  public double[] computePillarVertex(int zcornId)
  {
    final float[] P12 = new float[6];

    int pillar_index = zcornId / m_numPillarVertices;
    System.arraycopy(m_coord, pillar_index * 6, P12, 0, 6);

    double[] xyz = new double[3];
    generatePillarVertex(P12, m_zcorn[zcornId], xyz);
    return xyz;
  }

  @Override
  public float getProperty(int i, int j, int k)
  {
    return m_zcorn[computeZCornId(i, j, k)];
  }

  // methods for EGrid readers to fill up the grid zcorn
  public void pushBackEGridZCorn(float zcorn)
  {
    setEGridZCorn(m_numZcornAdded, zcorn);
    ++m_numZcornAdded;
  }

  private void setEGridZCorn(int eGridZCornId, float zcorn)
  {
    int dk = eGridZCornId / m_4DimIDimJ;
    int k = dk / 2;
    int kPos = dk % 2;
    int dij = eGridZCornId % m_4DimIDimJ;
    int dj = dij / m_2DimI;
    int j = dj / 2;
    int jPos = dj % 2;
    int di = dij % m_2DimI;
    int i = di / 2;
    int iPos = di % 2;
    int ZcornPos = kPos << 2 | jPos << 1 | (iPos ^ jPos);
    int reindexedZcorn =
        (j + jPos) * m_numPillarSliceIVertices + (i + iPos) * m_numPillarVertices + m_numCorners * (k + kPos)
            + ZcornPos;

    m_zcorn[reindexedZcorn] = zcorn;
  }

  @Override
  public void setDim(int numCellI, int numCellJ, int numCellK)
  {
    clear();
    m_numZcornAdded = 0;
    m_dimI = numCellI;
    m_dimJ = numCellJ;
    m_dimK = numCellK;
    m_2DimI = 2 * m_dimI;
    m_4DimIDimJ = 2 * m_2DimI * m_dimJ;
    m_numCorners = m_numHFacesAtKLevel * 4;
    // gap between a 2 zcorn indices in the same cell is 4 when only 4 corners
    // at a k level, 12 if 8 corners
    m_upperFaceZCornIdShift = m_numHFacesAtKLevel == 1 ? 4 : 12;
    m_numPillarVertices = m_numCorners * (m_dimK + 1);
    m_numPillarSliceIVertices = m_numCorners * (m_dimK + 1) * (m_dimI + 1);
    m_numZcorns = m_numPillarSliceIVertices * (m_dimJ + 1);

    int oldLength;

    // allocate pillar coordinates
    int npillars = (m_dimI + 1) * (m_dimJ + 1);
    int ndata = 6 * npillars;
    if ( m_coord == null )
      m_coord = new float[ndata];
    else
      m_coord = Arrays.copyOf(m_coord, ndata);

    // allocate z corner (8 or 4 per each level in each pillar)

    if ( m_zcorn == null )
    {
      m_zcorn = new float[(int) m_numZcorns];
      oldLength = 0;
    }
    else
    {
      oldLength = m_zcorn.length;
      m_zcorn = Arrays.copyOf(m_zcorn, (int) m_numZcorns);
    }
    if ( m_zcorn.length > oldLength )
      Arrays.fill(m_zcorn, oldLength, m_zcorn.length, UNDEF);

    if ( m_actnum == null )
    {
      ndata = m_dimI * m_dimJ * m_dimK;
      m_actnum = new boolean[ndata];
      oldLength = 0;
    }
    else
    {
      // allocate actnum
      ndata = m_dimI * m_dimJ * m_dimK;
      oldLength = m_actnum.length;
      m_actnum = Arrays.copyOf(m_actnum, ndata);
    }

    if ( m_actnum.length > oldLength )
      Arrays.fill(m_actnum, oldLength, m_actnum.length, true);

    m_poro = new float[ndata];
  }

  @Override
  public boolean isValid(int i, int j, int k)
  {
    return isValid((j * m_dimI * m_dimK) + i * m_dimK + k);
  }

  @Override
  public int getNextUpperCellFaceZCornId(int zcornId)
  {
    return zcornId + m_numCorners;
  }

  @Override
  public int getNextUpperFaceZCornId(int lowerFaceZCornId)
  {
    return lowerFaceZCornId + m_upperFaceZCornIdShift;
  }

  protected int computeZCornId(int i, int j, int k)
  {
    return j * m_numPillarSliceIVertices + i * m_numPillarVertices + m_numCorners * k;
  }

  private int m_numZcornAdded;
  // number of vertices at each pillar level
  int m_numCorners;
  // number of vertices on a each pillar
  private int m_numPillarVertices;
  // number of vertices on a slice of pillar along the I-Dimension
  private int m_numPillarSliceIVertices;

  @Override
  public float getPorosity(int i, int j, int k)
  {

    return m_poro[(j * m_dimI * m_dimK) + i * m_dimK + k];
  }

  @Override
  public float getDepth(int i, int j, int k)
  {
    return m_zcorn[j * m_numPillarSliceIVertices + i * m_numPillarVertices + m_numCorners * k];
  }
}
