package meshvizxlm.mapping.isosurface;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.swing.BorderFactory;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.border.EtchedBorder;

import com.openinventor.inventor.nodes.SoDrawStyle;
import com.openinventor.inventor.nodes.SoPickStyle;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoShapeHints;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.sensors.SoTimerSensor;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;
import com.openinventor.meshvizxlm.extractors.MiExtractorCallback;
import com.openinventor.meshvizxlm.mapping.nodes.*;

import meshvizxlm.mapping.MyVec3Set;
import meshvizxlm.mesh.MbSampleMeshBuilder;
import meshvizxlm.mesh.data.MbScalarSetI;
import meshvizxlm.mesh.data.MbScalarSetIjk;
import meshvizxlm.mesh.geometry.MbMeshGeometry;
import meshvizxlm.mesh.volumes.MbHexahedronMeshIjk;
import meshvizxlm.mesh.volumes.MbVertexHexahedronMeshIjk;
import meshvizxlm.mesh.volumes.MbVolumeMeshTetrahedron;
import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{

  class MyIsosurfExtractorCallback implements MiExtractorCallback
  {
    @Override
    public void beginCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      if ( topoChanged )
        System.out.println("isosurface computation started...");
      if ( dataSetChanged )
        System.out.println("dataset computation started...");
    }

    @Override
    public void endCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      if ( topoChanged )
        System.out.println("isosurface computation finished");
      if ( dataSetChanged )
        System.out.println("dataset computation finished...");
    }
  }

  class MySkinExtractorCallback implements MiExtractorCallback
  {
    @Override
    public void beginCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      if ( topoChanged )
        System.out.println("skin computation started");
      else if ( geomChanged )
        System.out.println("start updating geometry of the skin...");

    }

    @Override
    public void endCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      if ( topoChanged )
        System.out.println("skin computation finished");
      else if ( geomChanged )
        System.out.println("skin geometry update finished...");
    }
  }

  private static final int NUM_CELL_I = 10;
  private static final int NUM_CELL_J = 10;
  private static final int NUM_CELL_K = 10;

  private IViewerExaminer m_renderArea;
  private MbVolumeMeshTetrahedron m_meshTetra;
  private MbHexahedronMeshIjk m_meshHexaIjk;
  private MbVertexHexahedronMeshIjk m_meshVertexHexaIjk;
  private MoMeshIsosurface m_isosurface;
  private MoMesh m_moMesh;
  private SoSwitch m_vectorSwitch;
  private MoPredefinedColorMapping m_colMap;
  private MoMeshSkin m_skin;
  private MoMeshVector m_vector;
  private SoTimerSensor m_animationTimer;

  public Main()
  {
    double[] min = { 0, 0, 0 };
    double[] max = { 10, 10, 10 };
    MbSampleMeshBuilder meshBuilder = new MbSampleMeshBuilder();

    m_meshTetra = meshBuilder.getMeshTetrahedron(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, min, max);
    m_meshHexaIjk = meshBuilder.getHexahedronMeshIjk(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, min, max);
    m_meshVertexHexaIjk = meshBuilder.getVertexHexahedronMeshIjk(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, min, max);

    m_animationTimer = new SoTimerSensor(new Runnable()
    {
      int m_numRender = 0;

      @Override
      public void run()
      {
        MbMeshGeometry geom = m_meshTetra.getGeometry();
        double delta;
        if ( this.m_numRender / 20 % 2 == 0 )
          delta = -0.1;
        else
          delta = 0.1;
        this.m_numRender++;

        final long endNodeId = m_meshTetra.getTopology().getEndNodeId();
        double[] xyz;
        for ( long i = 0; i < endNodeId; ++i )
        {
          xyz = geom.getCoord(i);
          geom.setX(i, xyz[0] + delta);
          geom.setY(i, xyz[1] + delta);
          geom.setZ(i, xyz[2] + delta);
        }
        m_moMesh.touch();
      }
    });
  }

  @Override
  public void start()
  {
    SoSeparator root = buildSceneGraph();

    m_renderArea = ViewerComponentsFactory.createViewerExaminer();
    m_renderArea.setSceneGraph(root);
    m_renderArea.viewAll();

    // SWING part
    SliderPanel isoValueSlider = new SliderPanel(0, 200, 100);
    isoValueSlider.addInfoText("Iso value : ");
    isoValueSlider.setSliderSize(new Dimension(300, 20));
    isoValueSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_isosurface.isoValue.setValue(value);
      }
    });

    final JCheckBox animCheckBox = new JCheckBox();
    animCheckBox.setText("animate the mesh geometry");
    animCheckBox.setSelected(m_animationTimer.isScheduled());
    animCheckBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( !m_animationTimer.isScheduled() )
          m_animationTimer.schedule();
        else
          m_animationTimer.unschedule();
      }
    });

    final JCheckBox vectorCheckbox = new JCheckBox();
    vectorCheckbox.setText("display vectors");
    vectorCheckbox.setSelected(m_vectorSwitch.whichChild.getValue() == SoSwitch.SO_SWITCH_ALL);
    vectorCheckbox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( vectorCheckbox.isSelected() )
          m_vectorSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
        else
          m_vectorSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
      }
    });

    JLabel label = new JLabel("  Mesh: ");
    String[] items = new String[] { "Tetrahedron", "Hexahedron IJK", "Vertex Hexahedron IJK" };
    final JComboBox<String> meshBox = new JComboBox<>(items);
    meshBox.setSelectedIndex(0);
    meshBox.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        switch ( meshBox.getSelectedIndex() )
        {
        case 0 : // Tetrahedron
          m_moMesh.setMesh(m_meshTetra);
          MbScalarSetI scalarSet = m_meshTetra.getScalarSet("$CellId");
          m_colMap.minValue.setValue((float) scalarSet.getMin());
          m_colMap.maxValue.setValue((float) scalarSet.getMax());
          animCheckBox.setEnabled(true);
          m_vector.scaleFactor.setValue(1f);
          break;
        case 1 : // hexahedron IJK
          m_moMesh.setMesh(m_meshHexaIjk);
          MbScalarSetIjk scalarSetIjk = m_meshHexaIjk.getScalarSetIjk("$CellId");
          m_colMap.minValue.setValue((float) scalarSetIjk.getMin());
          m_colMap.maxValue.setValue((float) scalarSetIjk.getMax());
          animCheckBox.setEnabled(false);
          animCheckBox.setSelected(false);
          m_vector.scaleFactor.setValue(1f);
          break;
        case 2 : // Vertex hexahedron IJK
          m_moMesh.setMesh(m_meshVertexHexaIjk);
          scalarSetIjk = m_meshVertexHexaIjk.getScalarSetIjk("$CellId");
          m_colMap.minValue.setValue((float) scalarSetIjk.getMin());
          m_colMap.maxValue.setValue((float) scalarSetIjk.getMax());
          animCheckBox.setEnabled(false);
          animCheckBox.setSelected(false);
          m_vector.scaleFactor.setValue(0.1f);
          break;
        default:
          break;
        }
      }
    });

    JPanel panel2 = new JPanel();
    panel2.setBorder(BorderFactory.createEtchedBorder(EtchedBorder.LOWERED, null, null));
    GridBagLayout gridBag = new GridBagLayout();
    GridBagConstraints constraints = new GridBagConstraints();
    panel2.setLayout(gridBag);

    constraints.anchor = GridBagConstraints.LINE_START;
    constraints.weightx = 0.;
    constraints.ipady = 30;
    gridBag.setConstraints(label, constraints);

    constraints.ipady = 0;
    constraints.gridwidth = GridBagConstraints.REMAINDER;
    gridBag.setConstraints(animCheckBox, constraints);
    gridBag.setConstraints(vectorCheckbox, constraints);

    constraints.weightx = 1.;
    gridBag.setConstraints(isoValueSlider, constraints);
    gridBag.setConstraints(meshBox, constraints);
    panel2.add(animCheckBox);
    panel2.add(vectorCheckbox);
    panel2.add(label);
    panel2.add(meshBox);
    panel2.add(isoValueSlider);

    final Component component = m_renderArea.getComponent();
    component.setPreferredSize(new Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(panel2, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    m_renderArea.dispose();
  }

  private SoSeparator buildSceneGraph()
  {
    SoShapeHints sh = new SoShapeHints();
    sh.vertexOrdering.setValue(SoShapeHints.VertexOrderings.CLOCKWISE);

    m_moMesh = new MoMesh();
    m_moMesh.setMesh(m_meshTetra);

    m_colMap = new MoPredefinedColorMapping();
    // Adjust range to scalar set 1
    m_colMap.minValue.setValue((float) m_meshTetra.getScalarSet("$CellId").getMin());
    m_colMap.maxValue.setValue((float) m_meshTetra.getScalarSet("$CellId").getMax());
    m_colMap.predefColorMap.setValue(MoPredefinedColorMapping.PredefColorMapping.STANDARD);

    MoDataBinding dataBinding = new MoDataBinding();
    dataBinding.dataBinding.setValue(MoDataBinding.DataBinding.PER_CELL);

    // Isosurface dataset
    MoScalarSetI moScalarSetI1 = new MoScalarSetI();
    moScalarSetI1.setScalarSet(m_meshTetra.getScalarSet("$Sphere"));

    MoScalarSetIjk moScalarSetIjk1 = new MoScalarSetIjk();
    moScalarSetIjk1.setScalarSet(m_meshVertexHexaIjk.getScalarSetIjk("$Sphere"));

    // Colors dataset
    MoScalarSetI moScalarSetI2 = new MoScalarSetI();
    moScalarSetI2.setScalarSet(m_meshTetra.getScalarSet("$CellId"));

    MoScalarSetIjk moScalarSetIjk2 = new MoScalarSetIjk();
    moScalarSetIjk2.setScalarSet(m_meshHexaIjk.getScalarSetIjk("$CellId"));

    m_isosurface = new MoMeshIsosurface();
    m_isosurface.isoValue.setValue(100);
    m_isosurface.isoScalarSetId.setValue(0);
    m_isosurface.colorScalarSetId.setValue(1);
    m_isosurface.setExtractorCallback(new MyIsosurfExtractorCallback());

    // Transparent Mesh Skin
    SoDrawStyle style = new SoDrawStyle();
    style.style.setValue(SoDrawStyle.Styles.LINES);
    style.linePattern.setValue((short) 0xf0f0);

    SoPickStyle pickStyle = new SoPickStyle();
    pickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);

    m_skin = new MoMeshSkin();
    m_skin.colorScalarSetId.setValue(1);
    m_skin.setExtractorCallback(new MySkinExtractorCallback());

    SoSeparator skinSep = new SoSeparator();
    {
      skinSep.addChild(style);
      skinSep.addChild(pickStyle);
      skinSep.addChild(m_skin);
    }

    MoMesh isoMesh = new MoMesh();
    isoMesh.connectFrom(m_isosurface);

    MyVec3Set myVec3Set = new MyVec3Set(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K);
    MoVec3SetI moVec3Set = new MoVec3SetI();
    moVec3Set.setVec3Set(myVec3Set);

    MoVec3SetIjk moVec3SetIjk = new MoVec3SetIjk();
    moVec3SetIjk.setVec3Set(m_meshVertexHexaIjk.getVec3SetIjk("$MyGeometry"));

    m_vectorSwitch = new SoSwitch();
    MoDataBinding vdataBinding = new MoDataBinding();
    vdataBinding.dataBinding.setValue(MoDataBinding.DataBinding.PER_NODE);
    m_vector = new MoMeshVector();
    m_vector.thicknessFactor.setValue(0.5f);
    m_vector.colorScalarSetId.setValue(-1);
    {
      m_vectorSwitch.addChild(vdataBinding);
      m_vectorSwitch.addChild(m_vector);
    }

    SoSeparator root = new SoSeparator();
    { // assemble scene graph
      root.addChild(sh);
      root.addChild(m_moMesh);
      root.addChild(m_colMap);
      root.addChild(dataBinding);
      root.addChild(moScalarSetI1);
      root.addChild(moScalarSetIjk1);
      root.addChild(moScalarSetI2);
      root.addChild(moScalarSetIjk2);
      root.addChild(m_isosurface);

      root.addChild(skinSep);

      root.addChild(isoMesh);
      root.addChild(moVec3Set);
      root.addChild(moVec3SetIjk);
      root.addChild(m_vectorSwitch);
    }

    return root;
  }

  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("Isosurface Example");
  }
}
