package meshvizxlm.mapping.polyhedralmesh;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.File;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;

import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.border.TitledBorder;

import com.openinventor.inventor.SbPlane;
import com.openinventor.inventor.SbVec3d;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.draggers.SoJackDragger;
import com.openinventor.inventor.manips.SoClipPlaneManip;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;
import com.openinventor.meshvizxlm.MxMeshViz;
import com.openinventor.meshvizxlm.extractors.MiExtractorCallback;
import com.openinventor.meshvizxlm.mapping.nodes.*;
import com.openinventor.meshvizxlm.mesh.data.DataBinding;
import com.openinventor.meshvizxlm.meshextracted.data.MeXScalardSetI;
import com.openinventor.meshvizxlm.meshextracted.data.MeXVec3dSetI;

import meshvizxlm.mesh.data.MbScalarSetI;
import meshvizxlm.mesh.data.MbVec3SetI;
import meshvizxlm.mesh.geometry.MbMeshGeometry;
import meshvizxlm.mesh.volumes.MbVolumeMeshPolyhedron;
import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  private static DecimalFormat scFormat = new DecimalFormat("#.###E0", new DecimalFormatSymbols(Locale.ENGLISH));
  private static DecimalFormat decFormat = new DecimalFormat("#.######", new DecimalFormatSymbols(Locale.ENGLISH));

  class ProbeCallback implements MoMeshPointProbe.MoProbeCallback
  {
    private SoSwitch m_displaySwitch;
    private SoText2 m_indexText;
    private SoText2 m_valueText;

    public ProbeCallback(SoSwitch displaySwitch, SoText2 indexText, SoText2 valueText)
    {
      m_displaySwitch = displaySwitch;
      m_indexText = indexText;
      m_valueText = valueText;
    }

    @Override
    public void motionCallback(long cellId, MeXScalardSetI scalars, MeXVec3dSetI vectors)
    {
      if ( cellId == MxMeshViz.UNDEFINED_ID )
      {
        m_displaySwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
      }
      else
      {
        m_displaySwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);

        StringBuffer buf = new StringBuffer("cell: ");
        buf.append(cellId);

        m_indexText.string.setValue(buf.toString());

        buf = new StringBuffer();
        if ( scalars.getSize() > 0 )
        {
          buf.append("value: ");
          buf.append(format(scalars.get(0)));
          buf.append(" ");
        }
        if ( vectors.getSize() > 0 )
        {
          double[] vec = vectors.get(0);
          buf.append("vector: ");
          buf.append("(");
          buf.append(format(vec[0]));
          buf.append(",");
          buf.append(format(vec[1]));
          buf.append(",");
          buf.append(format(vec[2]));
          buf.append(")");
          buf.append(" ");
        }
        m_valueText.string.setValue(buf.toString());
      }
    }

    @Override
    public void motionCallback(long cellIdI, long cellIdJ, long cellIdK, MeXScalardSetI scalars, MeXVec3dSetI vectors)
    {}

    @Override
    public void motionCallback(long cellIdI, long cellIdJ, MeXScalardSetI scalars, MeXVec3dSetI vectors)
    {}

    private String format(double d)
    {
      if ( d == 0 || (d >= 0.00001 && d < 10000) || (d <= -0.00001 && d > -10000) )
        return decFormat.format(d);
      return scFormat.format(d);
    }
  }

  class MyIsosurfExtractorCallback implements MiExtractorCallback
  {
    @Override
    public void beginCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      if ( topoChanged )
        System.out.println("isosurface computation started...");
      if ( dataSetChanged )
        System.out.println("dataset computation started...");
    }

    @Override
    public void endCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      if ( topoChanged )
        System.out.println("isosurface computation finished");
      if ( dataSetChanged )
        System.out.println("dataset computation finished...");
    }
  }

  class MySkinExtractorCallback implements MiExtractorCallback
  {
    @Override
    public void beginCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      if ( topoChanged )
        System.out.println("skin computation started");
      else if ( geomChanged )
        System.out.println("start updating geometry of the skin...");

    }

    @Override
    public void endCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      if ( topoChanged )
        System.out.println("skin computation finished");
      else if ( geomChanged )
        System.out.println("skin geometry update finished...");
    }
  }

  private IViewerExaminer m_renderArea;
  private MbVolumeMeshPolyhedron m_meshpoly;
  private MoMeshIsosurface m_isosurface;
  private MoMesh m_moMesh;
  private SoSwitch m_vectorSwitch;
  private MoPredefinedColorMapping m_colMap;
  private PolyhedralMeshReader m_reader;
  private String m_dataPath;
  private MbScalarSetI m_scalars;
  private int currentMesh;
  private MoScalarSetI m_moScalarSetI;
  private boolean m_fromMemory;
  private MoVec3SetI m_moVec3Set;
  private SliderPanel m_isoValueSlider;
  private MoDrawStyle m_drawStyle;
  private SoClipPlaneManip m_planeManip;
  private SoSwitch m_planeSliceSwitch;
  private SoSwitch m_isoSurfSwitch;
  private float m_sliderMin;
  private float m_sliderMax;
  private SoSwitch m_skinSwitch;

  // Probing
  private SoJackDragger m_probeDragger;
  private SoSwitch m_probeSwitch;
  private SoSwitch m_probeTextSwitch;

  public Main()
  {
    String meshvizxlmDataPrefix = SoPreferences.getValue("OIVJHOME") + File.separator + "examples" + File.separator
        + "meshvizxlm" + File.separator + "data";
    String[] pkgNameSplit = this.getClass().getPackage().getName().split("\\.");
    m_dataPath = meshvizxlmDataPrefix + File.separator + pkgNameSplit[pkgNameSplit.length - 1] + File.separator;

    m_reader = new PolyhedralMeshReader(m_dataPath + "cone.dat");
    m_reader.readMesh();
    defineMesh(m_reader.getXcoords(), m_reader.getYcoords(), m_reader.getZcoords(), m_reader.getFaceNodeList(),
        m_reader.getNumNodePerFaceList(), m_reader.getCellFacets(), m_reader.getNumCells(), m_reader.getScalars());
    currentMesh = 3;
    m_sliderMax = 5;
    m_sliderMin = (float) .4;
  }

  private void defineMesh(double[] xArray, double[] yArray, double[] zArray, int[] faceNodeList,
      int[] numNodePerFaceList, List<ArrayList<Integer>> cellFacets, int numCells, double[] scalarArray)
  {
    m_meshpoly =
        new MbVolumeMeshPolyhedron(xArray, yArray, zArray, faceNodeList, numNodePerFaceList, cellFacets, numCells);

    MbVec3SetI vec = new MbVec3SetI(xArray, yArray, zArray);
    vec.setName("$MyGeometry");
    m_meshpoly.addVec3Set(vec);

    m_scalars = new MbScalarSetI(scalarArray, "$Scalars", DataBinding.PER_NODE);
    m_meshpoly.addScalarSet(m_scalars);
  }

  /*
   * Tool do draw some very simple mesh
   */
  private void buildDodecahedron()
  {
    double[] xArray =
        { 0, 4.574530124664307e-001, -3.600445032119751e+000, -1.571496009826660e+000, -1.571496009826660e+000,
            -1.571496009826660e+000, -1.571496009826660e+000, 4.574530124664307e-001, -3.600445032119751e+000,
            -6.883353710174561e+000, -6.883353710174561e+000, 3.740361690521240e+000, 3.740361690521240e+000,
            -4.854404449462891e+000, 1.711412668228149e+000, 1.711412668228149e+000, -4.854404449462891e+000,
            -4.854404449462891e+000, -4.854404449462891e+000, 1.711412668228149e+000, 1.711412668228149e+000 };

    double[] yArray =
        { 0, 1.312415313720703e+001, 1.312415313720703e+001, 9.841244697570801e+000, 5.783347129821777e+000,
            9.841244697570801e+000, 5.783347129821777e+000, 2.500438213348389e+000, 2.500438213348389e+000,
            7.812295913696289e+000, 7.812295913696289e+000, 7.812295913696289e+000, 7.812295913696289e+000,
            1.109520435333252e+001, 1.109520435333252e+001, 1.109520435333252e+001, 1.109520435333252e+001,
            4.529387474060059e+000, 4.529387474060059e+000, 4.529387474060059e+000, 4.529387474060059e+000 };

    double[] zArray =
        { 0, 0.000000000000000e+000, 0.000000000000000e+000, 5.311857700347900e+000, 5.311857700347900e+000,
            -5.311857700347900e+000, -5.311857700347900e+000, 0.000000000000000e+000, 0.000000000000000e+000,
            2.028949022293091e+000, -2.028949022293091e+000, -2.028949022293091e+000, 2.028949022293091e+000,
            3.282908678054810e+000, 3.282908678054810e+000, -3.282908678054810e+000, -3.282908678054810e+000,
            -3.282908678054810e+000, 3.282908678054810e+000, -3.282908678054810e+000, 3.282908678054810e+000 };

    double[] color = Arrays.copyOf(yArray, 21);
    // @formatter:off
    int[] facets =
      { 9, 18, 8, 17, 10,
        11, 19, 7, 20, 12,
        16, 10, 17, 6, 5,
        5, 6, 19, 11, 15,
        7, 19, 6, 17, 8,
        12, 20, 4, 3, 14,
        13, 3, 4, 18, 9,
        4, 20, 7, 8, 18,
        1, 14, 3, 13, 2,
        2, 13, 9, 10, 16,
        15, 11, 12, 14, 1,
        1, 2, 16, 5, 15 };
    // @formatter:on
    int[] numNodes = new int[] { 0, 5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60 };

    List<ArrayList<Integer>> cellFacets = new ArrayList<ArrayList<Integer>>(12);
    cellFacets.add(new ArrayList<Integer>());
    for ( int i = 0; i < 12; i++ )
      cellFacets.get(0).add(i);

    defineMesh(xArray, yArray, zArray, facets, numNodes, cellFacets, 1, color);
  }

  private void buildPolyhedron(double a)
  {
    double[] xArray = { -10, 10, 10, -10, -10, 10, 10, -10 };
    double[] yArray = { -5, -5, 5, 5, -5, -5, 5, 5 };
    double[] zArray = { -5, -5, -5, -5, 5, 5, 5, 5 };

    double[] color = Arrays.copyOf(yArray, 8);

    int[] facets = { 0, 1, 2, 3, 0, 4, 5, 1, 3, 2, 6, 7, 4, 7, 6, 5, 0, 3, 7, 4, 1, 5, 6, 2 };
    int[] numNodes = { 0, 4, 8, 12, 16, 20, 24, 28 };
    List<ArrayList<Integer>> cellFacets = new ArrayList<ArrayList<Integer>>(6);
    cellFacets.add(new ArrayList<Integer>());
    for ( int i = 0; i < 6; i++ )
      cellFacets.get(0).add(i);

    defineMesh(xArray, yArray, zArray, facets, numNodes, cellFacets, 1, color);
  }

  private void buildOctogon()
  {
    double[] xArray = new double[] { 0, 1.325747108459473e+001, 9.201438903808594e+000, -5.906926393508911e-001,
        -1.038282203674316e+001, -1.443885612487793e+001, -1.038282012939453e+001, -5.906918644905090e-001,
        9.201440811157227e+000, 1.325747108459473e+001, 9.201438903808594e+000, -5.906926393508911e-001,
        -1.038282203674316e+001, -1.443885612487793e+001, -1.038282012939453e+001, -5.906918644905090e-001,
        9.201440811157227e+000, 1.325747108459473e+001, 9.201438903808594e+000, -5.906926393508911e-001,
        -1.038282203674316e+001, -1.443885612487793e+001, -1.038282012939453e+001, -5.906918644905090e-001,
        9.201440811157227e+000 };

    double[] yArray = new double[] { 0, 2.500442504882813e+000, 2.500442504882813e+000, 2.500442504882813e+000,
        2.500442504882813e+000, 2.500442504882813e+000, 2.500442504882813e+000, 2.500442504882813e+000,
        2.500442504882813e+000, 2.500442504882813e+000, 2.500442504882813e+000, 2.500442504882813e+000,
        2.500442504882813e+000, 2.500442504882813e+000, 2.500442504882813e+000, 2.500442504882813e+000,
        2.500442504882813e+000, 1.694888305664063e+001, 1.694888305664063e+001, 1.694888305664063e+001,
        1.694888305664063e+001, 1.694888305664063e+001, 1.694888305664063e+001, 1.694888305664063e+001,
        1.694888305664063e+001, };

    double[] zArray = new double[] { 0, -6.793365478515625e-001, -1.047146701812744e+001, -1.452750015258789e+001,
        -1.047146701812744e+001, -6.793353557586670e-001, 9.112794876098633e+000, 1.316882705688477e+001,
        9.112790107727051e+00, -6.793365478515625e-001, -1.047146701812744e+001, -1.452750015258789e+001,
        -1.047146701812744e+001, -6.793353557586670e-001, 9.112794876098633e+000, 1.316882705688477e+001,
        9.112790107727051e+000, -6.793365478515625e-001, -1.047146701812744e+001, -1.452750015258789e+001,
        -1.047146701812744e+001, -6.793353557586670e-001, 9.112794876098633e+000, 1.316882705688477e+001,
        9.112790107727051e+000 };

    double[] color = Arrays.copyOf(yArray, 25);
    int[] numNodes = new int[] { 0, 4, 8, 12, 16, 20, 24, 28, 32, 40, 48 };
    int[] facets = new int[] { 9, 17, 18, 10, 16, 24, 17, 9, 10, 18, 19, 11, 11, 19, 20, 12, 12, 20, 21, 13, 13, 21, 22,
        14, 14, 22, 23, 15, 15, 23, 24, 16, 18, 17, 24, 23, 22, 21, 20, 19, 9, 10, 11, 12, 13, 14, 15, 16 };

    List<ArrayList<Integer>> cellFacets = new ArrayList<ArrayList<Integer>>(10);
    cellFacets.add(new ArrayList<Integer>());
    for ( int i = 0; i < 10; i++ )
      cellFacets.get(0).add(i);

    defineMesh(xArray, yArray, zArray, facets, numNodes, cellFacets, 1, color);
  }

  @Override
  public void start()
  {
    SoSeparator root = buildSceneGraph();
    buildGUI(root);
  }

  /*
   * Build GUI based on previously built scenegraph
   */
  private void buildGUI(SoSeparator root)
  {
    m_renderArea = ViewerComponentsFactory.createViewerExaminer();
    m_renderArea.setSceneGraph(root);
    m_renderArea.viewAll();

    // Build GUI

    // Interactive input
    String[] items = new String[] { "Simple Polyhedron", "Octogon", "Dodecahedron", "Cone", "Sphere", "Smiley" };

    JPanel mainGuiPanel = new JPanel();
    mainGuiPanel.setBounds(0, 138, 450, 162);
    mainGuiPanel.setLayout(new GridLayout(4, 1, 0, 0));
    JPanel firstHorizontalPanel = new JPanel();
    firstHorizontalPanel.setAlignmentY(Component.TOP_ALIGNMENT);
    firstHorizontalPanel.setAlignmentX(Component.LEFT_ALIGNMENT);
    mainGuiPanel.add(firstHorizontalPanel);
    firstHorizontalPanel.setLayout(new BoxLayout(firstHorizontalPanel, BoxLayout.X_AXIS));

    // Choice to display or not the Skin
    JPanel skinPanel = new JPanel();
    skinPanel.setAlignmentY(Component.TOP_ALIGNMENT);
    skinPanel.setAlignmentX(Component.LEFT_ALIGNMENT);
    skinPanel.setBorder(new TitledBorder(null, "Skin", TitledBorder.LEADING, TitledBorder.TOP, null, null));
    firstHorizontalPanel.add(skinPanel);
    skinPanel.setLayout(new BoxLayout(skinPanel, BoxLayout.X_AXIS));

    final JCheckBox skinVisibility = new JCheckBox("Show Mesh Skin");
    skinPanel.add(skinVisibility);
    skinVisibility.setSelected(true);

    // Probing
    JPanel probingPanel = new JPanel();
    firstHorizontalPanel.add(probingPanel);
    probingPanel.setAlignmentX(Component.LEFT_ALIGNMENT);
    probingPanel.setAlignmentY(Component.TOP_ALIGNMENT);
    probingPanel.setBorder(new TitledBorder(null, "Probe", TitledBorder.LEADING, TitledBorder.TOP, null, null));
    probingPanel.setLayout(new BoxLayout(probingPanel, BoxLayout.X_AXIS));

    final JCheckBox probeVisibility = new JCheckBox("Show Probe");
    probingPanel.add(probeVisibility);
    probeVisibility.setSelected(false);
    probeVisibility.setEnabled(false);

    // Shape selection
    JPanel shapePanel = new JPanel();
    shapePanel.setAlignmentY(Component.TOP_ALIGNMENT);
    shapePanel.setAlignmentX(Component.LEFT_ALIGNMENT);
    shapePanel.setBorder(new TitledBorder(null, "Shape Type", TitledBorder.LEADING, TitledBorder.TOP, null, null));
    mainGuiPanel.add(shapePanel);
    shapePanel.setLayout(new BoxLayout(shapePanel, BoxLayout.X_AXIS));

    final JRadioButton optionPlaneSlice = new JRadioButton("Plane Slice");
    optionPlaneSlice.setSelected(false);
    shapePanel.add(optionPlaneSlice);

    final JRadioButton optionIsosurface = new JRadioButton("IsoSurface");
    optionIsosurface.setSelected(true);
    shapePanel.add(optionIsosurface);

    // Slider to change isovalue
    m_isoValueSlider = new SliderPanel(m_sliderMin, m_sliderMax, m_isosurface.isoValue.getValue(), 1);
    m_isoValueSlider.addInfoText("Iso value : ");
    m_isoValueSlider.setSliderSize(new Dimension(200, 20));
    shapePanel.add(m_isoValueSlider);

    // DrawStyle selection
    JPanel drawStylePanel = new JPanel();
    mainGuiPanel.add(drawStylePanel);
    drawStylePanel.setAlignmentX(Component.LEFT_ALIGNMENT);
    drawStylePanel.setAlignmentY(Component.TOP_ALIGNMENT);
    drawStylePanel.setBorder(new TitledBorder(null, "Draw Style", TitledBorder.LEADING, TitledBorder.TOP, null, null));
    drawStylePanel.setLayout(new BoxLayout(drawStylePanel, BoxLayout.X_AXIS));

    // Faces
    final JCheckBox faceCheckbox = new JCheckBox("Faces");
    faceCheckbox.setSelected(true);
    drawStylePanel.add(faceCheckbox);

    // Edges
    final JCheckBox edgesCheckbox = new JCheckBox("Edges");
    edgesCheckbox.setSelected(false);
    drawStylePanel.add(edgesCheckbox);

    // Points
    final JCheckBox pointCheckbox = new JCheckBox("Point");
    pointCheckbox.setSelected(false);
    drawStylePanel.add(pointCheckbox);

    // Combo Box to let user select the wanted mesh
    JPanel meshTypePanel = new JPanel();
    mainGuiPanel.add(meshTypePanel);
    meshTypePanel.setAlignmentX(Component.LEFT_ALIGNMENT);
    meshTypePanel.setAlignmentY(Component.TOP_ALIGNMENT);
    meshTypePanel.setBorder(new TitledBorder(null, "Mesh Type", TitledBorder.LEADING, TitledBorder.TOP, null, null));
    meshTypePanel.setLayout(new BoxLayout(meshTypePanel, BoxLayout.X_AXIS));
    final JComboBox<String> meshBox = new JComboBox<>(items);
    meshTypePanel.add(meshBox);

    /*
     * GUI Listener
     */
    m_isoValueSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_isosurface.isoValue.setValue(value);
      }
    });

    // Mesh box listener
    meshBox.setSelectedIndex(3); // default mesh is cone
    meshBox.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        boolean isDirty = false;

        switch ( meshBox.getSelectedIndex() )
        {
        case 0 : // Simple Polyhedron
          if ( currentMesh != 0 )
          {
            currentMesh = 0;
            m_fromMemory = true;
            m_sliderMax = 5;
            m_sliderMin = -5;
            m_isosurface.isoValue.setValue(0);
            probeVisibility.setEnabled(true);
            isDirty = true;
          }
          break;
        case 1 : // Octogon
          if ( currentMesh != 1 )
          {
            currentMesh = 1;
            m_fromMemory = true;
            m_sliderMax = 17;
            m_sliderMin = (float) 2.5;
            m_isosurface.isoValue.setValue(9);
            probeVisibility.setEnabled(true);
            isDirty = true;
          }
          break;
        case 2 : // Dodecahedron
          if ( currentMesh != 2 )
          {
            currentMesh = 2;
            m_fromMemory = true;
            m_sliderMax = 13;
            m_sliderMin = (float) 2.5;
            m_isosurface.isoValue.setValue(8);
            probeVisibility.setEnabled(true);
            isDirty = true;
          }
          break;
        case 3 : // Cone
          if ( currentMesh != 3 )
          {
            currentMesh = 3;
            m_reader.setFilename(m_dataPath + "cone.dat");
            m_fromMemory = false;
            m_sliderMax = 5;
            m_sliderMin = (float) .4;
            m_isosurface.isoValue.setValue((float) 1.4);
            probeVisibility.setEnabled(false);
            probeVisibility.setSelected(false);
            isDirty = true;
          }
          break;
        case 4 : // Sphere
          if ( currentMesh != 4 )
          {
            m_reader.setFilename(m_dataPath + "sphere.dat");
            currentMesh = 4;
            m_fromMemory = false;
            m_sliderMax = (float) 2.8;
            m_sliderMin = 0;
            m_isosurface.isoValue.setValue((float) 1.4);
            probeVisibility.setEnabled(false);
            probeVisibility.setSelected(false);
            isDirty = true;
          }
          break;
        case 5 : // Smiley
          if ( currentMesh != 5 )
          {
            currentMesh = 5;
            m_reader.setFilename(m_dataPath + "smiley.dat");
            m_fromMemory = false;
            m_sliderMax = (float) 5.2;
            m_sliderMin = 0;
            m_isosurface.isoValue.setValue((float) 3.2);
            probeVisibility.setEnabled(false);
            probeVisibility.setSelected(false);
            isDirty = true;
          }
          break;
        }

        if ( isDirty )
        {
          if ( !m_fromMemory )
          {
            m_reader.readMesh();
            defineMesh(m_reader.getXcoords(), m_reader.getYcoords(), m_reader.getZcoords(), m_reader.getFaceNodeList(),
                m_reader.getNumNodePerFaceList(), m_reader.getCellFacets(), m_reader.getNumCells(),
                m_reader.getScalars());
          }
          else
          {
            if ( currentMesh == 0 )
              buildPolyhedron(0.5);
            else if ( currentMesh == 1 )
              buildOctogon();
            else
              buildDodecahedron();
          }
          m_moMesh.setMesh(m_meshpoly);
          update();
        }
      }
    });

    // Draw Style selection
    faceCheckbox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( faceCheckbox.isSelected() )
          m_drawStyle.displayFaces.setValue(true);
        else
          m_drawStyle.displayFaces.setValue(false);
      }
    });

    edgesCheckbox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( edgesCheckbox.isSelected() )
          m_drawStyle.displayEdges.setValue(true);
        else
          m_drawStyle.displayEdges.setValue(false);
      }
    });

    pointCheckbox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( pointCheckbox.isSelected() )
          m_drawStyle.displayPoints.setValue(true);

        else
          m_drawStyle.displayPoints.setValue(false);
      }
    });

    // Shape type selection
    optionIsosurface.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( optionIsosurface.isSelected() )
        {
          optionPlaneSlice.setSelected(false);
          m_isoSurfSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
          m_planeSliceSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
          m_isoValueSlider.setEnabled(true);
        }
        else
          optionPlaneSlice.setSelected(true);

      }
    });

    optionPlaneSlice.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( optionPlaneSlice.isSelected() )
        {
          optionIsosurface.setSelected(false);
          m_isoSurfSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
          m_planeSliceSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
          m_isoValueSlider.setEnabled(false);
        }
        else
          optionIsosurface.setSelected(true);
      }
    });

    // Skin visibility selection
    skinVisibility.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( skinVisibility.isSelected() )
          m_skinSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
        else
          m_skinSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
      }
    });

    // probe visibility selection
    probeVisibility.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( probeVisibility.isSelected() )
        {
          m_probeSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
          m_probeTextSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
        }
        else
        {
          m_probeSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
          m_probeTextSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
        }
      }
    });

    final Component component = m_renderArea.getComponent();
    component.setPreferredSize(new Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(mainGuiPanel, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    m_renderArea.dispose();
  }

  private SoSeparator buildSceneGraph()
  {
    SoShapeHints sh = new SoShapeHints();
    sh.vertexOrdering.setValue(SoShapeHints.VertexOrderings.CLOCKWISE);

    m_moMesh = new MoMesh();
    m_moMesh.setMesh(m_meshpoly);

    m_drawStyle = new MoDrawStyle();
    m_drawStyle.displayFaces.setValue(true);
    m_drawStyle.displayEdges.setValue(false);
    m_drawStyle.displayPoints.setValue(false);

    // Manipulator to move the plane slice plane (as a clip plane)
    m_planeManip = new SoClipPlaneManip();
    m_planeManip.plane.setValue(new SbPlane(new SbVec3f(0, 1, 0), 0f));
    // Turn off clip plane visibility to avoid flickering of plane slice faces.
    m_planeManip.on.setValue(false);
    m_planeManip.draggerPosition.setValue(getPlaneCenter());

    // Cross Section
    MoMeshPlaneSlice planeSlice = new MoMeshPlaneSlice();
    planeSlice.plane.connectFrom(m_planeManip.plane);
    // Use data set 0 for coloring
    planeSlice.colorScalarSetId.setValue(0);

    m_planeSliceSwitch = new SoSwitch();
    m_planeSliceSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
    m_planeSliceSwitch.addChild(m_planeManip);
    m_planeSliceSwitch.addChild(m_planeManip);
    m_planeSliceSwitch.addChild(planeSlice);

    m_colMap = new MoPredefinedColorMapping();
    // Adjust range to scalar set 1
    m_colMap.minValue.setValue((float) m_meshpoly.getScalarSet("$Scalars").getMin());
    m_colMap.maxValue.setValue((float) m_meshpoly.getScalarSet("$Scalars").getMax());
    m_colMap.predefColorMap.setValue(MoPredefinedColorMapping.PredefColorMapping.STANDARD);

    m_moScalarSetI = new MoScalarSetI();
    m_moScalarSetI.setScalarSet(m_meshpoly.getScalarSet("$Scalars"));

    m_isosurface = new MoMeshIsosurface();
    m_isosurface.isoValue.setValue((float) 1.4);
    m_isosurface.isoScalarSetId.setValue(0);
    m_isosurface.colorScalarSetId.setValue(0);
    m_isosurface.setExtractorCallback(new MyIsosurfExtractorCallback());

    // Transparent Mesh Skin
    SoDrawStyle skinStyle = new SoDrawStyle();
    skinStyle.style.setValue(SoDrawStyle.Styles.LINES);
    skinStyle.linePattern.setValue((short) 0xffff);

    SoPickStyle pickStyle = new SoPickStyle();
    pickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);

    SoLightModel lModel = new SoLightModel();
    lModel.model.setValue(SoLightModel.Models.BASE_COLOR);

    MoMeshSkin skin = new MoMeshSkin();
    skin.colorScalarSetId.setValue(0);
    skin.setExtractorCallback(new MySkinExtractorCallback());

    m_skinSwitch = new SoSwitch();
    {
      m_skinSwitch.addChild(skinStyle);
      m_skinSwitch.addChild(pickStyle);
      m_skinSwitch.addChild(lModel);
      m_skinSwitch.addChild(skin);
    }
    m_skinSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
    SoSeparator skinSep = new SoSeparator();
    skinSep.addChild(m_skinSwitch);

    MoMesh isoMesh = new MoMesh();
    isoMesh.connectFrom(m_isosurface);

    m_moVec3Set = new MoVec3SetI();
    m_moVec3Set.setVec3Set(m_meshpoly.getVec3Set("$MyGeometry"));

    m_vectorSwitch = new SoSwitch();
    MoDataBinding vdataBinding = new MoDataBinding();
    vdataBinding.dataBinding.setValue(MoDataBinding.DataBinding.PER_CELL);
    MoMeshVector vector = new MoMeshVector();
    {
      m_vectorSwitch.addChild(vdataBinding);
      m_vectorSwitch.addChild(vector);
    }

    m_isoSurfSwitch = new SoSwitch();
    m_isoSurfSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
    m_isoSurfSwitch.addChild(m_isosurface);
    m_isoSurfSwitch.addChild(isoMesh);

    // Probing
    MoMeshPointProbe pointProbe = new MoMeshPointProbe();
    m_probeDragger = new SoJackDragger();
    pointProbe.position.connectFrom(m_probeDragger.translation);

    m_probeSwitch = new SoSwitch();
    m_probeSwitch.whichChild.setValue(SoSwitch.WhichChild.SO_SWITCH_NONE.getValue());
    {
      m_probeSwitch.addChild(m_probeDragger);
      m_probeSwitch.addChild(pointProbe);
    }

    SoSeparator scene3D = new SoSeparator();
    scene3D.setName("Scene_3D");
    { // assemble scene graph
      scene3D.addChild(sh);
      scene3D.addChild(m_moMesh);
      scene3D.addChild(m_colMap);
      scene3D.addChild(m_moScalarSetI);
      scene3D.addChild(m_probeSwitch);
      scene3D.addChild(skinSep);
      scene3D.addChild(m_drawStyle);
      scene3D.addChild(m_planeSliceSwitch);
      scene3D.addChild(m_isoSurfSwitch);
      scene3D.addChild(m_moVec3Set);
      scene3D.addChild(m_vectorSwitch);
    }

    // 2D view
    SoOrthographicCamera camera2D = new SoOrthographicCamera();
    camera2D.viewportMapping.setValue(SoCamera.ViewportMappings.LEAVE_ALONE);

    // Text display
    SoSeparator cellTextSep = new SoSeparator();

    SoFont font = new SoFont();
    font.size.setValue(12);

    SoTranslation indexTextTrans = new SoTranslation();
    indexTextTrans.translation.setValue(-0.95F, 0.95F, -1.F);
    SoText2 cellIndexText = new SoText2();
    cellIndexText.justification.setValue(SoText2.Justifications.LEFT);

    SoTranslation scalarTextTrans = new SoTranslation();
    scalarTextTrans.translation.setValue(0.f, -0.05f, 0);
    SoText2 cellValueText = new SoText2();
    cellValueText.justification.setValue(SoText2.Justifications.LEFT);

    m_probeTextSwitch = new SoSwitch();
    m_probeTextSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
    m_probeTextSwitch.addChild(cellTextSep);
    {
      cellTextSep.addChild(font);
      cellTextSep.addChild(indexTextTrans);
      cellTextSep.addChild(cellIndexText);
      cellTextSep.addChild(scalarTextTrans);
      cellTextSep.addChild(cellValueText);
    }

    ProbeCallback probeCB = new ProbeCallback(m_probeTextSwitch, cellIndexText, cellValueText);
    pointProbe.setProbeCallback(probeCB);

    SoSeparator scene2D = new SoSeparator();
    scene2D.setName("Scene_2D");
    {
      scene2D.addChild(camera2D);
      scene2D.addChild(m_probeTextSwitch);
    }

    SoSeparator root = new SoSeparator();
    { // assemble scene graph
      root.addChild(scene3D);
      root.addChild(scene2D);
    }

    return root;
  }

  private void update()
  {
    m_colMap.minValue.setValue((float) m_meshpoly.getScalarSet("$Scalars").getMin());
    m_colMap.maxValue.setValue((float) m_meshpoly.getScalarSet("$Scalars").getMax());
    m_colMap.predefColorMap.setValue(MoPredefinedColorMapping.PredefColorMapping.STANDARD);

    // update
    m_moScalarSetI.setScalarSet(m_meshpoly.getScalarSet("$Scalars"));
    m_moVec3Set.setVec3Set(m_meshpoly.getVec3Set("$MyGeometry"));

    m_isoValueSlider.setSliderMax(m_sliderMax);
    m_isoValueSlider.setSliderMin(m_sliderMin);
    m_isoValueSlider.setSliderValue(m_isosurface.isoValue.getValue());
    m_isoValueSlider.setBoxSelected(true);

    m_planeManip.draggerPosition.setValue(getPlaneCenter());
    m_planeManip.plane.setValue(new SbPlane(new SbVec3f(0, 1, 0), 0f));
    m_renderArea.viewAll();

  }

  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("Polyhedral mesh Example");
  }

  private SbVec3d getPlaneCenter()
  {
    MbMeshGeometry geom = m_meshpoly.getGeometry();
    return new SbVec3d((geom.getXMax() - Math.abs(geom.getXMin())) / 2.,
        (geom.getYMax() - Math.abs(geom.getYMin())) / 2., (geom.getZMax() - Math.abs(geom.getZMin())) / 2.);
  }
}