package meshvizxlm.mapping.polyhedralmesh;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;

public class PolyhedralMeshReader
{
  private int m_numVariables = 0;
  private int m_numElements = 0;
  private double m_numNodes = 0;
  private int m_numFaces = 0;
  private int m_totalNumFaceNodes = 0;

  private int[] m_numNodePerFaceList;

  private int[] m_faceNodeList;
  private List<ArrayList<Integer>> m_elementFaces;

  private double[] m_coordsX;
  private double[] m_coordsY;
  private double[] m_coordsZ;

  private double[] m_scalars;

  private BufferedReader m_reader;
  private String m_fileName;

  public PolyhedralMeshReader()
  {}

  public void setFilename(String filename) throws UnsupportedOperationException
  {
    try
    {
      FileInputStream file = new FileInputStream(filename);
      m_reader = new BufferedReader(new InputStreamReader(file));
    }
    catch (FileNotFoundException e)
    {
      throw new UnsupportedOperationException("PolyhedralMeshReader: file not found: " + filename);
    }
  }

  private void clean()
  {
    m_coordsX = null;
    m_coordsY = null;
    m_coordsZ = null;
    m_scalars = null;

    if ( m_elementFaces != null )
    {
      for ( int i = 0; i < m_elementFaces.size(); i++ )
        m_elementFaces.get(i).clear();

      m_elementFaces.clear();
    }

    m_numVariables = 0;
    m_numElements = 0;
    m_numNodes = 0;
    m_numFaces = 0;
    m_totalNumFaceNodes = 0;

    m_numNodePerFaceList = null;
    m_faceNodeList = null;
  }

  private void extractZone()
  {
    try
    {
      String line;
      String[] words;
      String[] values;
      double value = 0;

      // Search for keyword ZONE
      line = readUntil("Nodes");

      // Extract Nodes, Faces and Element.
      values = line.split(",");
      for ( int v = 0; v < values.length; v++ )
      {
        words = values[v].split("=");

        if ( !words[0].trim().equalsIgnoreCase("zonetype") )
          value = Double.parseDouble(words[1]);

        if ( words[0].trim().equalsIgnoreCase("nodes") )
        {
          m_numNodes = value;
          m_coordsX = new double[(int) m_numNodes];
          m_coordsY = new double[(int) m_numNodes];
          m_coordsZ = new double[(int) m_numNodes];
        }
        else if ( words[0].trim().equalsIgnoreCase("faces") )
          m_numFaces = (int) value;
        else if ( words[0].trim().equalsIgnoreCase("elements") )
        {
          m_numElements = (int) value;
          m_elementFaces = new ArrayList<ArrayList<Integer>>((int) m_numElements);
          for ( int i = 0; i < m_numElements; i++ )
            m_elementFaces.add(new ArrayList<Integer>());
        }
      }
    }
    catch (Exception e)
    {
      throw new UnsupportedOperationException("PolyhedralMeshReader: file format not supported.");
    }
  }

  private String readUntil(String wordToReach) throws UnsupportedOperationException
  {
    try
    {
      String line;

      while ( (line = m_reader.readLine()) != null && !line.trim().startsWith(wordToReach) )
      {}

      return line;
    }
    catch (IOException e)
    {
      throw new UnsupportedOperationException("PolyhedralMeshReader: I/O exception occured.");
    }
    catch (Exception e)
    {
      throw new UnsupportedOperationException("PolyhedralMeshReader: file format not supported.");
    }
  }

  private void extractNumFaceNodes(String line)
  {
    String[] words;
    String[] value = line.split(",");

    for ( String separateValue : value )
    {
      words = separateValue.split("=");
      if ( words[0].trim().equalsIgnoreCase("TotalNumFaceNodes") )
        m_totalNumFaceNodes = Integer.parseInt(words[1]);
    }
  }

  private String readData()
  {
    try
    {
      int ndataRead = 0;

      String line;
      String[] values;
      double[] rawCoords = new double[(int) (m_numVariables * m_numNodes)];

      // Loop over data until we reach end of data section (marked with
      while ( (line = m_reader.readLine()) != null && !line.contains("# node") )
      {
        values = line.trim().split(" ");
        for ( String v : values )
        {
          rawCoords[ndataRead] = Double.valueOf(v);
          ndataRead++;
        }
      }

      // Extract X/Y/Z coords
      for ( int node = 0; node < m_numNodes; node++ )
      {
        m_coordsX[node] = rawCoords[node];
        m_coordsY[node] = rawCoords[node + (int) m_numNodes];
        m_coordsZ[node] = rawCoords[node + (int) (2 * m_numNodes)];
      }

      int num = m_numVariables - 3;

      m_scalars = new double[(int) (m_numNodes * (num))];

      for ( int d = 0; d < num; ++d )
        for ( int node = 0; node < m_numNodes; ++node )
        {
          double xc = m_coordsX[node] * m_coordsX[node];
          double yc = m_coordsY[node] * m_coordsY[node];
          double zc = m_coordsZ[node] * m_coordsZ[node];
          m_scalars[node] = Math.sqrt(xc + yc + zc);
        }

      return line;
    }
    catch (IOException e)
    {
      throw new UnsupportedOperationException("PolyhedralMeshReader: I/O exception occured.");
    }
    catch (Exception e)
    {
      throw new UnsupportedOperationException("PolyhedralMeshReader: file format not supported.");
    }
  }

  private String readNumNodeFace()
  {
    try
    {
      String line;
      String[] values;
      m_numNodePerFaceList = new int[m_numFaces + 1];
      int curValue = 0;
      m_numNodePerFaceList[curValue++] = 0;

      while ( (line = m_reader.readLine()) != null && !line.trim().startsWith("#") )
      {
        values = line.trim().split(" ");
        for ( String v : values )
        {
          m_numNodePerFaceList[curValue] = Integer.parseInt(v) + m_numNodePerFaceList[curValue - 1];
          curValue++;
        }
      }
      return line;
    }
    catch (IOException e)
    {
      throw new UnsupportedOperationException("PolyhedralMeshReader: I/O exception occured.");
    }
    catch (Exception e)
    {
      throw new UnsupportedOperationException("PolyhedralMeshReader: file format not supported.");
    }
  }

  private String readFaceNodes()
  {
    try
    {
      String line;
      String[] values;
      m_faceNodeList = new int[m_totalNumFaceNodes];
      int curValue = 0;

      while ( (line = m_reader.readLine()) != null && !line.trim().startsWith("#") )
      {
        values = line.trim().split(" ");
        for ( String v : values )
          m_faceNodeList[curValue++] = Integer.parseInt(v) - 1;
      }
      return line;
    }
    catch (IOException e)
    {
      throw new UnsupportedOperationException("PolyhedralMeshReader: I/O exception occured.");
    }
    catch (Exception e)
    {
      throw new UnsupportedOperationException("PolyhedralMeshReader: file format not supported.");
    }
  }

  private String readElementFaces()
  {
    try
    {
      String line;
      String[] values;

      int[] tmpBuffer = new int[m_numFaces];
      int curValue = 0;

      while ( (line = m_reader.readLine()) != null && !line.trim().startsWith("#") )
      {
        values = line.trim().split(" ");
        for ( String v : values )
          tmpBuffer[curValue++] = Integer.parseInt(v);
      }

      int index = 0;
      for ( int n = 0; n < m_numFaces; ++n )
      {
        index = tmpBuffer[n] - 1;
        if ( index >= 0 )
          m_elementFaces.get(index).add(n);
      }

      return line;
    }
    catch (IOException e)
    {
      throw new UnsupportedOperationException("PolyhedralMeshReader: I/O exception occured.");
    }
    catch (Exception e)
    {
      throw new UnsupportedOperationException("PolyhedralMeshReader: file format not supported.");
    }
  }

  public void readMesh() throws UnsupportedOperationException
  {
    clean();
    try
    {
      if ( m_reader == null )
        throw new FileNotFoundException();

      String line = readUntil("VARIABLES");
      while ( line != null && !line.trim().startsWith("ZONE") )
      {
        if ( line.contains("\"") )
          m_numVariables++;

        line = m_reader.readLine();
      }

      // Now line starts with wanted KeyWord ZONE, extract infos
      extractZone();

      // Reach Data section
      while ( (line = m_reader.readLine()) != null && !line.trim().startsWith("DT") )
      {
        if ( line.trim().contains("TotalNumFaceNodes") )
          extractNumFaceNodes(line);
      }

      // Reach keyword DT. Extract data
      line = readData();

      // End of data section. Extract others properties
      while ( line != null )
      {
        if ( line.trim().contains("# node count") )
          line = readNumNodeFace();

        if ( line.trim().contains("# face nodes") )
          line = readFaceNodes();

        if ( line.trim().contains("# left elements") || line.trim().contains("# right elements") )
          line = readElementFaces();

        if ( line != null && !line.trim().startsWith("#") )
          line = m_reader.readLine();
      }

      m_reader.close();
    }
    catch (FileNotFoundException e)
    {
      throw new UnsupportedOperationException("PolyhedralMeshReader: file not found: " + m_fileName);
    }
    catch (IOException e)
    {
      throw new UnsupportedOperationException("PolyhedralMeshReader: I/O exception occured.");
    }
    catch (Exception e)
    {
      throw new UnsupportedOperationException("PolyhedralMeshReader: file format not supported.");
    }
  }

  // Accessors to access mesh properties
  public PolyhedralMeshReader(String filename)
  {
    m_fileName = filename;
    setFilename(m_fileName);
  }

  public double[] getXcoords()
  {
    return m_coordsX;
  }

  public double[] getYcoords()
  {
    return m_coordsY;
  }

  public double[] getZcoords()
  {
    return m_coordsZ;
  }

  public int[] getFaceNodeList()
  {
    return m_faceNodeList;
  }

  public int[] getNumNodePerFaceList()
  {
    return m_numNodePerFaceList;
  }

  public int getNumCells()
  {
    return m_numElements;
  }

  public List<ArrayList<Integer>> getCellFacets()
  {
    return m_elementFaces;
  }

  public double[] getScalars()
  {
    return m_scalars;
  }
}
