package meshvizxlm.mapping.spherecylinderplaneslices;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.border.EtchedBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import com.openinventor.inventor.SbPlane;
import com.openinventor.inventor.SbVec3d;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.actions.SoGLRenderAction.TransparencyTypes;
import com.openinventor.inventor.draggers.SoJackDragger;
import com.openinventor.inventor.manips.SoClipPlaneManip;
import com.openinventor.inventor.nodes.SoPickStyle;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoShapeHints;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.sensors.SoFieldSensor;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.meshvizxlm.extractors.MiExtractorCallback;
import com.openinventor.meshvizxlm.mapping.nodes.*;
import com.openinventor.meshvizxlm.mesh.cell.MiCellFilterI;
import com.openinventor.meshvizxlm.mesh.cell.MiCellFilterIjk;

import meshvizxlm.mesh.MbSampleMeshBuilder;
import meshvizxlm.mesh.data.MbScalarSetI;
import meshvizxlm.mesh.data.MbScalarSetIjk;
import meshvizxlm.mesh.geometry.MbMeshGeometry;
import meshvizxlm.mesh.volumes.MbHexahedronMeshIjk;
import meshvizxlm.mesh.volumes.MbVertexHexahedronMeshIjk;
import meshvizxlm.mesh.volumes.MbVolumeMeshHexahedron;
import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  // ///// INNER CLASS
  class MyExtractorCallback implements MiExtractorCallback
  {
    @Override
    public void beginCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      if ( topoChanged )
        System.out.println("shape computation ...");
      if ( dataSetChanged )
        System.out.println("dataset computation ...");
    }

    @Override
    public void endCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {}
  }

  class CellFilterI implements MiCellFilterI
  {
    @Override
    public boolean acceptCell(long cellIndex)
    {
      int index = (int) cellIndex;
      int k = index / (DIM * DIM);
      int ij = index % (DIM * DIM);
      int j = ij / DIM;
      int i = ij % DIM;

      if ( DIM / 3. < i && i <= 2 * DIM / 3. && DIM / 3. < j && j <= 2 * DIM / 3. && k > DIM / 2. )
        return false;

      return true;
    }

    @Override
    public long getTimeStamp()
    {
      return 0;
    }
  }

  class CellFilterIjk implements MiCellFilterIjk
  {
    @Override
    public boolean acceptCell(int i, int j, int k)
    {
      if ( DIM / 3. < i && i <= 2 * DIM / 3. && DIM / 3. < j && j <= 2 * DIM / 3. && k > DIM / 2. )
        return false;

      return true;
    }

    @Override
    public long getTimeStamp()
    {
      return 0;
    }
  }

  private static final int DIM = 30;
  private static final String SCALARSET_IJK_NAME = "$CellId";
  private static final String SCALARSET_I_NAME = "$ScalarSetX";

  private SoSwitch m_filterSwitch;
  private SoSwitch m_sliceSwitch;
  private SoSwitch m_skinSwitch;

  private SoFieldSensor m_cylinderSensor;
  private SoFieldSensor m_draggerScaleSensor;

  private IViewerExaminer m_renderArea;
  private MbVolumeMeshHexahedron m_meshHexa;
  private MbHexahedronMeshIjk m_meshHexaIjk;
  private MbVertexHexahedronMeshIjk m_meshVertexHexaIjk;

  private MoMesh m_moMesh;
  private MoCellFilter m_moCellFilter;
  private CellFilterI m_cellFilterI;
  private CellFilterIjk m_cellFilterIjk;
  private MoPredefinedColorMapping m_colMap;
  private MoScalarSetI m_moScalarSetI;
  private MoScalarSetIjk m_moScalarSetIjk;

  private JCheckBox m_filterBox;

  public Main()
  {
    double[] min = { 0, 0, 0 };
    double[] max = { 10, 10, 10 };

    MbSampleMeshBuilder meshBuilder = new MbSampleMeshBuilder();
    m_meshHexa = meshBuilder.getMeshHexahedron(DIM, DIM, DIM, min, max);
    m_meshHexaIjk = meshBuilder.getHexahedronMeshIjk(DIM, DIM, DIM, min, max);
    m_meshVertexHexaIjk = meshBuilder.getVertexHexahedronMeshIjk(DIM, DIM, DIM, min, max);

    m_cellFilterI = new CellFilterI();
    m_cellFilterIjk = new CellFilterIjk();
  }

  @Override
  public void start()
  {
    SoSeparator root = buildSceneGraph();

    m_renderArea = ViewerComponentsFactory.createViewerExaminer();
    m_renderArea.getRenderArea().setTransparencyType(TransparencyTypes.SORTED_PIXEL);
    m_renderArea.setSceneGraph(root);
    m_renderArea.viewAll();

    // SWING part
    m_filterBox = new JCheckBox("Cell filter");
    m_filterBox.addChangeListener(new ChangeListener()
    {
      @Override
      public void stateChanged(ChangeEvent e)
      {
        if ( m_filterBox.isSelected() )
          m_filterSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
        else
          m_filterSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
      }
    });

    String[] items2 = new String[] { "Outline", "Transparent mesh skin", "None" };
    final JComboBox<String> skinBox = new JComboBox<>(items2);

    JLabel meshLabel = new JLabel("Mesh      ");
    String[] meshItems = new String[] { "Hexahedron", "Hexahedron IJK", "Vertex Hexahedron IJK" };
    final JComboBox<String> meshBox = new JComboBox<>(meshItems);
    meshBox.setSelectedIndex(2);
    meshBox.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        switch ( meshBox.getSelectedIndex() )
        {
        case 0 : // Hexahedron
          m_moMesh.setMesh(m_meshHexa);
          m_moCellFilter.setCellFilter(m_cellFilterI);
          MbScalarSetI scalarSet = m_meshHexa.getScalarSet(SCALARSET_I_NAME);
          m_colMap.minValue.setValue((float) scalarSet.getMin());
          m_colMap.maxValue.setValue((float) scalarSet.getMax());
          m_moScalarSetI.setScalarSet(scalarSet);
          break;
        case 1 : // hexahedron IJK
          m_moMesh.setMesh(m_meshHexaIjk);
          m_moCellFilter.setCellFilter(m_cellFilterIjk);
          scalarSet = m_meshHexaIjk.getScalarSet(SCALARSET_I_NAME);
          m_colMap.minValue.setValue((float) scalarSet.getMin());
          m_colMap.maxValue.setValue((float) scalarSet.getMax());
          m_moScalarSetI.setScalarSet(scalarSet);
          break;
        case 2 : // vertex hexahedron IJK
          m_moMesh.setMesh(m_meshVertexHexaIjk);
          m_moCellFilter.setCellFilter(m_cellFilterIjk);
          MbScalarSetIjk scalarSetIjk = m_meshVertexHexaIjk.getScalarSetIjk(SCALARSET_IJK_NAME);
          m_colMap.minValue.setValue((float) scalarSetIjk.getMin());
          m_colMap.maxValue.setValue((float) scalarSetIjk.getMax());
          m_moScalarSetIjk.setScalarSet(scalarSetIjk);
          break;
        default:
          break;
        }
      }
    });

    JLabel label = new JLabel("Slice       ");
    String[] items = new String[] { "Plane", "Cylinder", "Sphere" };
    final JComboBox<String> sliceBox = new JComboBox<>(items);
    sliceBox.setSelectedIndex(0);
    sliceBox.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        m_sliceSwitch.whichChild.setValue(sliceBox.getSelectedIndex());
      }
    });

    JLabel label2 = new JLabel("Skin        ");
    skinBox.setSelectedIndex(1);
    skinBox.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        int index = skinBox.getSelectedIndex();
        if ( index == 2 )
          m_skinSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
        else
          m_skinSwitch.whichChild.setValue(index);
      }
    });

    JPanel meshPanel = new JPanel();
    meshPanel.setLayout(new BoxLayout(meshPanel, BoxLayout.X_AXIS));
    meshPanel.add(meshLabel);
    meshPanel.add(meshBox);

    JPanel slicePanel = new JPanel();
    slicePanel.setLayout(new BoxLayout(slicePanel, BoxLayout.X_AXIS));
    slicePanel.add(label);
    slicePanel.add(sliceBox);

    JPanel skinPanel = new JPanel();
    skinPanel.setLayout(new BoxLayout(skinPanel, BoxLayout.X_AXIS));
    skinPanel.add(label2);
    skinPanel.add(skinBox);

    JPanel panel2 = new JPanel();
    panel2.setLayout(new BoxLayout(panel2, BoxLayout.Y_AXIS));
    panel2.setBorder(BorderFactory.createEtchedBorder(EtchedBorder.LOWERED, null, null));
    panel2.add(m_filterBox);
    panel2.add(meshPanel);
    panel2.add(slicePanel);
    panel2.add(skinPanel);

    final Component component = m_renderArea.getComponent();
    component.setPreferredSize(new Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(panel2, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    m_renderArea.dispose();
  }

  private SoSeparator buildSceneGraph()
  {
    MbMeshGeometry geom = m_meshHexa.getGeometry();
    SbVec3d center = new SbVec3d((geom.getXMax() - geom.getXMin()) / 2., (geom.getYMax() - geom.getYMin()) / 2.,
        (geom.getZMax() - geom.getZMin()) / 2.);

    SoShapeHints sh = new SoShapeHints();
    sh.vertexOrdering.setValue(SoShapeHints.VertexOrderings.CLOCKWISE);

    // Manipulator to move the plane slice plane (as a clip plane)
    final SoClipPlaneManip manip = new SoClipPlaneManip();
    manip.plane.setValue(new SbPlane(new SbVec3f(1, 0, 1), 0.5f));
    // Turn off clip plane visibility to avoid flickering of plane slice faces.
    manip.on.setValue(false);
    manip.draggerPosition.setValue(center);

    m_moMesh = new MoMesh();
    m_moMesh.setMesh(m_meshVertexHexaIjk);

    m_filterSwitch = new SoSwitch();
    m_filterSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
    m_moCellFilter = new MoCellFilter();
    m_moCellFilter.setCellFilter(m_cellFilterIjk);

    MoDataBinding dataBinding = new MoDataBinding();
    dataBinding.dataBinding.setValue(MoDataBinding.DataBinding.PER_NODE);

    MbScalarSetIjk scalarSet = m_meshVertexHexaIjk.getScalarSetIjk(SCALARSET_IJK_NAME);
    m_colMap = new MoPredefinedColorMapping();
    m_colMap.minValue.setValue((float) scalarSet.getMin());
    m_colMap.maxValue.setValue((float) scalarSet.getMax());
    m_colMap.predefColorMap.setValue(MoPredefinedColorMapping.PredefColorMapping.BLUE_RED);

    m_moScalarSetI = new MoScalarSetI();

    m_moScalarSetIjk = new MoScalarSetIjk();
    m_moScalarSetIjk.setScalarSet(scalarSet);

    m_sliceSwitch = new SoSwitch();
    m_sliceSwitch.whichChild.setValue(0);

    MyExtractorCallback callback = new MyExtractorCallback();

    // Cross Section
    MoMeshPlaneSlice planeSlice = new MoMeshPlaneSlice();
    planeSlice.plane.connectFrom(manip.plane);
    // Use data set 0 for coloring
    planeSlice.colorScalarSetId.setValue(0);
    planeSlice.setExtractorCallback(callback);

    final MoMeshCylinderSlice cylinderSlice = new MoMeshCylinderSlice();
    cylinderSlice.center.connectFrom(((SoJackDragger) manip.getDragger()).translation);
    cylinderSlice.direction.setValue(manip.plane.getValue().getNormal());
    cylinderSlice.radius.setValue(1);
    cylinderSlice.colorScalarSetId.setValue(0);
    cylinderSlice.setExtractorCallback(callback);

    m_cylinderSensor = new SoFieldSensor(new Runnable()
    {

      @Override
      public void run()
      {
        cylinderSlice.direction.setValue(manip.plane.getValue().getNormal());
      }
    });
    m_cylinderSensor.attach(manip.plane);

    final MoMeshSphereSlice sphereSlice = new MoMeshSphereSlice();
    sphereSlice.center.connectFrom(((SoJackDragger) manip.getDragger()).translation);
    sphereSlice.radius.setValue(1);
    sphereSlice.colorScalarSetId.setValue(0);
    sphereSlice.setExtractorCallback(callback);

    m_skinSwitch = new SoSwitch();
    m_skinSwitch.whichChild.setValue(1);

    // Transparent Mesh Skin
    SoSeparator skinSep = new SoSeparator();

    MoMaterial mat = new MoMaterial();
    mat.transparency.setValue((float) 0.6);

    SoPickStyle pickStyle = new SoPickStyle();
    pickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);

    MoMeshSkin skin = new MoMeshSkin();
    skin.colorScalarSetId.setValue(-1);

    // Outline
    MoMeshOutline outline = new MoMeshOutline();
    outline.colorScalarSetId.setValue(0);

    SoSeparator root = new SoSeparator();
    {
      m_filterSwitch.addChild(m_moCellFilter);

      m_sliceSwitch.addChild(planeSlice);
      m_sliceSwitch.addChild(cylinderSlice);
      m_sliceSwitch.addChild(sphereSlice);

      skinSep.addChild(mat);
      skinSep.addChild(pickStyle);
      skinSep.addChild(skin);

      m_skinSwitch.addChild(outline);
      m_skinSwitch.addChild(skinSep);

      root.addChild(sh);
      root.addChild(manip);
      root.addChild(m_moMesh);
      root.addChild(m_filterSwitch);
      root.addChild(m_colMap);
      root.addChild(dataBinding);
      root.addChild(m_moScalarSetI);
      root.addChild(m_moScalarSetIjk);
      root.addChild(m_sliceSwitch);
      root.addChild(m_skinSwitch);
    }

    m_draggerScaleSensor = new SoFieldSensor(new Runnable()
    {

      @Override
      public void run()
      {
        float newRadius = ((SoJackDragger) manip.getDragger()).scaleFactor.getValue().getX();
        cylinderSlice.radius.setValue(newRadius);
        sphereSlice.radius.setValue(newRadius);
      }
    });
    m_draggerScaleSensor.attach(((SoJackDragger) manip.getDragger()).scaleFactor);

    return root;
  }

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain("Sphere, cylinder, plane slices Example");
  }

}
