package meshvizxlm.mesh.cell;

import com.openinventor.meshvizxlm.extractors.cell.MxCellExtract;
import com.openinventor.meshvizxlm.extractors.cell.MxHexahedronCellExtract;
import com.openinventor.meshvizxlm.mesh.cell.MiVolumeCell;
import com.openinventor.meshvizxlm.mesh.geometry.MiGeometryI;

public class MbHexahedronCell implements MiVolumeCell
{

  protected static final int[][] FACET_NODE = { { 0, 1, 2, 3 }, { 4, 7, 6, 5 }, { 0, 3, 7, 4 }, { 1, 5, 6, 2 },
      { 0, 4, 5, 1 }, { 3, 2, 6, 7 } };

  protected int[] m_nodeIds;

  public MbHexahedronCell(int n0, int n1, int n2, int n3, int n4, int n5, int n6, int n7)
  {

    this.m_nodeIds = new int[8];
    this.m_nodeIds[0] = n0;
    this.m_nodeIds[1] = n1;
    this.m_nodeIds[2] = n2;
    this.m_nodeIds[3] = n3;
    this.m_nodeIds[4] = n4;
    this.m_nodeIds[5] = n5;
    this.m_nodeIds[6] = n6;
    this.m_nodeIds[7] = n7;
  }

  @Override
  public int[] getIsosurfTopology(int caseId, int[] edges)
  {
    return MxHexahedronCellExtract.getIsosurfTopology(caseId, edges);
  }

  @Override
  public int getMarchingCaseId(boolean[] nodesSign, long beginNodeId)
  {
    int case_id = 0;
    int bId = (int) beginNodeId;

    if ( nodesSign[this.m_nodeIds[0] - bId] )
      case_id = 1;
    if ( nodesSign[this.m_nodeIds[1] - bId] )
      case_id |= 2;
    if ( nodesSign[this.m_nodeIds[2] - bId] )
      case_id |= 4;
    if ( nodesSign[this.m_nodeIds[3] - bId] )
      case_id |= 8;
    if ( nodesSign[this.m_nodeIds[4] - bId] )
      case_id |= 16;
    if ( nodesSign[this.m_nodeIds[5] - bId] )
      case_id |= 32;
    if ( nodesSign[this.m_nodeIds[6] - bId] )
      case_id |= 64;
    if ( nodesSign[this.m_nodeIds[7] - bId] )
      case_id |= 128;

    if ( case_id == 255 )
      case_id = 0;
    return case_id;
  }

  @Override
  public long[] getNodesIndexOfFacet(int facet, long[] facetNodes)
  {
    if ( facetNodes == null || facetNodes.length != 4 )
      facetNodes = new long[4];

    facetNodes[0] = this.m_nodeIds[FACET_NODE[facet][0]];
    facetNodes[1] = this.m_nodeIds[FACET_NODE[facet][1]];
    facetNodes[2] = this.m_nodeIds[FACET_NODE[facet][2]];
    facetNodes[3] = this.m_nodeIds[FACET_NODE[facet][3]];

    return facetNodes;
  }

  @Override
  public double[] getFacetCenter(int facet, MiGeometryI geometry)
  {
    int[] facetNodeInds = FACET_NODE[facet];
    double[] facetCenter = { 0, 0, 0 };
    double[] coord;

    for ( int i = 0; i < 4; ++i )
    {
      coord = geometry.getCoord(m_nodeIds[facetNodeInds[i]]);
      facetCenter[0] += coord[0];
      facetCenter[1] += coord[1];
      facetCenter[2] += coord[2];
    }

    facetCenter[0] /= 4;
    facetCenter[1] /= 4;
    facetCenter[2] /= 4;

    return facetCenter;
  }

  @Override
  public int getNumFacets()
  {
    return 6;
  }

  @Override
  public long getNodeIndex(int node)
  {
    return this.m_nodeIds[node];
  }

  @Override
  public int getNumNodes()
  {
    return 8;
  }

  @Override
  public boolean isPointInsideCell(MiGeometryI meshGeometry, double[] point, double[] weights)
  {
    return MxHexahedronCellExtract.isPointInsideCell(meshGeometry, this, point, weights);
  }

  @Override
  public double getRelativeSize(MiGeometryI meshGeometry)
  {
    return MxHexahedronCellExtract.getLongestEdgeLength(meshGeometry, this);
  }

  @Override
  public double[] getCenter(MiGeometryI geometry)
  {
    return MxCellExtract.getCenter(geometry, this);
  }

}
